/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.common.util;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import org.apache.hive.common.util.Murmur3;

public class BloomFilter {
    public static final double DEFAULT_FPP = 0.05;
    protected BitSet bitSet;
    protected int numBits;
    protected int numHashFunctions;
    public static final int START_OF_SERIALIZED_LONGS = 5;

    public BloomFilter() {
    }

    public BloomFilter(long expectedEntries) {
        this(expectedEntries, 0.05);
    }

    static void checkArgument(boolean expression, String message) {
        if (!expression) {
            throw new IllegalArgumentException(message);
        }
    }

    public BloomFilter(long expectedEntries, double fpp) {
        BloomFilter.checkArgument(expectedEntries > 0L, "expectedEntries should be > 0");
        BloomFilter.checkArgument(fpp > 0.0 && fpp < 1.0, "False positive probability should be > 0.0 & < 1.0");
        int nb = BloomFilter.optimalNumOfBits(expectedEntries, fpp);
        this.numBits = nb + (64 - nb % 64);
        this.numHashFunctions = BloomFilter.optimalNumOfHashFunctions(expectedEntries, this.numBits);
        this.bitSet = new BitSet(this.numBits);
    }

    public BloomFilter(long[] bits, int numFuncs) {
        this.bitSet = new BitSet(bits);
        this.numBits = bits.length * 64;
        this.numHashFunctions = numFuncs;
    }

    static int optimalNumOfHashFunctions(long n, long m3) {
        return Math.max(1, (int)Math.round((double)m3 / (double)n * Math.log(2.0)));
    }

    static int optimalNumOfBits(long n, double p) {
        return (int)((double)(-n) * Math.log(p) / (Math.log(2.0) * Math.log(2.0)));
    }

    public void add(byte[] val) {
        if (val == null) {
            this.addBytes(val, -1, -1);
        } else {
            this.addBytes(val, 0, val.length);
        }
    }

    public void addBytes(byte[] val, int offset, int length) {
        long hash64 = val == null ? 2862933555777941757L : Murmur3.hash64(val, offset, length);
        this.addHash(hash64);
    }

    private void addHash(long hash64) {
        int hash1 = (int)hash64;
        int hash2 = (int)(hash64 >>> 32);
        for (int i = 1; i <= this.numHashFunctions; ++i) {
            int combinedHash = hash1 + (i + 1) * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            int pos = combinedHash % this.numBits;
            this.bitSet.set(pos);
        }
    }

    public void addString(String val) {
        if (val == null) {
            this.add(null);
        } else {
            this.add(val.getBytes());
        }
    }

    public void addLong(long val) {
        this.addHash(this.getLongHash(val));
    }

    public void addDouble(double val) {
        this.addLong(Double.doubleToLongBits(val));
    }

    public boolean test(byte[] val) {
        if (val == null) {
            return this.testBytes(val, -1, -1);
        }
        return this.testBytes(val, 0, val.length);
    }

    public boolean testBytes(byte[] val, int offset, int length) {
        long hash64 = val == null ? 2862933555777941757L : Murmur3.hash64(val, offset, length);
        return this.testHash(hash64);
    }

    private boolean testHash(long hash64) {
        int hash1 = (int)hash64;
        int hash2 = (int)(hash64 >>> 32);
        for (int i = 1; i <= this.numHashFunctions; ++i) {
            int pos;
            int combinedHash = hash1 + (i + 1) * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            if (this.bitSet.get(pos = combinedHash % this.numBits)) continue;
            return false;
        }
        return true;
    }

    public boolean testString(String val) {
        if (val == null) {
            return this.test(null);
        }
        return this.test(val.getBytes());
    }

    public boolean testLong(long val) {
        return this.testHash(this.getLongHash(val));
    }

    private long getLongHash(long key) {
        key = (key ^ 0xFFFFFFFFFFFFFFFFL) + (key << 21);
        key ^= key >> 24;
        key = key + (key << 3) + (key << 8);
        key ^= key >> 14;
        key = key + (key << 2) + (key << 4);
        key ^= key >> 28;
        key += key << 31;
        return key;
    }

    public boolean testDouble(double val) {
        return this.testLong(Double.doubleToLongBits(val));
    }

    public long sizeInBytes() {
        return this.getBitSize() / 8;
    }

    public int getBitSize() {
        return this.bitSet.getData().length * 64;
    }

    public int getNumHashFunctions() {
        return this.numHashFunctions;
    }

    public long[] getBitSet() {
        return this.bitSet.getData();
    }

    public String toString() {
        return "m: " + this.numBits + " k: " + this.numHashFunctions;
    }

    public void merge(BloomFilter that) {
        if (this == that || this.numBits != that.numBits || this.numHashFunctions != that.numHashFunctions) {
            throw new IllegalArgumentException("BloomFilters are not compatible for merging. this - " + this.toString() + " that - " + that.toString());
        }
        this.bitSet.putAll(that.bitSet);
    }

    public void reset() {
        this.bitSet.clear();
    }

    public static void serialize(OutputStream out, BloomFilter bloomFilter) throws IOException {
        DataOutputStream dataOutputStream = new DataOutputStream(out);
        dataOutputStream.writeByte(bloomFilter.numHashFunctions);
        dataOutputStream.writeInt(bloomFilter.getBitSet().length);
        for (long value : bloomFilter.getBitSet()) {
            dataOutputStream.writeLong(value);
        }
    }

    public static BloomFilter deserialize(InputStream in) throws IOException {
        if (in == null) {
            throw new IOException("Input stream is null");
        }
        try {
            DataInputStream dataInputStream = new DataInputStream(in);
            byte numHashFunc = dataInputStream.readByte();
            int numLongs = dataInputStream.readInt();
            long[] data = new long[numLongs];
            for (int i = 0; i < numLongs; ++i) {
                data[i] = dataInputStream.readLong();
            }
            return new BloomFilter(data, numHashFunc);
        }
        catch (RuntimeException e) {
            IOException io = new IOException("Unable to deserialize BloomFilter");
            io.initCause(e);
            throw io;
        }
    }

    public static void mergeBloomFilterBytes(byte[] bf1Bytes, int bf1Start, int bf1Length, byte[] bf2Bytes, int bf2Start, int bf2Length) {
        int idx;
        if (bf1Length != bf2Length) {
            throw new IllegalArgumentException("bf1Length " + bf1Length + " does not match bf2Length " + bf2Length);
        }
        for (idx = 0; idx < 5; ++idx) {
            if (bf1Bytes[bf1Start + idx] == bf2Bytes[bf2Start + idx]) continue;
            throw new IllegalArgumentException("bf1 NumHashFunctions/NumBits does not match bf2");
        }
        for (idx = 5; idx < bf1Length; ++idx) {
            int n = bf1Start + idx;
            bf1Bytes[n] = (byte)(bf1Bytes[n] | bf2Bytes[bf2Start + idx]);
        }
    }

    public class BitSet {
        private final long[] data;

        public BitSet(long bits) {
            this(new long[(int)Math.ceil((double)bits / 64.0)]);
        }

        public BitSet(long[] data) {
            assert (data.length > 0) : "data length is zero!";
            this.data = data;
        }

        public void set(int index) {
            int n = index >>> 6;
            this.data[n] = this.data[n] | 1L << index;
        }

        public boolean get(int index) {
            return (this.data[index >>> 6] & 1L << index) != 0L;
        }

        public long bitSize() {
            return (long)this.data.length * 64L;
        }

        public long[] getData() {
            return this.data;
        }

        public void putAll(BitSet array) {
            assert (this.data.length == array.data.length) : "BitArrays must be of equal length (" + this.data.length + "!= " + array.data.length + ")";
            for (int i = 0; i < this.data.length; ++i) {
                int n = i;
                this.data[n] = this.data[n] | array.data[i];
            }
        }

        public void clear() {
            Arrays.fill(this.data, 0L);
        }
    }
}

