/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.io.IOException;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import org.apache.iceberg.ManifestEntry;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.ManifestFiles;
import org.apache.iceberg.ManifestReader;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.common.DynClasses;
import org.apache.iceberg.common.DynConstructors;
import org.apache.iceberg.common.DynMethods;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.hadoop.Configurable;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Iterables;
import org.apache.iceberg.relocated.com.google.common.collect.MapMaker;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.Tasks;
import org.apache.iceberg.util.ThreadPools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CatalogUtil {
    private static final Logger LOG = LoggerFactory.getLogger(CatalogUtil.class);
    public static final String ICEBERG_CATALOG_TYPE = "type";
    public static final String ICEBERG_CATALOG_TYPE_HADOOP = "hadoop";
    public static final String ICEBERG_CATALOG_TYPE_HIVE = "hive";
    public static final String ICEBERG_CATALOG_HIVE = "org.apache.iceberg.hive.HiveCatalog";
    public static final String ICEBERG_CATALOG_HADOOP = "org.apache.iceberg.hadoop.HadoopCatalog";

    private CatalogUtil() {
    }

    public static void dropTableData(FileIO io, TableMetadata metadata) {
        HashSet<String> manifestListsToDelete = Sets.newHashSet();
        HashSet<ManifestFile> manifestsToDelete = Sets.newHashSet();
        for (Snapshot snapshot : metadata.snapshots()) {
            Iterables.addAll(manifestsToDelete, snapshot.allManifests(io));
            if (snapshot.manifestListLocation() == null) continue;
            manifestListsToDelete.add(snapshot.manifestListLocation());
        }
        LOG.info("Manifests to delete: {}", (Object)Joiner.on(", ").join(manifestsToDelete));
        boolean gcEnabled = PropertyUtil.propertyAsBoolean(metadata.properties(), "gc.enabled", true);
        if (gcEnabled) {
            CatalogUtil.deleteFiles(io, manifestsToDelete);
        }
        Tasks.foreach(Iterables.transform(manifestsToDelete, ManifestFile::path)).executeWith(ThreadPools.getWorkerPool()).noRetry().suppressFailureWhenFinished().onFailure((manifest, exc) -> LOG.warn("Delete failed for manifest: {}", manifest, (Object)exc)).run(io::deleteFile);
        Tasks.foreach(manifestListsToDelete).executeWith(ThreadPools.getWorkerPool()).noRetry().suppressFailureWhenFinished().onFailure((list, exc) -> LOG.warn("Delete failed for manifest list: {}", list, (Object)exc)).run(io::deleteFile);
        Tasks.foreach(Iterables.transform(metadata.previousFiles(), TableMetadata.MetadataLogEntry::file)).executeWith(ThreadPools.getWorkerPool()).noRetry().suppressFailureWhenFinished().onFailure((metadataFile, exc) -> LOG.warn("Delete failed for previous metadata file: {}", metadataFile, (Object)exc)).run(io::deleteFile);
        Tasks.foreach(metadata.metadataFileLocation()).noRetry().suppressFailureWhenFinished().onFailure((metadataFile, exc) -> LOG.warn("Delete failed for metadata file: {}", metadataFile, (Object)exc)).run(io::deleteFile);
    }

    private static void deleteFiles(FileIO io, Set<ManifestFile> allManifests) {
        ConcurrentMap deletedFiles = new MapMaker().concurrencyLevel(ThreadPools.WORKER_THREAD_POOL_SIZE).weakKeys().makeMap();
        Tasks.foreach(allManifests).noRetry().suppressFailureWhenFinished().executeWith(ThreadPools.getWorkerPool()).onFailure((item, exc) -> LOG.warn("Failed to get deleted files: this may cause orphaned data files", (Throwable)exc)).run(manifest -> {
            try (ManifestReader<?> reader = ManifestFiles.open(manifest, io);){
                for (ManifestEntry manifestEntry : reader.entries()) {
                    String path = manifestEntry.file().path().toString().intern();
                    Boolean alreadyDeleted = deletedFiles.putIfAbsent(path, true);
                    if (alreadyDeleted != null && alreadyDeleted.booleanValue()) continue;
                    try {
                        io.deleteFile(path);
                    }
                    catch (RuntimeException e) {
                        LOG.warn("Delete failed for data file: {}", (Object)path, (Object)e);
                    }
                }
            }
            catch (IOException e) {
                throw new RuntimeIOException(e, "Failed to read manifest file: %s", manifest.path());
            }
        });
    }

    public static Catalog loadCatalog(String impl, String catalogName, Map<String, String> properties, Object hadoopConf) {
        Catalog catalog;
        DynConstructors.Ctor ctor;
        Preconditions.checkNotNull(impl, "Cannot initialize custom Catalog, impl class name is null");
        try {
            ctor = DynConstructors.builder(Catalog.class).impl(impl, new Class[0]).buildChecked();
        }
        catch (NoSuchMethodException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize Catalog implementation %s: %s", impl, e.getMessage()), e);
        }
        try {
            catalog = (Catalog)ctor.newInstance(new Object[0]);
        }
        catch (ClassCastException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize Catalog, %s does not implement Catalog.", impl), e);
        }
        CatalogUtil.configureHadoopConf(catalog, hadoopConf);
        catalog.initialize(catalogName, properties);
        return catalog;
    }

    public static Catalog buildIcebergCatalog(String name, Map<String, String> options, Object conf) {
        String catalogImpl = options.get("catalog-impl");
        if (catalogImpl == null) {
            String catalogType = PropertyUtil.propertyAsString(options, ICEBERG_CATALOG_TYPE, ICEBERG_CATALOG_TYPE_HIVE);
            switch (catalogType.toLowerCase(Locale.ENGLISH)) {
                case "hive": {
                    catalogImpl = ICEBERG_CATALOG_HIVE;
                    break;
                }
                case "hadoop": {
                    catalogImpl = ICEBERG_CATALOG_HADOOP;
                    break;
                }
                default: {
                    throw new UnsupportedOperationException("Unknown catalog type: " + catalogType);
                }
            }
        } else {
            String catalogType = options.get(ICEBERG_CATALOG_TYPE);
            Preconditions.checkArgument(catalogType == null, "Cannot create catalog %s, both type and catalog-impl are set: type=%s, catalog-impl=%s", (Object)name, (Object)catalogType, (Object)catalogImpl);
        }
        return CatalogUtil.loadCatalog(catalogImpl, name, options, conf);
    }

    public static FileIO loadFileIO(String impl, Map<String, String> properties, Object hadoopConf) {
        FileIO fileIO;
        DynConstructors.Ctor ctor;
        LOG.info("Loading custom FileIO implementation: {}", (Object)impl);
        try {
            ctor = DynConstructors.builder(FileIO.class).loader(CatalogUtil.class.getClassLoader()).impl(impl, new Class[0]).buildChecked();
        }
        catch (NoSuchMethodException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize FileIO, missing no-arg constructor: %s", impl), e);
        }
        try {
            fileIO = (FileIO)ctor.newInstance(new Object[0]);
        }
        catch (ClassCastException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize FileIO, %s does not implement FileIO.", impl), e);
        }
        CatalogUtil.configureHadoopConf(fileIO, hadoopConf);
        fileIO.initialize(properties);
        return fileIO;
    }

    public static void configureHadoopConf(Object maybeConfigurable, Object conf) {
        DynMethods.BoundMethod setConf;
        Class configurationClass;
        Class configurableInterface;
        Preconditions.checkArgument(maybeConfigurable != null, "Cannot configure: null Configurable");
        if (conf == null) {
            return;
        }
        if (maybeConfigurable instanceof Configurable) {
            ((Configurable)maybeConfigurable).setConf(conf);
            return;
        }
        ClassLoader maybeConfigurableLoader = maybeConfigurable.getClass().getClassLoader();
        try {
            configurableInterface = DynClasses.builder().loader(maybeConfigurableLoader).impl("org.apache.hadoop.conf.Configurable").buildChecked();
        }
        catch (ClassNotFoundException e) {
            return;
        }
        if (!configurableInterface.isInstance(maybeConfigurable)) {
            return;
        }
        try {
            configurationClass = DynClasses.builder().loader(maybeConfigurableLoader).impl("org.apache.hadoop.conf.Configuration").buildChecked();
        }
        catch (ClassNotFoundException e) {
            throw new UnsupportedOperationException("Failed to load Configuration after loading Configurable", e);
        }
        ValidationException.check(configurationClass.isInstance(conf), "%s is not an instance of Configuration from the classloader for %s", conf, maybeConfigurable);
        try {
            setConf = DynMethods.builder("setConf").impl(configurableInterface, configurationClass).buildChecked().bind(maybeConfigurable);
        }
        catch (NoSuchMethodException e) {
            throw new UnsupportedOperationException("Failed to load Configuration.setConf after loading Configurable", e);
        }
        setConf.invoke(conf);
    }
}

