/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.sql.parser.sql.dialect.handler.dml;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.shardingsphere.sql.parser.sql.common.segment.dml.order.OrderBySegment;
import org.apache.shardingsphere.sql.parser.sql.common.segment.dml.pagination.limit.LimitSegment;
import org.apache.shardingsphere.sql.parser.sql.common.segment.generic.OutputSegment;
import org.apache.shardingsphere.sql.parser.sql.common.segment.generic.WithSegment;
import org.apache.shardingsphere.sql.parser.sql.common.statement.dml.DeleteStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.handler.SQLStatementHandler;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.mysql.MySQLStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.mysql.dml.MySQLDeleteStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.sqlserver.SQLServerStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.sqlserver.dml.SQLServerDeleteStatement;

import java.util.Optional;

/**
 * Delete statement handler for different dialect SQL statements.
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class DeleteStatementHandler implements SQLStatementHandler {
    
    /**
     * Get order by segment.
     *
     * @param deleteStatement delete statement
     * @return order by segment
     */
    public static Optional<OrderBySegment> getOrderBySegment(final DeleteStatement deleteStatement) {
        if (deleteStatement instanceof MySQLStatement) {
            return ((MySQLDeleteStatement) deleteStatement).getOrderBy();
        }
        return Optional.empty();
    }
    
    /**
     * Get limit segment.
     *
     * @param deleteStatement delete statement
     * @return limit segment
     */
    public static Optional<LimitSegment> getLimitSegment(final DeleteStatement deleteStatement) {
        if (deleteStatement instanceof MySQLStatement) {
            return ((MySQLDeleteStatement) deleteStatement).getLimit();
        }
        return Optional.empty();
    }
    
    /**
     * Get output segment.
     * 
     * @param deleteStatement delete statement
     * @return output segment
     */
    public static Optional<OutputSegment> getOutputSegment(final DeleteStatement deleteStatement) {
        if (deleteStatement instanceof SQLServerStatement) {
            return ((SQLServerDeleteStatement) deleteStatement).getOutputSegment();
        }
        return Optional.empty();
    }
    
    /**
     * Get with segment.
     *
     * @param deleteStatement delete statement
     * @return with segment
     */
    public static Optional<WithSegment> getWithSegment(final DeleteStatement deleteStatement) {
        if (deleteStatement instanceof SQLServerStatement) {
            return ((SQLServerDeleteStatement) deleteStatement).getWithSegment();
        }
        return Optional.empty();
    }
    
    /**
     * Set order by segment.
     *
     * @param deleteStatement delete statement
     * @param orderBySegment order by segment
     */
    public static void setOrderBySegment(final DeleteStatement deleteStatement, final OrderBySegment orderBySegment) {
        if (deleteStatement instanceof MySQLStatement) {
            ((MySQLDeleteStatement) deleteStatement).setOrderBy(orderBySegment);
        }
    }
    
    /**
     * Set limit segment.
     *
     * @param deleteStatement delete statement
     * @param limitSegment limit segment
     */
    public static void setLimitSegment(final DeleteStatement deleteStatement, final LimitSegment limitSegment) {
        if (deleteStatement instanceof MySQLStatement) {
            ((MySQLDeleteStatement) deleteStatement).setLimit(limitSegment);
        }
    }
    
    /**
     * Set output segment.
     *
     * @param deleteStatement delete statement
     * @param outputSegment output segment
     */
    public static void setOutputSegment(final DeleteStatement deleteStatement, final OutputSegment outputSegment) {
        if (deleteStatement instanceof SQLServerStatement) {
            ((SQLServerDeleteStatement) deleteStatement).setOutputSegment(outputSegment);
        }
    }
    
    /**
     * Set with segment.
     *
     * @param deleteStatement delete statement
     * @param withSegment with segment
     */
    public static void setWithSegment(final DeleteStatement deleteStatement, final WithSegment withSegment) {
        if (deleteStatement instanceof SQLServerStatement) {
            ((SQLServerDeleteStatement) deleteStatement).setWithSegment(withSegment);
        }
    }
}
