/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.sql.parser.sql.dialect.handler.ddl;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.shardingsphere.sql.parser.sql.common.segment.generic.table.SimpleTableSegment;
import org.apache.shardingsphere.sql.parser.sql.common.statement.ddl.DropIndexStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.handler.SQLStatementHandler;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.mysql.MySQLStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.mysql.ddl.MySQLDropIndexStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.opengauss.OpenGaussStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.opengauss.ddl.OpenGaussDropIndexStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.postgresql.PostgreSQLStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.postgresql.ddl.PostgreSQLDropIndexStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.sqlserver.SQLServerStatement;
import org.apache.shardingsphere.sql.parser.sql.dialect.statement.sqlserver.ddl.SQLServerDropIndexStatement;

import java.util.Optional;

/**
 * Drop index statement handler for different dialect SQL statements.
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class DropIndexStatementHandler implements SQLStatementHandler {
    
    /**
     * Get simple table segment.
     *
     * @param dropIndexStatement drop index statement
     * @return simple table segment
     */
    public static Optional<SimpleTableSegment> getSimpleTableSegment(final DropIndexStatement dropIndexStatement) {
        if (dropIndexStatement instanceof MySQLStatement) {
            return ((MySQLDropIndexStatement) dropIndexStatement).getTable();
        }
        if (dropIndexStatement instanceof SQLServerStatement) {
            return ((SQLServerDropIndexStatement) dropIndexStatement).getTable();
        }
        return Optional.empty();
    }
    
    /**
     * Judge whether contains exist clause or not.
     *
     * @param dropIndexStatement drop index statement
     * @return whether contains exist clause or not
     */
    public static boolean ifExists(final DropIndexStatement dropIndexStatement) {
        if (dropIndexStatement instanceof PostgreSQLStatement) {
            return ((PostgreSQLDropIndexStatement) dropIndexStatement).isIfExists();
        }
        if (dropIndexStatement instanceof SQLServerStatement) {
            return ((SQLServerDropIndexStatement) dropIndexStatement).isIfExists();
        }
        if (dropIndexStatement instanceof OpenGaussStatement) {
            return ((OpenGaussDropIndexStatement) dropIndexStatement).isIfExists();
        }
        return false;
    }
}
