/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.CheckForNull;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.document.ClusterNodeInfoDocument;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.MissingLastRevSeeker;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.StableRevisionComparator;
import org.apache.jackrabbit.oak.plugins.document.UnsavedModifications;
import org.apache.jackrabbit.oak.plugins.document.mongo.MongoDocumentStore;
import org.apache.jackrabbit.oak.plugins.document.mongo.MongoMissingLastRevSeeker;
import org.apache.jackrabbit.oak.plugins.document.util.MapFactory;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LastRevRecoveryAgent {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final DocumentNodeStore nodeStore;
    private final MissingLastRevSeeker missingLastRevUtil;

    public LastRevRecoveryAgent(DocumentNodeStore nodeStore) {
        this.nodeStore = nodeStore;
        this.missingLastRevUtil = nodeStore.getDocumentStore() instanceof MongoDocumentStore ? new MongoMissingLastRevSeeker((MongoDocumentStore)nodeStore.getDocumentStore()) : new MissingLastRevSeeker(nodeStore.getDocumentStore());
    }

    public int recover(int clusterId) {
        ClusterNodeInfoDocument nodeInfo = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
        long leaseTime = 60000L;
        long asyncDelay = this.nodeStore.getAsyncDelay();
        if (nodeInfo != null && this.isRecoveryNeeded(nodeInfo)) {
            long leaseEnd = nodeInfo.getLeaseEndTime();
            NodeDocument root = this.missingLastRevUtil.getRoot();
            Revision lastRev = root.getLastRev().get(clusterId);
            long startTime = lastRev != null ? lastRev.getTimestamp() : leaseEnd - 60000L - asyncDelay;
            this.log.info("Recovering candidates modified after: [{}] for clusterId [{}]", (Object)Utils.timestampToString(startTime), (Object)clusterId);
            return this.recoverCandidates(clusterId, startTime);
        }
        this.log.debug("No recovery needed for clusterId {}", (Object)clusterId);
        return 0;
    }

    public int recover(Iterator<NodeDocument> suspects, int clusterId) {
        return this.recover(suspects, clusterId, false);
    }

    public int recover(Iterator<NodeDocument> suspects, int clusterId, boolean dryRun) {
        UnsavedModifications unsaved = new UnsavedModifications();
        UnsavedModifications unsavedParents = new UnsavedModifications();
        ConcurrentMap<String, Revision> knownLastRevs = MapFactory.getInstance().create();
        long count = 0L;
        while (suspects.hasNext()) {
            Revision lastModifiedRev;
            Revision lastRevForParents;
            Revision currentLastRev;
            NodeDocument doc = suspects.next();
            if (++count % 100000L == 0L) {
                this.log.info("Scanned {} suspects so far...", (Object)count);
            }
            if ((currentLastRev = doc.getLastRev().get(clusterId)) != null) {
                knownLastRevs.put(doc.getPath(), currentLastRev);
            }
            if ((lastRevForParents = Utils.max(lastModifiedRev = this.determineLastModification(doc, clusterId), currentLastRev)) == null) continue;
            String path = doc.getPath();
            while (!PathUtils.denotesRoot(path)) {
                path = PathUtils.getParentPath(path);
                unsavedParents.put(path, lastRevForParents);
            }
        }
        for (String parentPath : unsavedParents.getPaths()) {
            Revision calcLastRev = unsavedParents.get(parentPath);
            Revision knownLastRev = (Revision)knownLastRevs.get(parentPath);
            if (knownLastRev != null && calcLastRev.compareRevisionTime(knownLastRev) <= 0) continue;
            unsaved.put(parentPath, calcLastRev);
        }
        int size = unsaved.getPaths().size();
        String updates = unsaved.toString();
        if (dryRun) {
            this.log.info("Dry run of lastRev recovery identified [{}] documents for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        } else {
            unsaved.persist(this.nodeStore, new ReentrantLock());
            this.log.info("Updated lastRev of [{}] documents while performing lastRev recovery for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        }
        return size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int recoverCandidates(int clusterId, long startTime) {
        boolean lockAcquired = this.missingLastRevUtil.acquireRecoveryLock(clusterId);
        if (!lockAcquired) {
            this.log.info("Last revision recovery already being performed by some other node. Would not attempt recovery");
            return 0;
        }
        Iterable<NodeDocument> suspects = this.missingLastRevUtil.getCandidates(startTime);
        this.log.debug("Performing Last Revision recovery for cluster {}", (Object)clusterId);
        try {
            int n = this.recover(suspects.iterator(), clusterId);
            return n;
        }
        finally {
            Utils.closeIfCloseable(suspects);
            this.missingLastRevUtil.releaseRecoveryLock(clusterId);
        }
    }

    @CheckForNull
    private Revision determineLastModification(NodeDocument doc, int clusterId) {
        ClusterPredicate cp = new ClusterPredicate(clusterId);
        Iterable revs = Iterables.mergeSorted((Iterable)ImmutableList.of((Object)Iterables.filter(doc.getLocalCommitRoot().keySet(), (Predicate)cp), (Object)Iterables.filter(doc.getLocalRevisions().keySet(), (Predicate)cp)), StableRevisionComparator.REVERSE);
        Revision lastModified = null;
        for (Revision rev : revs) {
            lastModified = Utils.max(lastModified, doc.getCommitRevision(rev));
        }
        return lastModified;
    }

    public boolean isRecoveryNeeded() {
        return this.missingLastRevUtil.isRecoveryNeeded(this.nodeStore.getClock().getTime());
    }

    public void performRecoveryIfNeeded() {
        if (this.isRecoveryNeeded()) {
            List<Integer> clusterIds = this.getRecoveryCandidateNodes();
            this.log.info("Starting last revision recovery for following clusterId {}", clusterIds);
            for (int clusterId : clusterIds) {
                this.recover(clusterId);
            }
        }
    }

    public List<Integer> getRecoveryCandidateNodes() {
        Iterable<ClusterNodeInfoDocument> clusters = this.missingLastRevUtil.getAllClusters();
        ArrayList candidateClusterNodes = Lists.newArrayList();
        for (ClusterNodeInfoDocument nodeInfo : clusters) {
            if (!this.isRecoveryNeeded(nodeInfo)) continue;
            candidateClusterNodes.add(Integer.valueOf(nodeInfo.getId()));
        }
        return candidateClusterNodes;
    }

    private boolean isRecoveryNeeded(ClusterNodeInfoDocument nodeInfo) {
        return nodeInfo != null && nodeInfo.isActive() && this.nodeStore.getClock().getTime() > nodeInfo.getLeaseEndTime() && !nodeInfo.isBeingRecovered();
    }

    private static class ClusterPredicate
    implements Predicate<Revision> {
        private final int clusterId;

        private ClusterPredicate(int clusterId) {
            this.clusterId = clusterId;
        }

        public boolean apply(Revision input) {
            return this.clusterId == input.getClusterId();
        }
    }
}

