/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jackrabbit.oak.plugins.document.rdb;

import java.io.Closeable;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.sql.DataSource;

import org.apache.jackrabbit.oak.commons.StringUtils;
import org.apache.jackrabbit.oak.plugins.blob.CachingBlobStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.spi.blob.AbstractBlobStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.AbstractIterator;

public class RDBBlobStore extends CachingBlobStore implements Closeable {

    /**
     * Creates a {@linkplain RDBBlobStore} instance using the provided
     * {@link DataSource} using the given {@link RDBOptions}.
     */
    public RDBBlobStore(DataSource ds, RDBOptions options) {
        try {
            initialize(ds, options);
        } catch (Exception ex) {
            throw new DocumentStoreException("initializing RDB blob store", ex);
        }
    }

    /**
     * Creates a {@linkplain RDBBlobStore} instance using the provided
     * {@link DataSource} using default {@link RDBOptions}.
     */
    public RDBBlobStore(DataSource ds) {
        this(ds, new RDBOptions());
    }

    @Override
    public void close() {
        if (!this.tablesToBeDropped.isEmpty()) {
            LOG.debug("attempting to drop: " + this.tablesToBeDropped);
            for (String tname : this.tablesToBeDropped) {
                Connection con = null;
                try {
                    con = this.ch.getRWConnection();
                    Statement stmt = null;
                    try {
                        stmt = con.createStatement();
                        stmt.execute("drop table " + tname);
                        stmt.close();
                        stmt = null;
                        con.commit();
                    } catch (SQLException ex) {
                        this.ch.closeStatement(stmt);
                        LOG.debug("attempting to drop: " + tname);
                    }
                } catch (SQLException ex) {
                    LOG.debug("attempting to drop: " + tname);
                } finally {
                    try {
                        if (con != null) {
                            con.close();
                        }
                    } catch (SQLException ex) {
                        LOG.debug("on close ", ex);
                    }
                }
            }
        }
        this.ch = null;
    }

    @Override
    protected void finalize() {
        if (this.ch != null && this.callStack != null) {
            LOG.debug("finalizing RDBDocumentStore that was not disposed", this.callStack);
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(RDBBlobStore.class);

    // blob size we need to support
    private static final int MINBLOB = 2 * 1024 * 1024;

    // ID size we need to support; is 2 * (hex) size of digest length
    private static final int IDSIZE;
    static {
        try {
            MessageDigest md = MessageDigest.getInstance(AbstractBlobStore.HASH_ALGORITHM);
            IDSIZE = md.getDigestLength() * 2;
        } catch (NoSuchAlgorithmException ex) {
            LOG.error ("can't determine digest length for blob store", ex);
            throw new RuntimeException(ex);
        }
    }

    private Exception callStack;

    private RDBConnectionHandler ch;

    // from options
    private String tnData;
    private String tnMeta;
    private Set<String> tablesToBeDropped = new HashSet<String>();

    private static void versionCheck(DatabaseMetaData md, int xmaj, int xmin, String description) throws SQLException {
        int maj = md.getDatabaseMajorVersion();
        int min = md.getDatabaseMinorVersion();
        if (maj < xmaj || (maj == xmaj && min < xmin)) {
            LOG.info("Unsupported " + description + " version: " + maj + "." + min + ", expected at least " + xmaj + "." + xmin);
        }
    }

    /**
     * Defines variation in the capabilities of different RDBs.
     */
    protected enum DB {
        H2("H2") {
            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                versionCheck(md, 1, 4, description);
            }
        },

        DB2("DB2") {
            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                versionCheck(md, 10, 5, description);
            }

            @Override
            public String getDataTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varchar(" + IDSIZE + ") not null primary key, DATA blob(" + MINBLOB + "))";
            }
        },

        MSSQL("Microsoft SQL Server") {
            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                versionCheck(md, 11, 0, description);
            }

            @Override
            public String getDataTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varchar(" + IDSIZE + ") not null primary key, DATA varbinary(max))";
            }
        },

        MYSQL("MySQL") {
            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                versionCheck(md, 5, 5, description);
            }

            @Override
            public String getDataTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varchar(" + IDSIZE + ") not null primary key, DATA mediumblob)";
            }
        },

        ORACLE("Oracle") {
            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                versionCheck(md, 12, 1, description);
            }

            @Override
            public String getMetaTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varchar(" + IDSIZE + ") not null primary key, LVL number, LASTMOD number)";
            }
        },

        POSTGRES("PostgreSQL") {
            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                versionCheck(md, 9, 3, description);
            }

            @Override
            public String getDataTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varchar(" + IDSIZE + ") not null primary key, DATA bytea)";
            }
        },

        DEFAULT("default") {
        };

        public void checkVersion(DatabaseMetaData md) throws SQLException {
            LOG.info("Unknown database type: " + md.getDatabaseProductName());
        }

        public String getDataTableCreationStatement(String tableName) {
            return "create table " + tableName + " (ID varchar(" + IDSIZE + ") not null primary key, DATA blob)";
        }

        public String getMetaTableCreationStatement(String tableName) {
            return "create table " + tableName + " (ID varchar(" + IDSIZE + ") not null primary key, LVL int, LASTMOD bigint)";
        }

        protected String description;

        private DB(String description) {
            this.description = description;
        }

        @Override
        public String toString() {
            return this.description;
        }

        @Nonnull
        public static DB getValue(String desc) {
            for (DB db : DB.values()) {
                if (db.description.equals(desc)) {
                    return db;
                } else if (db == DB2 && desc.startsWith("DB2/")) {
                    return db;
                }
            }

            LOG.error("DB type " + desc + " unknown, trying default settings");
            DEFAULT.description = desc + " - using default settings";
            return DEFAULT;
        }
    }

    private void initialize(DataSource ds, RDBOptions options) throws Exception {

        this.tnData = RDBJDBCTools.createTableName(options.getTablePrefix(), "DATASTORE_DATA");
        this.tnMeta = RDBJDBCTools.createTableName(options.getTablePrefix(), "DATASTORE_META");

        this.ch = new RDBConnectionHandler(ds);
        Connection con = this.ch.getRWConnection();
        DatabaseMetaData md = con.getMetaData();
        String dbDesc = md.getDatabaseProductName() + " " + md.getDatabaseProductVersion();
        String driverDesc = md.getDriverName() + " " + md.getDriverVersion();
        String dbUrl = md.getURL();

        List<String> tablesCreated = new ArrayList<String>();
        List<String> tablesPresent = new ArrayList<String>();

        Statement createStatement = null;

        try {
            for (String tableName : new String[] { this.tnData, this.tnMeta }) {
                PreparedStatement checkStatement = null;
                try {
                    checkStatement = con.prepareStatement("select ID from " + tableName + " where ID = ?");
                    checkStatement.setString(1, "0");
                    checkStatement.executeQuery();
                    checkStatement.close();
                    checkStatement = null;
                    con.commit();
                    tablesPresent.add(tableName);
                } catch (SQLException ex) {
                    this.ch.closeStatement(checkStatement);
 
                    // table does not appear to exist
                    con.rollback();

                    DB db = DB.getValue(md.getDatabaseProductName());

                    createStatement = con.createStatement();

                    if (this.tnMeta.equals(tableName)) {
                        String ct = db.getMetaTableCreationStatement(tableName);
                        createStatement.execute(ct);
                    } else {
                        String ct = db.getDataTableCreationStatement(tableName);
                        createStatement.execute(ct);
                    }

                    createStatement.close();
                    createStatement = null;

                    con.commit();

                    tablesCreated.add(tableName);
                }
            }

            if (options.isDropTablesOnClose()) {
                tablesToBeDropped.addAll(tablesCreated);
            }

            LOG.info("RDBBlobStore instantiated for database " + dbDesc + ", using driver: " + driverDesc + ", connecting to: "
                    + dbUrl);
            if (!tablesPresent.isEmpty()) {
                LOG.info("Tables present upon startup: " + tablesPresent);
            }
            if (!tablesCreated.isEmpty()) {
                LOG.info("Tables created upon startup: " + tablesCreated
                        + (options.isDropTablesOnClose() ? " (will be dropped on exit)" : ""));
            }

            this.callStack = LOG.isDebugEnabled() ? new Exception("call stack of RDBBlobStore creation") : null;
        } finally {
            this.ch.closeStatement(createStatement);
            this.ch.closeConnection(con);
        }
    }

    private long minLastModified;

    @Override
    protected void storeBlock(byte[] digest, int level, byte[] data) throws IOException {
        try {
            storeBlockInDatabase(digest, level, data);
        } catch (SQLException e) {
            throw new IOException(e);
        }
    }

    private void storeBlockInDatabase(byte[] digest, int level, byte[] data) throws SQLException {

        String id = StringUtils.convertBytesToHex(digest);
        cache.put(id, data);
        Connection con = this.ch.getRWConnection();

        try {
            long now = System.currentTimeMillis();
            PreparedStatement prep = con.prepareStatement("update " + this.tnMeta + " set LASTMOD = ? where ID = ?");
            int count;
            try {
                prep.setLong(1, now);
                prep.setString(2, id);
                count = prep.executeUpdate();
            }
            catch (SQLException ex) {
                LOG.error("trying to update metadata", ex);
                throw new RuntimeException("trying to update metadata", ex);
            }
            finally {
                prep.close();
            }
            if (count == 0) {
                try {
                    prep = con.prepareStatement("insert into " + this.tnData + "(ID, DATA) values(?, ?)");
                    try {
                        prep.setString(1, id);
                        prep.setBytes(2, data);
                        prep.execute();
                    } finally {
                        prep.close();
                    }
                } catch (SQLException ex) {
                    // TODO: this code used to ignore exceptions here, assuming that it might be a case where the blob is already in the database (maybe this requires inspecting the exception code)
                    String message = "insert document failed for id " + id + " with length " + data.length + " (check max size of datastore_data.data)";
                    LOG.error(message, ex);
                    throw new RuntimeException(message, ex);
                }
                try {
                    prep = con.prepareStatement("insert into " + this.tnMeta + "(ID, LVL, LASTMOD) values(?, ?, ?)");
                    try {
                        prep.setString(1, id);
                        prep.setInt(2, level);
                        prep.setLong(3, now);
                        prep.execute();
                    } finally {
                        prep.close();
                    }
                } catch (SQLException e) {
                    // already exists - ok
                }
            }
        } finally {
            con.commit();
            this.ch.closeConnection(con);
        }
    }

    // needed in test
    protected byte[] readBlockFromBackend(byte[] digest) throws Exception {
        String id = StringUtils.convertBytesToHex(digest);
        Connection con = this.ch.getROConnection();
        byte[] data;

        try {
            PreparedStatement prep = con.prepareStatement("select DATA from " + this.tnData + " where ID = ?");
            try {
                prep.setString(1, id);
                ResultSet rs = prep.executeQuery();
                if (!rs.next()) {
                    throw new IOException("Datastore block " + id + " not found");
                }
                data = rs.getBytes(1);
            } finally {
                prep.close();
            }
        } finally {
            con.commit();
            this.ch.closeConnection(con);
        }
        return data;
    }

    @Override
    protected byte[] readBlockFromBackend(BlockId blockId) throws Exception {

        String id = StringUtils.convertBytesToHex(blockId.getDigest());
        byte[] data = cache.get(id);

        if (data == null) {
            Connection con = this.ch.getROConnection();

            try {
                PreparedStatement prep = con.prepareStatement("select DATA from " + this.tnData + " where ID = ?");
                try {
                    prep.setString(1, id);
                    ResultSet rs = prep.executeQuery();
                    if (!rs.next()) {
                        throw new IOException("Datastore block " + id + " not found");
                    }
                    data = rs.getBytes(1);
                } finally {
                    prep.close();
                }
                cache.put(id, data);
            } finally {
                con.commit();
                this.ch.closeConnection(con);
            }
        }
        // System.out.println("    read block " + id + " blockLen: " +
        // data.length + " [0]: " + data[0]);
        if (blockId.getPos() == 0) {
            return data;
        }
        int len = (int) (data.length - blockId.getPos());
        if (len < 0) {
            return new byte[0];
        }
        byte[] d2 = new byte[len];
        System.arraycopy(data, (int) blockId.getPos(), d2, 0, len);
        return d2;
    }

    @Override
    public void startMark() throws IOException {
        minLastModified = System.currentTimeMillis();
        markInUse();
    }

    @Override
    protected boolean isMarkEnabled() {
        return minLastModified != 0;
    }

    @Override
    protected void mark(BlockId blockId) throws Exception {
        Connection con = this.ch.getRWConnection();
        PreparedStatement prep = null;
        try {
            if (minLastModified == 0) {
                return;
            }
            String id = StringUtils.convertBytesToHex(blockId.getDigest());
            prep = con.prepareStatement("update " + this.tnMeta + " set LASTMOD = ? where ID = ? and LASTMOD < ?");
            prep.setLong(1, System.currentTimeMillis());
            prep.setString(2, id);
            prep.setLong(3, minLastModified);
            prep.executeUpdate();
            prep.close();
        } finally {
            this.ch.closeStatement(prep);
            con.commit();
            this.ch.closeConnection(con);
        }
    }

    @Override
    public int sweep() throws IOException {
        try {
            return sweepFromDatabase();
        } catch (SQLException e) {
            throw new IOException(e);
        }
    }

    private int sweepFromDatabase() throws SQLException {
        Connection con = this.ch.getRWConnection();
        PreparedStatement prepCheck = null, prepDelMeta = null, prepDelData = null;
        ResultSet rs = null;
        try {
            int count = 0;
            prepCheck = con.prepareStatement("select ID from " + this.tnMeta + " where LASTMOD < ?");
            prepCheck.setLong(1, minLastModified);
            rs = prepCheck.executeQuery();
            ArrayList<String> ids = new ArrayList<String>();
            while (rs.next()) {
                ids.add(rs.getString(1));
            }
            rs.close();
            rs = null;
            prepCheck.close();
            prepCheck = null;

            prepDelMeta = con.prepareStatement("delete from " + this.tnMeta + " where ID = ?");
            prepDelData = con.prepareStatement("delete from " + this.tnData + " where ID = ?");

            for (String id : ids) {
                prepDelMeta.setString(1, id);
                prepDelMeta.execute();
                prepDelData.setString(1, id);
                prepDelData.execute();
                count++;
            }
            prepDelMeta.close();
            prepDelMeta = null;
            prepDelData.close();
            prepDelData = null;
            minLastModified = 0;
            return count;
        } finally {
            this.ch.closeResultSet(rs);
            this.ch.closeStatement(prepCheck);
            this.ch.closeStatement(prepDelMeta);
            this.ch.closeStatement(prepDelData);
            con.commit();
            this.ch.closeConnection(con);
        }
    }

    @Override
    public boolean deleteChunks(List<String> chunkIds, long maxLastModifiedTime) throws Exception {

        // sanity check
        if (chunkIds.isEmpty()) {
            // sanity check, nothing to do
            return true;
        }

        Connection con = this.ch.getRWConnection();
        PreparedStatement prepMeta = null;
        PreparedStatement prepData = null;

        try {
            StringBuilder inClause = new StringBuilder();
            int batch = chunkIds.size();
            for (int i = 0; i < batch; i++) {
                inClause.append('?');
                if (i != batch - 1) {
                    inClause.append(',');
                }
            }

            if (maxLastModifiedTime > 0) {
                prepMeta = con.prepareStatement("delete from " + this.tnMeta + " where ID in (" + inClause.toString()
                        + ") and LASTMOD <= ?");
                prepMeta.setLong(batch + 1, maxLastModifiedTime);

                prepData = con.prepareStatement("delete from " + this.tnData + " where ID in (" + inClause.toString()
                        + ") and not exists(select * from " + this.tnMeta + " m where ID = m.ID and m.LASTMOD <= ?)");
                prepData.setLong(batch + 1, maxLastModifiedTime);
            } else {
                prepMeta = con.prepareStatement("delete from " + this.tnMeta + " where ID in (" + inClause.toString() + ")");
                prepData = con.prepareStatement("delete from " + this.tnData + " where ID in (" + inClause.toString() + ")");
            }

            for (int idx = 0; idx < batch; idx++) {
                prepMeta.setString(idx + 1, chunkIds.get(idx));
                prepData.setString(idx + 1, chunkIds.get(idx));
            }

            prepMeta.execute();
            prepData.execute();
            prepMeta.close();
            prepMeta = null;
            prepData.close();
            prepData = null;
        } finally {
            this.ch.closeStatement(prepMeta);
            this.ch.closeStatement(prepData);
            con.commit();
            this.ch.closeConnection(con);
        }

        return true;
    }

    @Override
    public Iterator<String> getAllChunkIds(long maxLastModifiedTime) throws Exception {
        return new ChunkIdIterator(this.ch, maxLastModifiedTime, this.tnMeta);
    }

    /**
     * Reads chunk IDs in batches.
     */
    private static class ChunkIdIterator extends AbstractIterator<String> {

        private long maxLastModifiedTime;
        private RDBConnectionHandler ch;
        private static int BATCHSIZE = 1024 * 64;
        private List<String> results = new LinkedList<String>();
        private String lastId = null;
        private String metaTable;

        public ChunkIdIterator(RDBConnectionHandler ch, long maxLastModifiedTime, String metaTable) {
            this.maxLastModifiedTime = maxLastModifiedTime;
            this.ch = ch;
            this.metaTable = metaTable;
        }

        @Override
        protected String computeNext() {
            if (!results.isEmpty()) {
                return results.remove(0);
            } else {
                // need to refill
                if (refill()) {
                    return computeNext();
                } else {
                    return endOfData();
                }
            }
        }

        private boolean refill() {
            StringBuffer query = new StringBuffer();
            query.append("select ID from " + metaTable);
            if (maxLastModifiedTime > 0) {
                query.append(" where LASTMOD <= ?");
                if (lastId != null) {
                    query.append(" and ID > ?");
                }
            } else {
                if (lastId != null) {
                    query.append(" where ID > ?");
                }
            }
            query.append(" order by ID");

            Connection connection = null;
            try {
                connection = this.ch.getROConnection();
                PreparedStatement prep = null;
                ResultSet rs = null;
                try {
                    prep = connection.prepareStatement(query.toString());
                    int idx = 1;
                    if (maxLastModifiedTime > 0) {
                        prep.setLong(idx++, maxLastModifiedTime);
                    }
                    if (lastId != null) {
                        prep.setString(idx, lastId);
                    }
                    prep.setFetchSize(BATCHSIZE);
                    rs = prep.executeQuery();
                    while (rs.next()) {
                        lastId = rs.getString(1);
                        results.add(lastId);
                    }
                    rs.close();
                    rs = null;
                    return !results.isEmpty();
                } finally {
                    this.ch.closeResultSet(rs);
                    this.ch.closeStatement(prep);
                    connection.commit();
                    this.ch.closeConnection(connection);
                }
            } catch (SQLException ex) {
                LOG.debug("error executing ID lookup", ex);
                this.ch.rollbackConnection(connection);
                this.ch.closeConnection(connection);
                return false;
            }
        }
    }
}
