/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Joiner;
import com.google.common.base.StandardSystemProperty;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterators;
import com.google.common.collect.UnmodifiableIterator;
import java.io.IOException;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.jackrabbit.oak.commons.sort.StringSort;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.NodeDocumentIdComparator;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.VersionGCSupport;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VersionGarbageCollector {
    private static final int DELETE_BATCH_SIZE = 450;
    private final DocumentNodeStore nodeStore;
    private final VersionGCSupport versionStore;
    private int overflowToDiskThreshold = 100000;
    private static final Logger log = LoggerFactory.getLogger(VersionGarbageCollector.class);
    private static final Set<NodeDocument.SplitDocType> GC_TYPES = EnumSet.of(NodeDocument.SplitDocType.DEFAULT_LEAF, NodeDocument.SplitDocType.COMMIT_ROOT_ONLY);

    VersionGarbageCollector(DocumentNodeStore nodeStore, VersionGCSupport gcSupport) {
        this.nodeStore = nodeStore;
        this.versionStore = gcSupport;
    }

    public VersionGCStats gc(long maxRevisionAge, TimeUnit unit) throws IOException {
        long maxRevisionAgeInMillis = unit.toMillis(maxRevisionAge);
        Stopwatch sw = Stopwatch.createStarted();
        VersionGCStats stats = new VersionGCStats();
        long oldestRevTimeStamp = this.nodeStore.getClock().getTime() - maxRevisionAgeInMillis;
        Revision headRevision = this.nodeStore.getHeadRevision();
        log.info("Starting revision garbage collection. Revisions older than [{}] would be removed", (Object)Utils.timestampToString(oldestRevTimeStamp));
        Revision checkpoint = this.nodeStore.getCheckpoints().getOldestRevisionToKeep();
        if (checkpoint != null && checkpoint.getTimestamp() < oldestRevTimeStamp) {
            log.info("Ignoring version gc as valid checkpoint [{}] found while need to collect versions older than [{}]", (Object)checkpoint.toReadableString(), (Object)Utils.timestampToString(oldestRevTimeStamp));
            stats.ignoredGCDueToCheckPoint = true;
            return stats;
        }
        this.collectDeletedDocuments(stats, headRevision, oldestRevTimeStamp);
        this.collectSplitDocuments(stats, oldestRevTimeStamp);
        sw.stop();
        log.info("Version garbage collected in {}. {}", (Object)sw, (Object)stats);
        return stats;
    }

    public void setOverflowToDiskThreshold(int overflowToDiskThreshold) {
        this.overflowToDiskThreshold = overflowToDiskThreshold;
    }

    private void collectSplitDocuments(VersionGCStats stats, long oldestRevTimeStamp) {
        this.versionStore.deleteSplitDocuments(GC_TYPES, oldestRevTimeStamp, stats);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void collectDeletedDocuments(VersionGCStats stats, Revision headRevision, long oldestRevTimeStamp) throws IOException {
        int docsTraversed = 0;
        int progressBatchSize = 10000;
        StringSort docIdsToDelete = new StringSort(this.overflowToDiskThreshold, NodeDocumentIdComparator.INSTANCE);
        try {
            stats.collectDeletedDocs.start();
            Iterable<NodeDocument> itr = this.versionStore.getPossiblyDeletedDocs(oldestRevTimeStamp);
            try {
                for (NodeDocument doc : itr) {
                    if (++docsTraversed % 10000 == 0) {
                        log.info("Iterated through {} documents so far. {} found to be deleted", (Object)docsTraversed, (Object)docIdsToDelete.getSize());
                    }
                    if (doc.getNodeAtRevision(this.nodeStore, headRevision, null) != null) continue;
                    docIdsToDelete.add(doc.getId());
                    for (NodeDocument prevDoc : ImmutableList.copyOf(doc.getAllPreviousDocs())) {
                        docIdsToDelete.add(prevDoc.getId());
                    }
                }
            }
            finally {
                Utils.closeIfCloseable(itr);
            }
            stats.collectDeletedDocs.stop();
            if (docIdsToDelete.isEmpty()) {
                return;
            }
            docIdsToDelete.sort();
            log.info("Proceeding to delete [{}] documents", (Object)docIdsToDelete.getSize());
            stats.deleteDeletedDocs.start();
            UnmodifiableIterator idListItr = Iterators.partition(docIdsToDelete.getIds(), (int)450);
            int deletedCount = 0;
            int lastLoggedCount = 0;
            while (idListItr.hasNext()) {
                List deletionBatch = (List)idListItr.next();
                deletedCount += deletionBatch.size();
                if (log.isDebugEnabled()) {
                    StringBuilder sb = new StringBuilder("Performing batch deletion of documents with following ids. \n");
                    Joiner.on((String)StandardSystemProperty.LINE_SEPARATOR.value()).appendTo(sb, (Iterable)deletionBatch);
                    log.debug(sb.toString());
                }
                log.debug("Deleted [{}] documents so far", (Object)deletedCount);
                if (deletedCount - lastLoggedCount >= 10000) {
                    lastLoggedCount = deletedCount;
                    double progress = (double)deletedCount * 1.0 / (double)docIdsToDelete.getSize() * 100.0;
                    String msg = String.format("Deleted %d (%1.2f%%) documents so far", deletedCount, progress);
                    log.info(msg);
                }
                this.nodeStore.getDocumentStore().remove(Collection.NODES, deletionBatch);
            }
            this.nodeStore.invalidateDocChildrenCache();
            stats.deleteDeletedDocs.stop();
            stats.deletedDocGCCount = (int)((long)stats.deletedDocGCCount + docIdsToDelete.getSize());
        }
        finally {
            docIdsToDelete.close();
        }
    }

    public static class VersionGCStats {
        boolean ignoredGCDueToCheckPoint;
        int deletedDocGCCount;
        int splitDocGCCount;
        int intermediateSplitDocGCCount;
        final Stopwatch collectDeletedDocs = Stopwatch.createUnstarted();
        final Stopwatch deleteDeletedDocs = Stopwatch.createUnstarted();

        public String toString() {
            return "VersionGCStats{ignoredGCDueToCheckPoint=" + this.ignoredGCDueToCheckPoint + ", deletedDocGCCount=" + this.deletedDocGCCount + ", splitDocGCCount=" + this.splitDocGCCount + ", intermediateSplitDocGCCount=" + this.intermediateSplitDocGCCount + ", timeToCollectDeletedDocs=" + this.collectDeletedDocs + ", timeTakenToDeleteDocs=" + this.deleteDeletedDocs + '}';
        }
    }
}

