/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document.rdb;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.Striped;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.locks.Lock;
import java.util.zip.GZIPOutputStream;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.sql.DataSource;
import org.apache.jackrabbit.oak.cache.CacheStats;
import org.apache.jackrabbit.oak.cache.CacheValue;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.Document;
import org.apache.jackrabbit.oak.plugins.document.DocumentMK;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.StableRevisionComparator;
import org.apache.jackrabbit.oak.plugins.document.UpdateOp;
import org.apache.jackrabbit.oak.plugins.document.UpdateUtils;
import org.apache.jackrabbit.oak.plugins.document.cache.CacheInvalidationStats;
import org.apache.jackrabbit.oak.plugins.document.rdb.RDBConnectionHandler;
import org.apache.jackrabbit.oak.plugins.document.rdb.RDBDocumentSerializer;
import org.apache.jackrabbit.oak.plugins.document.rdb.RDBJDBCTools;
import org.apache.jackrabbit.oak.plugins.document.rdb.RDBOptions;
import org.apache.jackrabbit.oak.plugins.document.rdb.RDBRow;
import org.apache.jackrabbit.oak.plugins.document.util.StringValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDBDocumentStore
implements DocumentStore {
    private String droppedTables = "";
    private static final String MODIFIED = "_modified";
    private static final String MODCOUNT = "_modCount";
    private static final String COLLISIONSMODCOUNT = "_collisionsModCount";
    private static final String ID = "_id";
    private static final Logger LOG = LoggerFactory.getLogger(RDBDocumentStore.class);
    private final Comparator<Revision> comparator = StableRevisionComparator.REVERSE;
    private Exception callStack;
    private RDBConnectionHandler ch;
    private Set<String> tablesToBeDropped = new HashSet<String>();
    private String tnNodes;
    private String tnClusterNodes;
    private String tnSettings;
    private static final int CHAR2OCTETRATIO = 3;
    private int dataLimitInOctets = 16384;
    private static final int RETRIES = 10;
    protected static final boolean USECMODCOUNT = true;
    private DB db;
    private Map<String, String> metadata;
    private static final Set<String> INDEXEDPROPERTIES = new HashSet<String>(Arrays.asList("_modified", "_bin", "_deletedOnce"));
    private static final Set<String> COLUMNPROPERTIES = new HashSet<String>(Arrays.asList("_id", "_bin", "_deletedOnce", "_collisionsModCount", "_modified", "_modCount"));
    private final RDBDocumentSerializer SR = new RDBDocumentSerializer(this, COLUMNPROPERTIES);
    private static final boolean NOGZIP = Boolean.getBoolean("org.apache.jackrabbit.oak.plugins.document.rdb.RDBDocumentStore.NOGZIP");
    private static final int CHUNKSIZE = Integer.getInteger("org.apache.jackrabbit.oak.plugins.document.rdb.RDBDocumentStore.CHUNKSIZE", 64);
    private Cache<CacheValue, NodeDocument> nodesCache;
    private CacheStats cacheStats;
    private final Striped<Lock> locks = Striped.lock((int)64);

    public RDBDocumentStore(DataSource ds, DocumentMK.Builder builder, RDBOptions options) {
        try {
            this.initialize(ds, builder, options);
        }
        catch (Exception ex) {
            throw new DocumentStoreException("initializing RDB document store", ex);
        }
    }

    public RDBDocumentStore(DataSource ds, DocumentMK.Builder builder) {
        this(ds, builder, new RDBOptions());
    }

    @Override
    public <T extends Document> T find(Collection<T> collection, String id) {
        return this.find(collection, id, Integer.MAX_VALUE);
    }

    @Override
    public <T extends Document> T find(Collection<T> collection, String id, int maxCacheAge) {
        return this.readDocumentCached(collection, id, maxCacheAge);
    }

    @Override
    @Nonnull
    public <T extends Document> List<T> query(Collection<T> collection, String fromKey, String toKey, int limit) {
        return this.query(collection, fromKey, toKey, null, 0L, limit);
    }

    @Override
    @Nonnull
    public <T extends Document> List<T> query(Collection<T> collection, String fromKey, String toKey, String indexedProperty, long startValue, int limit) {
        return this.internalQuery(collection, fromKey, toKey, indexedProperty, startValue, limit);
    }

    @Override
    public <T extends Document> void remove(Collection<T> collection, String id) {
        this.delete(collection, id);
        this.invalidateCache(collection, id, true);
    }

    @Override
    public <T extends Document> void remove(Collection<T> collection, List<String> ids) {
        for (String id : ids) {
            this.invalidateCache(collection, id, true);
        }
        this.delete(collection, ids);
    }

    @Override
    public <T extends Document> boolean create(Collection<T> collection, List<UpdateOp> updateOps) {
        return this.internalCreate(collection, updateOps);
    }

    @Override
    public <T extends Document> void update(Collection<T> collection, List<String> keys, UpdateOp updateOp) {
        this.internalUpdate(collection, keys, updateOp);
    }

    @Override
    public <T extends Document> T createOrUpdate(Collection<T> collection, UpdateOp update) {
        return this.internalCreateOrUpdate(collection, update, true, false);
    }

    @Override
    public <T extends Document> T findAndUpdate(Collection<T> collection, UpdateOp update) {
        return this.internalCreateOrUpdate(collection, update, false, true);
    }

    @Override
    public CacheInvalidationStats invalidateCache() {
        for (NodeDocument nd : this.nodesCache.asMap().values()) {
            nd.markUpToDate(0L);
        }
        return null;
    }

    @Override
    public <T extends Document> void invalidateCache(Collection<T> collection, String id) {
        this.invalidateCache(collection, id, false);
    }

    private <T extends Document> void invalidateCache(Collection<T> collection, String id, boolean remove) {
        if (collection == Collection.NODES) {
            this.invalidateNodesCache(id, remove);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invalidateNodesCache(String id, boolean remove) {
        StringValue key = new StringValue(id);
        Lock lock = this.getAndLock(id);
        try {
            if (remove) {
                this.nodesCache.invalidate((Object)key);
            } else {
                NodeDocument entry = (NodeDocument)this.nodesCache.getIfPresent((Object)key);
                if (entry != null) {
                    entry.markUpToDate(0L);
                }
            }
        }
        finally {
            lock.unlock();
        }
    }

    public String getDroppedTables() {
        return this.droppedTables;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void dispose() {
        if (!this.tablesToBeDropped.isEmpty()) {
            String dropped = "";
            LOG.debug("attempting to drop: " + this.tablesToBeDropped);
            for (String tname : this.tablesToBeDropped) {
                Connection con = null;
                try {
                    con = this.ch.getRWConnection();
                    Statement stmt = null;
                    try {
                        stmt = con.createStatement();
                        stmt.execute("drop table " + tname);
                        stmt.close();
                        con.commit();
                        dropped = dropped + tname + " ";
                    }
                    catch (SQLException ex) {
                        LOG.debug("attempting to drop: " + tname, (Throwable)ex);
                    }
                    finally {
                        this.ch.closeStatement(stmt);
                    }
                }
                catch (SQLException ex) {
                    LOG.debug("attempting to drop: " + tname, (Throwable)ex);
                }
                finally {
                    this.ch.closeConnection(con);
                }
            }
            this.droppedTables = dropped.trim();
        }
        this.ch = null;
    }

    @Override
    public <T extends Document> T getIfCached(Collection<T> collection, String id) {
        if (collection != Collection.NODES) {
            return null;
        }
        NodeDocument doc = (NodeDocument)this.nodesCache.getIfPresent((Object)new StringValue(id));
        return RDBDocumentStore.castAsT(doc);
    }

    @Override
    public CacheStats getCacheStats() {
        return this.cacheStats;
    }

    @Override
    public Map<String, String> getMetadata() {
        return this.metadata;
    }

    private static void versionCheck(DatabaseMetaData md, int xmaj, int xmin, String description) throws SQLException {
        int maj = md.getDatabaseMajorVersion();
        int min = md.getDatabaseMinorVersion();
        if (maj < xmaj || maj == xmaj && min < xmin) {
            LOG.info("Unsupported " + description + " version: " + maj + "." + min + ", expected at least " + xmaj + "." + xmin);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void initialize(DataSource ds, DocumentMK.Builder builder, RDBOptions options) throws Exception {
        this.tnNodes = RDBJDBCTools.createTableName(options.getTablePrefix(), "NODES");
        this.tnClusterNodes = RDBJDBCTools.createTableName(options.getTablePrefix(), "CLUSTERNODES");
        this.tnSettings = RDBJDBCTools.createTableName(options.getTablePrefix(), "SETTINGS");
        this.ch = new RDBConnectionHandler(ds);
        this.callStack = LOG.isDebugEnabled() ? new Exception("call stack of RDBDocumentStore creation") : null;
        this.nodesCache = builder.buildDocumentCache(this);
        this.cacheStats = new CacheStats(this.nodesCache, "Document-Documents", builder.getWeigher(), builder.getDocumentCacheSize());
        Connection con = this.ch.getRWConnection();
        DatabaseMetaData md = con.getMetaData();
        String dbDesc = md.getDatabaseProductName() + " " + md.getDatabaseProductVersion();
        String driverDesc = md.getDriverName() + " " + md.getDriverVersion();
        String dbUrl = md.getURL();
        this.db = DB.getValue(md.getDatabaseProductName());
        this.metadata = ImmutableMap.builder().put((Object)"type", (Object)"rdb").put((Object)"db", (Object)md.getDatabaseProductName()).put((Object)"version", (Object)md.getDatabaseProductVersion()).build();
        this.db.checkVersion(md);
        if (!"".equals(this.db.getInitializationStatement())) {
            Statement stmt = null;
            try {
                stmt = con.createStatement();
                stmt.execute(this.db.getInitializationStatement());
                stmt.close();
                con.commit();
            }
            finally {
                this.ch.closeStatement(stmt);
            }
        }
        ArrayList<String> tablesCreated = new ArrayList<String>();
        ArrayList<String> tablesPresent = new ArrayList<String>();
        try {
            this.createTableFor(con, Collection.CLUSTER_NODES, tablesCreated, tablesPresent);
            this.createTableFor(con, Collection.NODES, tablesCreated, tablesPresent);
            this.createTableFor(con, Collection.SETTINGS, tablesCreated, tablesPresent);
        }
        finally {
            con.commit();
            con.close();
        }
        if (options.isDropTablesOnClose()) {
            this.tablesToBeDropped.addAll(tablesCreated);
        }
        LOG.info("RDBDocumentStore instantiated for database " + dbDesc + ", using driver: " + driverDesc + ", connecting to: " + dbUrl);
        if (!tablesPresent.isEmpty()) {
            LOG.info("Tables present upon startup: " + tablesPresent);
        }
        if (!tablesCreated.isEmpty()) {
            LOG.info("Tables created upon startup: " + tablesCreated + (options.isDropTablesOnClose() ? " (will be dropped on exit)" : ""));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createTableFor(Connection con, Collection<? extends Document> col, List<String> tablesCreated, List<String> tablesPresent) throws SQLException {
        String dbname = this.db.toString();
        if (con.getMetaData().getURL() != null) {
            dbname = dbname + " (" + con.getMetaData().getURL() + ")";
        }
        String tableName = this.getTable(col);
        PreparedStatement checkStatement = null;
        ResultSet checkResultSet = null;
        Statement creatStatement = null;
        try {
            checkStatement = con.prepareStatement("select DATA from " + tableName + " where ID = ?");
            checkStatement.setString(1, "0:/");
            checkResultSet = checkStatement.executeQuery();
            if (col.equals(Collection.NODES)) {
                ResultSetMetaData met = checkResultSet.getMetaData();
                this.dataLimitInOctets = met.getPrecision(1);
            }
            tablesPresent.add(tableName);
            this.ch.closeResultSet(checkResultSet);
        }
        catch (SQLException ex) {
            try {
                con.rollback();
                try {
                    creatStatement = con.createStatement();
                    creatStatement.execute(this.db.getTableCreationStatement(tableName));
                    creatStatement.close();
                    con.commit();
                    tablesCreated.add(tableName);
                    if (col.equals(Collection.NODES)) {
                        PreparedStatement pstmt = con.prepareStatement("select DATA from " + tableName + " where ID = ?");
                        pstmt.setString(1, "0:/");
                        ResultSet rs = pstmt.executeQuery();
                        ResultSetMetaData met = rs.getMetaData();
                        this.dataLimitInOctets = met.getPrecision(1);
                    }
                }
                catch (SQLException ex2) {
                    LOG.error("Failed to create table " + tableName + " in " + dbname, (Throwable)ex2);
                    throw ex2;
                }
                this.ch.closeResultSet(checkResultSet);
            }
            catch (Throwable throwable) {
                this.ch.closeResultSet(checkResultSet);
                this.ch.closeStatement(checkStatement);
                this.ch.closeStatement(creatStatement);
                throw throwable;
            }
            this.ch.closeStatement(checkStatement);
            this.ch.closeStatement(creatStatement);
        }
        this.ch.closeStatement(checkStatement);
        this.ch.closeStatement(creatStatement);
    }

    protected void finalize() {
        if (this.ch != null && this.callStack != null) {
            LOG.debug("finalizing RDBDocumentStore that was not disposed", (Throwable)this.callStack);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T extends Document> T readDocumentCached(final Collection<T> collection, final String id, int maxCacheAge) {
        long lastCheckTime;
        if (collection != Collection.NODES) {
            return this.readDocumentUncached(collection, id, null);
        }
        StringValue cacheKey = new StringValue(id);
        NodeDocument doc = null;
        if (maxCacheAge > 0 && (doc = (NodeDocument)this.nodesCache.getIfPresent((Object)cacheKey)) != null && (lastCheckTime = doc.getLastCheckTime()) != 0L) {
            if (maxCacheAge == Integer.MAX_VALUE) return RDBDocumentStore.castAsT(RDBDocumentStore.unwrap(doc));
            if (System.currentTimeMillis() - lastCheckTime < (long)maxCacheAge) {
                return RDBDocumentStore.castAsT(RDBDocumentStore.unwrap(doc));
            }
        }
        try {
            Lock lock = this.getAndLock(id);
            try {
                NodeDocument cachedDoc;
                long lastCheckTime2;
                if (maxCacheAge == 0) {
                    this.invalidateNodesCache(id, true);
                    doc = null;
                }
                if ((lastCheckTime2 = (doc = (NodeDocument)this.nodesCache.get((Object)cacheKey, (Callable)new Callable<NodeDocument>(cachedDoc = doc){
                    final /* synthetic */ NodeDocument val$cachedDoc;
                    {
                        this.val$cachedDoc = nodeDocument;
                    }

                    @Override
                    public NodeDocument call() throws Exception {
                        NodeDocument doc = (NodeDocument)RDBDocumentStore.this.readDocumentUncached(collection, id, this.val$cachedDoc);
                        if (doc != null) {
                            doc.seal();
                        }
                        return RDBDocumentStore.wrap(doc);
                    }
                })).getLastCheckTime()) != 0L) {
                    if (maxCacheAge == 0) return RDBDocumentStore.castAsT(RDBDocumentStore.unwrap(doc));
                    if (maxCacheAge == Integer.MAX_VALUE) {
                        return RDBDocumentStore.castAsT(RDBDocumentStore.unwrap(doc));
                    }
                }
                if (lastCheckTime2 != 0L && System.currentTimeMillis() - lastCheckTime2 < (long)maxCacheAge) {
                    return RDBDocumentStore.castAsT(RDBDocumentStore.unwrap(doc));
                }
                NodeDocument ndoc = (NodeDocument)this.readDocumentUncached(collection, id, cachedDoc);
                if (ndoc != null) {
                    ndoc.seal();
                }
                doc = RDBDocumentStore.wrap(ndoc);
                this.nodesCache.put((Object)cacheKey, (Object)doc);
                return RDBDocumentStore.castAsT(RDBDocumentStore.unwrap(doc));
            }
            finally {
                lock.unlock();
            }
        }
        catch (ExecutionException e) {
            throw new IllegalStateException("Failed to load document with " + id, e);
        }
    }

    @CheckForNull
    private <T extends Document> boolean internalCreate(Collection<T> collection, List<UpdateOp> updates) {
        try {
            for (List chunks : Lists.partition(updates, (int)CHUNKSIZE)) {
                ArrayList<T> docs = new ArrayList<T>();
                for (UpdateOp update : chunks) {
                    T doc = collection.newDocument(this);
                    update.increment(MODCOUNT, 1L);
                    if (this.hasChangesToCollisions(update)) {
                        update.increment(COLLISIONSMODCOUNT, 1L);
                    }
                    UpdateUtils.applyChanges(doc, update, this.comparator);
                    if (!update.getId().equals(((Document)doc).getId())) {
                        throw new DocumentStoreException("ID mismatch - UpdateOp: " + update.getId() + ", ID property: " + ((Document)doc).getId());
                    }
                    docs.add(doc);
                }
                this.insertDocuments(collection, docs);
                for (Document doc : docs) {
                    this.addToCache(collection, doc);
                }
            }
            return true;
        }
        catch (DocumentStoreException ex) {
            return false;
        }
    }

    @CheckForNull
    private <T extends Document> T internalCreateOrUpdate(Collection<T> collection, UpdateOp update, boolean allowCreate, boolean checkConditions) {
        T oldDoc = this.readDocumentCached(collection, update.getId(), Integer.MAX_VALUE);
        if (oldDoc == null) {
            if (!allowCreate) {
                return null;
            }
            if (!update.isNew()) {
                throw new DocumentStoreException("Document does not exist: " + update.getId());
            }
            T doc = collection.newDocument(this);
            if (checkConditions && !UpdateUtils.checkConditions(doc, update)) {
                return null;
            }
            update.increment(MODCOUNT, 1L);
            if (this.hasChangesToCollisions(update)) {
                update.increment(COLLISIONSMODCOUNT, 1L);
            }
            UpdateUtils.applyChanges(doc, update, this.comparator);
            try {
                this.insertDocuments(collection, Collections.singletonList(doc));
                this.addToCache(collection, doc);
                return oldDoc;
            }
            catch (DocumentStoreException ex) {
                oldDoc = this.readDocumentUncached(collection, update.getId(), null);
                if (oldDoc == null) {
                    LOG.error("insert failed, but document " + update.getId() + " is not present, aborting", (Throwable)ex);
                    throw ex;
                }
                return this.internalUpdate(collection, update, oldDoc, checkConditions, 10);
            }
        }
        T result = this.internalUpdate(collection, update, oldDoc, checkConditions, 10);
        if (allowCreate && result == null) {
            LOG.error("update of " + update.getId() + " failed, race condition?");
            throw new DocumentStoreException("update of " + update.getId() + " failed, race condition?");
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @CheckForNull
    private <T extends Document> T internalUpdate(Collection<T> collection, UpdateOp update, T oldDoc, boolean checkConditions, int maxRetries) {
        T doc = this.applyChanges(collection, oldDoc, update, checkConditions);
        if (doc == null) {
            return null;
        }
        Lock l = this.getAndLock(update.getId());
        try {
            boolean success = false;
            int retries = maxRetries;
            while (!success && retries > 0) {
                long lastmodcount = RDBDocumentStore.modcountOf(oldDoc);
                success = this.updateDocument(collection, doc, update, lastmodcount);
                if (!success) {
                    long newmodcount;
                    --retries;
                    oldDoc = this.readDocumentCached(collection, update.getId(), Integer.MAX_VALUE);
                    if (oldDoc != null && lastmodcount == (newmodcount = RDBDocumentStore.modcountOf(oldDoc))) {
                        oldDoc = this.readDocumentUncached(collection, update.getId(), null);
                    }
                    if (oldDoc == null) {
                        LOG.debug("failed to apply update because document is gone in the meantime: " + update.getId(), (Throwable)new Exception("call stack"));
                        T t = null;
                        return t;
                    }
                    doc = this.applyChanges(collection, oldDoc, update, checkConditions);
                    if (doc != null) continue;
                    T t = null;
                    return t;
                }
                if (collection != Collection.NODES) continue;
                this.applyToCache((NodeDocument)oldDoc, (NodeDocument)doc);
            }
            if (!success) {
                throw new DocumentStoreException("failed update of " + ((Document)doc).getId() + " (race?) after " + maxRetries + " retries");
            }
            T t = oldDoc;
            return t;
        }
        finally {
            l.unlock();
        }
    }

    @CheckForNull
    private <T extends Document> T applyChanges(Collection<T> collection, T oldDoc, UpdateOp update, boolean checkConditions) {
        T doc = collection.newDocument(this);
        oldDoc.deepCopy((Document)doc);
        if (checkConditions && !UpdateUtils.checkConditions(doc, update)) {
            return null;
        }
        if (this.hasChangesToCollisions(update)) {
            update.increment(COLLISIONSMODCOUNT, 1L);
        }
        update.increment(MODCOUNT, 1L);
        UpdateUtils.applyChanges(doc, update, this.comparator);
        ((Document)doc).seal();
        return doc;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @CheckForNull
    private <T extends Document> void internalUpdate(Collection<T> collection, List<String> ids, UpdateOp update) {
        if (RDBDocumentStore.isAppendableUpdate(update) && !RDBDocumentStore.requiresPreviousState(update)) {
            long modified = RDBDocumentStore.getModifiedFromUpdate(update);
            String appendData = this.SR.asString(update);
            for (List chunkedIds : Lists.partition(ids, (int)CHUNKSIZE)) {
                Map<String, Object> cachedDocs = Collections.emptyMap();
                if (collection == Collection.NODES) {
                    cachedDocs = new HashMap();
                    for (String key : chunkedIds) {
                        cachedDocs.put(key, this.nodesCache.getIfPresent((Object)new StringValue(key)));
                    }
                }
                Connection connection = null;
                String tableName = this.getTable(collection);
                boolean success = false;
                try {
                    connection = this.ch.getRWConnection();
                    success = this.dbBatchedAppendingUpdate(connection, tableName, chunkedIds, modified, appendData);
                    connection.commit();
                }
                catch (SQLException ex) {
                    success = false;
                    this.ch.rollbackConnection(connection);
                }
                finally {
                    this.ch.closeConnection(connection);
                }
                if (success) {
                    for (Map.Entry entry : cachedDocs.entrySet()) {
                        T oldDoc = RDBDocumentStore.castAsT((NodeDocument)entry.getValue());
                        if (oldDoc == null) {
                            this.nodesCache.invalidate((Object)new StringValue((String)entry.getKey()));
                            continue;
                        }
                        T newDoc = this.applyChanges(collection, oldDoc, update, true);
                        if (newDoc == null) continue;
                        this.applyToCache((NodeDocument)oldDoc, (NodeDocument)newDoc);
                    }
                    continue;
                }
                for (String string : chunkedIds) {
                    UpdateOp up = update.copy();
                    up = up.shallowCopy(string);
                    this.internalCreateOrUpdate(collection, up, false, true);
                }
            }
        } else {
            for (String id : ids) {
                UpdateOp up = update.copy();
                up = up.shallowCopy(id);
                this.internalCreateOrUpdate(collection, up, false, true);
            }
        }
    }

    private <T extends Document> List<T> internalQuery(Collection<T> collection, String fromKey, String toKey, String indexedProperty, long startValue, int limit) {
        Connection connection = null;
        String tableName = this.getTable(collection);
        ArrayList<T> result = new ArrayList<T>();
        if (indexedProperty != null && !INDEXEDPROPERTIES.contains(indexedProperty)) {
            String message = "indexed property " + indexedProperty + " not supported, query was '>= '" + startValue + "'; supported properties are " + INDEXEDPROPERTIES;
            LOG.info(message);
            throw new DocumentStoreException(message);
        }
        try {
            long now = System.currentTimeMillis();
            connection = this.ch.getROConnection();
            List<RDBRow> dbresult = this.dbQuery(connection, tableName, fromKey, toKey, indexedProperty, startValue, limit);
            connection.commit();
            for (RDBRow r : dbresult) {
                T doc = this.runThroughCache(collection, r, now);
                result.add(doc);
            }
        }
        catch (Exception ex) {
            LOG.error("SQL exception on query", (Throwable)ex);
            throw new DocumentStoreException(ex);
        }
        finally {
            this.ch.closeConnection(connection);
        }
        return result;
    }

    private <T extends Document> String getTable(Collection<T> collection) {
        if (collection == Collection.CLUSTER_NODES) {
            return this.tnClusterNodes;
        }
        if (collection == Collection.NODES) {
            return this.tnNodes;
        }
        if (collection == Collection.SETTINGS) {
            return this.tnSettings;
        }
        throw new IllegalArgumentException("Unknown collection: " + collection.toString());
    }

    @CheckForNull
    private <T extends Document> T readDocumentUncached(Collection<T> collection, String id, NodeDocument cachedDoc) {
        T t;
        RDBRow row;
        Connection connection;
        block8: {
            long lastmodcount;
            block7: {
                connection = null;
                String tableName = this.getTable(collection);
                lastmodcount = -1L;
                if (cachedDoc != null) {
                    lastmodcount = RDBDocumentStore.modcountOf(cachedDoc);
                }
                connection = this.ch.getROConnection();
                row = this.dbRead(connection, tableName, id, lastmodcount);
                connection.commit();
                if (row != null) break block7;
                T t2 = null;
                this.ch.closeConnection(connection);
                return t2;
            }
            if (lastmodcount != row.getModcount()) break block8;
            cachedDoc.markUpToDate(System.currentTimeMillis());
            T t3 = RDBDocumentStore.castAsT(cachedDoc);
            this.ch.closeConnection(connection);
            return t3;
        }
        try {
            t = this.SR.fromRow(collection, row);
            this.ch.closeConnection(connection);
        }
        catch (Exception ex) {
            try {
                throw new DocumentStoreException(ex);
            }
            catch (Throwable throwable) {
                this.ch.closeConnection(connection);
                throw throwable;
            }
        }
        return t;
    }

    private <T extends Document> void delete(Collection<T> collection, String id) {
        Connection connection = null;
        String tableName = this.getTable(collection);
        try {
            connection = this.ch.getRWConnection();
            this.dbDelete(connection, tableName, Collections.singletonList(id));
            connection.commit();
        }
        catch (Exception ex) {
            throw new DocumentStoreException(ex);
        }
        finally {
            this.ch.closeConnection(connection);
        }
    }

    private <T extends Document> void delete(Collection<T> collection, List<String> ids) {
        for (List sublist : Lists.partition(ids, (int)64)) {
            Connection connection = null;
            String tableName = this.getTable(collection);
            try {
                connection = this.ch.getRWConnection();
                this.dbDelete(connection, tableName, sublist);
                connection.commit();
            }
            catch (Exception ex) {
                throw new DocumentStoreException(ex);
            }
            finally {
                this.ch.closeConnection(connection);
            }
        }
    }

    private <T extends Document> boolean updateDocument(@Nonnull Collection<T> collection, @Nonnull T document, @Nonnull UpdateOp update, Long oldmodcount) {
        Connection connection = null;
        String tableName = this.getTable(collection);
        try {
            String appendData;
            connection = this.ch.getRWConnection();
            Long modified = (Long)document.get(MODIFIED);
            Number flagB = (Number)document.get("_bin");
            Boolean hasBinary = flagB != null && (long)flagB.intValue() == 1L;
            Boolean flagD = (Boolean)document.get("_deletedOnce");
            Boolean deletedOnce = flagD != null && flagD != false;
            Long modcount = (Long)document.get(MODCOUNT);
            Long cmodcount = (Long)document.get(COLLISIONSMODCOUNT);
            boolean success = false;
            if (RDBDocumentStore.isAppendableUpdate(update) && modcount % 16L != 0L && (appendData = this.SR.asString(update)).length() < this.dataLimitInOctets / 3) {
                try {
                    success = this.dbAppendingUpdate(connection, tableName, document.getId(), modified, hasBinary, deletedOnce, modcount, cmodcount, oldmodcount, appendData);
                    connection.commit();
                }
                catch (SQLException ex) {
                    RDBDocumentStore.continueIfStringOverflow(ex);
                    this.ch.rollbackConnection(connection);
                    success = false;
                }
            }
            if (!success) {
                String data = this.SR.asString(document);
                success = this.dbUpdate(connection, tableName, document.getId(), modified, hasBinary, deletedOnce, modcount, cmodcount, oldmodcount, data);
                connection.commit();
            }
            boolean bl = success;
            return bl;
        }
        catch (SQLException ex) {
            this.ch.rollbackConnection(connection);
            throw new DocumentStoreException(ex);
        }
        finally {
            this.ch.closeConnection(connection);
        }
    }

    private static void continueIfStringOverflow(SQLException ex) throws SQLException {
        String state = ex.getSQLState();
        if (!("22001".equals(state) || "72000".equals(state) && 1489 == ex.getErrorCode())) {
            throw ex;
        }
    }

    private static boolean isAppendableUpdate(UpdateOp update) {
        return true;
    }

    private static boolean requiresPreviousState(UpdateOp update) {
        for (Map.Entry<UpdateOp.Key, UpdateOp.Operation> change : update.getChanges().entrySet()) {
            UpdateOp.Operation op = change.getValue();
            if (op.type != UpdateOp.Operation.Type.CONTAINS_MAP_ENTRY) continue;
            return true;
        }
        return false;
    }

    private static long getModifiedFromUpdate(UpdateOp update) {
        for (Map.Entry<UpdateOp.Key, UpdateOp.Operation> change : update.getChanges().entrySet()) {
            UpdateOp.Operation op = change.getValue();
            if (op.type != UpdateOp.Operation.Type.MAX && op.type != UpdateOp.Operation.Type.SET || !MODIFIED.equals(change.getKey().getName())) continue;
            return Long.parseLong(op.value.toString());
        }
        return 0L;
    }

    private <T extends Document> void insertDocuments(Collection<T> collection, List<T> documents) {
        Connection connection = null;
        String tableName = this.getTable(collection);
        ArrayList<String> ids = new ArrayList<String>();
        try {
            connection = this.ch.getRWConnection();
            for (Document document : documents) {
                String data = this.SR.asString(document);
                Long modified = (Long)document.get(MODIFIED);
                Number flagB = (Number)document.get("_bin");
                Boolean hasBinary = flagB != null && (long)flagB.intValue() == 1L;
                Boolean flagD = (Boolean)document.get("_deletedOnce");
                Boolean deletedOnce = flagD != null && flagD != false;
                Long modcount = (Long)document.get(MODCOUNT);
                Long cmodcount = (Long)document.get(COLLISIONSMODCOUNT);
                String id = document.getId();
                ids.add(id);
                this.dbInsert(connection, tableName, id, modified, hasBinary, deletedOnce, modcount, cmodcount, data);
            }
            connection.commit();
        }
        catch (SQLException ex) {
            LOG.debug("insert of " + ids + " failed", (Throwable)ex);
            this.ch.rollbackConnection(connection);
            throw new DocumentStoreException(ex);
        }
        finally {
            this.ch.closeConnection(connection);
        }
    }

    private static byte[] asBytes(String data) {
        byte[] bytes;
        try {
            bytes = data.getBytes("UTF-8");
        }
        catch (UnsupportedEncodingException ex) {
            LOG.error("UTF-8 not supported??", (Throwable)ex);
            throw new DocumentStoreException(ex);
        }
        if (NOGZIP) {
            return bytes;
        }
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream(data.length());
            GZIPOutputStream gos = new GZIPOutputStream(bos){
                {
                    this.def.setLevel(1);
                }
            };
            gos.write(bytes);
            gos.close();
            return bos.toByteArray();
        }
        catch (IOException ex) {
            LOG.error("Error while gzipping contents", (Throwable)ex);
            throw new DocumentStoreException(ex);
        }
    }

    private void setIdInStatement(PreparedStatement stmt, int idx, String id) throws SQLException {
        if (this.db.isPrimaryColumnByteEncoded()) {
            try {
                stmt.setBytes(idx, id.getBytes("UTF-8"));
            }
            catch (UnsupportedEncodingException ex) {
                LOG.error("UTF-8 not supported??", (Throwable)ex);
                throw new DocumentStoreException(ex);
            }
        } else {
            stmt.setString(idx, id);
        }
    }

    private String getIdFromRS(ResultSet rs, int idx) throws SQLException {
        String id;
        if (this.db.isPrimaryColumnByteEncoded()) {
            try {
                id = new String(rs.getBytes(idx), "UTF-8");
            }
            catch (UnsupportedEncodingException ex) {
                LOG.error("UTF-8 not supported??", (Throwable)ex);
                throw new DocumentStoreException(ex);
            }
        } else {
            id = rs.getString(idx);
        }
        return id;
    }

    @CheckForNull
    private RDBRow dbRead(Connection connection, String tableName, String id, long lastmodcount) throws SQLException {
        boolean useCaseStatement = lastmodcount != -1L && this.db.allowsCaseInSelect();
        PreparedStatement stmt = useCaseStatement ? connection.prepareStatement("select MODIFIED, MODCOUNT, CMODCOUNT, HASBINARY, DELETEDONCE, DATA, BDATA from " + tableName + " where ID = ?") : connection.prepareStatement("select MODIFIED, MODCOUNT, CMODCOUNT, HASBINARY, DELETEDONCE, case MODCOUNT when ? then null else DATA end as DATA, case MODCOUNT when ? then null else BDATA end as BDATA from " + tableName + " where ID = ?");
        try {
            if (useCaseStatement) {
                this.setIdInStatement(stmt, 1, id);
            } else {
                stmt.setLong(1, lastmodcount);
                stmt.setLong(2, lastmodcount);
                this.setIdInStatement(stmt, 3, id);
            }
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                long modified = rs.getLong(1);
                long modcount = rs.getLong(2);
                long cmodcount = rs.getLong(3);
                long hasBinary = rs.getLong(4);
                long deletedOnce = rs.getLong(5);
                String data = rs.getString(6);
                byte[] bdata = rs.getBytes(7);
                RDBRow rDBRow = new RDBRow(id, hasBinary == 1L, deletedOnce == 1L, modified, modcount, cmodcount, data, bdata);
                return rDBRow;
            }
            RDBRow rDBRow = null;
            return rDBRow;
        }
        catch (SQLException ex) {
            LOG.error("attempting to read " + id + " (id length is " + id.length() + ")", (Throwable)ex);
            if ("22001".equals(ex.getSQLState())) {
                this.ch.rollbackConnection(connection);
                RDBRow rDBRow = null;
                return rDBRow;
            }
            throw ex;
        }
        finally {
            stmt.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<RDBRow> dbQuery(Connection connection, String tableName, String minId, String maxId, String indexedProperty, long startValue, int limit) throws SQLException {
        String t = "select ";
        if (limit != Integer.MAX_VALUE && this.db.getFetchFirstSyntax() == FETCHFIRSTSYNTAX.TOP) {
            t = t + "TOP " + limit + " ";
        }
        t = t + "ID, MODIFIED, MODCOUNT, CMODCOUNT, HASBINARY, DELETEDONCE, DATA, BDATA from " + tableName + " where ID > ? and ID < ?";
        if (indexedProperty != null) {
            if (MODIFIED.equals(indexedProperty)) {
                t = t + " and MODIFIED >= ?";
            } else if ("_bin".equals(indexedProperty)) {
                if (startValue != 1L) {
                    throw new DocumentStoreException("unsupported value for property _bin");
                }
                t = t + " and HASBINARY = 1";
            } else if ("_deletedOnce".equals(indexedProperty)) {
                if (startValue != 1L) {
                    throw new DocumentStoreException("unsupported value for property _deletedOnce");
                }
                t = t + " and DELETEDONCE = 1";
            } else {
                throw new DocumentStoreException("unsupported indexed property: " + indexedProperty);
            }
        }
        t = t + " order by ID";
        if (limit != Integer.MAX_VALUE) {
            switch (this.db.getFetchFirstSyntax()) {
                case LIMIT: {
                    t = t + " LIMIT " + limit;
                    break;
                }
                case FETCHFIRST: {
                    t = t + " FETCH FIRST " + limit + " ROWS ONLY";
                    break;
                }
            }
        }
        PreparedStatement stmt = connection.prepareStatement(t);
        ArrayList<RDBRow> result = new ArrayList<RDBRow>();
        try {
            int si = 1;
            this.setIdInStatement(stmt, si++, minId);
            this.setIdInStatement(stmt, si++, maxId);
            if (MODIFIED.equals(indexedProperty)) {
                stmt.setLong(si++, startValue);
            }
            if (limit != Integer.MAX_VALUE) {
                stmt.setFetchSize(limit);
            }
            ResultSet rs = stmt.executeQuery();
            while (rs.next() && result.size() < limit) {
                String id = this.getIdFromRS(rs, 1);
                if (id.compareTo(minId) < 0 || id.compareTo(maxId) > 0) {
                    throw new DocumentStoreException("unexpected query result: '" + minId + "' < '" + id + "' < '" + maxId + "' - broken DB collation?");
                }
                long modified = rs.getLong(2);
                long modcount = rs.getLong(3);
                long cmodcount = rs.getLong(4);
                long hasBinary = rs.getLong(5);
                long deletedOnce = rs.getLong(6);
                String data = rs.getString(7);
                byte[] bdata = rs.getBytes(8);
                result.add(new RDBRow(id, hasBinary == 1L, deletedOnce == 1L, modified, modcount, cmodcount, data, bdata));
            }
        }
        finally {
            stmt.close();
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean dbUpdate(Connection connection, String tableName, String id, Long modified, Boolean hasBinary, Boolean deletedOnce, Long modcount, Long cmodcount, Long oldmodcount, String data) throws SQLException {
        String t = "update " + tableName + " set MODIFIED = ?, HASBINARY = ?, DELETEDONCE = ?, MODCOUNT = ?, CMODCOUNT = ?, DSIZE = ?, DATA = ?, BDATA = ? where ID = ?";
        if (oldmodcount != null) {
            t = t + " and MODCOUNT = ?";
        }
        PreparedStatement stmt = connection.prepareStatement(t);
        try {
            int result;
            int si = 1;
            stmt.setObject(si++, (Object)modified, -5);
            stmt.setObject(si++, (Object)(hasBinary != false ? 1 : 0), 5);
            stmt.setObject(si++, (Object)(deletedOnce != false ? 1 : 0), 5);
            stmt.setObject(si++, (Object)modcount, -5);
            stmt.setObject(si++, (Object)(cmodcount == null ? Long.valueOf(0L) : cmodcount), -5);
            stmt.setObject(si++, (Object)data.length(), -5);
            if (data.length() < this.dataLimitInOctets / 3) {
                stmt.setString(si++, data);
                stmt.setBinaryStream(si++, (InputStream)null, 0);
            } else {
                stmt.setString(si++, "\"blob\"");
                byte[] bytes = RDBDocumentStore.asBytes(data);
                stmt.setBytes(si++, bytes);
            }
            this.setIdInStatement(stmt, si++, id);
            if (oldmodcount != null) {
                stmt.setObject(si++, (Object)oldmodcount, -5);
            }
            if ((result = stmt.executeUpdate()) != 1) {
                LOG.debug("DB update failed for " + tableName + "/" + id + " with oldmodcount=" + oldmodcount);
            }
            boolean bl = result == 1;
            return bl;
        }
        finally {
            stmt.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean dbAppendingUpdate(Connection connection, String tableName, String id, Long modified, Boolean hasBinary, Boolean deletedOnce, Long modcount, Long cmodcount, Long oldmodcount, String appendData) throws SQLException {
        StringBuilder t = new StringBuilder();
        t.append("update " + tableName + " set MODIFIED = " + this.db.getGreatestQueryString("MODIFIED") + ", HASBINARY = ?, DELETEDONCE = ?, MODCOUNT = ?, CMODCOUNT = ?, DSIZE = DSIZE + ?, ");
        t.append("DATA = " + this.db.getConcatQueryString(this.dataLimitInOctets, appendData.length()) + " ");
        t.append("where ID = ?");
        if (oldmodcount != null) {
            t.append(" and MODCOUNT = ?");
        }
        PreparedStatement stmt = connection.prepareStatement(t.toString());
        try {
            int result;
            int si = 1;
            stmt.setObject(si++, (Object)modified, -5);
            stmt.setObject(si++, (Object)(hasBinary != false ? 1 : 0), 5);
            stmt.setObject(si++, (Object)(deletedOnce != false ? 1 : 0), 5);
            stmt.setObject(si++, (Object)modcount, -5);
            stmt.setObject(si++, (Object)(cmodcount == null ? Long.valueOf(0L) : cmodcount), -5);
            stmt.setObject(si++, (Object)(1 + appendData.length()), -5);
            stmt.setString(si++, "," + appendData);
            this.setIdInStatement(stmt, si++, id);
            if (oldmodcount != null) {
                stmt.setObject(si++, (Object)oldmodcount, -5);
            }
            if ((result = stmt.executeUpdate()) != 1) {
                LOG.debug("DB append update failed for " + tableName + "/" + id + " with oldmodcount=" + oldmodcount);
            }
            boolean bl = result == 1;
            return bl;
        }
        finally {
            stmt.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean dbBatchedAppendingUpdate(Connection connection, String tableName, List<String> ids, Long modified, String appendData) throws SQLException {
        StringBuilder t = new StringBuilder();
        t.append("update " + tableName + " set MODIFIED = " + this.db.getGreatestQueryString("MODIFIED") + ", MODCOUNT = MODCOUNT + 1, DSIZE = DSIZE + ?, ");
        t.append("DATA = " + this.db.getConcatQueryString(this.dataLimitInOctets, appendData.length()) + " ");
        t.append("where ID in (");
        for (int i = 0; i < ids.size(); ++i) {
            if (i != 0) {
                t.append(',');
            }
            t.append('?');
        }
        t.append(")");
        PreparedStatement stmt = connection.prepareStatement(t.toString());
        try {
            int si = 1;
            stmt.setObject(si++, (Object)modified, -5);
            stmt.setObject(si++, (Object)(1 + appendData.length()), -5);
            stmt.setString(si++, "," + appendData);
            for (String id : ids) {
                this.setIdInStatement(stmt, si++, id);
            }
            int result = stmt.executeUpdate();
            if (result != ids.size()) {
                LOG.debug("DB update failed: only " + result + " of " + ids.size() + " updated. Table: " + tableName + ", IDs:" + ids);
            }
            boolean bl = result == ids.size();
            return bl;
        }
        finally {
            stmt.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean dbInsert(Connection connection, String tableName, String id, Long modified, Boolean hasBinary, Boolean deletedOnce, Long modcount, Long cmodcount, String data) throws SQLException {
        PreparedStatement stmt = connection.prepareStatement("insert into " + tableName + "(ID, MODIFIED, HASBINARY, DELETEDONCE, MODCOUNT, CMODCOUNT, DSIZE, DATA, BDATA) values (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        try {
            int si = 1;
            this.setIdInStatement(stmt, si++, id);
            stmt.setObject(si++, (Object)modified, -5);
            stmt.setObject(si++, (Object)(hasBinary != false ? 1 : 0), 5);
            stmt.setObject(si++, (Object)(deletedOnce != false ? 1 : 0), 5);
            stmt.setObject(si++, (Object)modcount, -5);
            stmt.setObject(si++, (Object)(cmodcount == null ? Long.valueOf(0L) : cmodcount), -5);
            stmt.setObject(si++, (Object)data.length(), -5);
            if (data.length() < this.dataLimitInOctets / 3) {
                stmt.setString(si++, data);
                stmt.setBinaryStream(si++, (InputStream)null, 0);
            } else {
                stmt.setString(si++, "\"blob\"");
                byte[] bytes = RDBDocumentStore.asBytes(data);
                stmt.setBytes(si++, bytes);
            }
            int result = stmt.executeUpdate();
            if (result != 1) {
                LOG.debug("DB insert failed for " + tableName + "/" + id);
            }
            boolean bl = result == 1;
            return bl;
        }
        finally {
            stmt.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void dbDelete(Connection connection, String tableName, List<String> ids) throws SQLException {
        PreparedStatement stmt;
        int cnt = ids.size();
        if (cnt == 1) {
            stmt = connection.prepareStatement("delete from " + tableName + " where ID=?");
        } else {
            StringBuilder inClause = new StringBuilder();
            for (int i = 0; i < cnt; ++i) {
                inClause.append('?');
                if (i == cnt - 1) continue;
                inClause.append(',');
            }
            stmt = connection.prepareStatement("delete from " + tableName + " where ID in (" + inClause.toString() + ")");
        }
        try {
            for (int i = 0; i < cnt; ++i) {
                this.setIdInStatement(stmt, i + 1, ids.get(i));
            }
            int result = stmt.executeUpdate();
            if (result != cnt) {
                LOG.debug("DB delete failed for " + tableName + "/" + ids);
            }
        }
        finally {
            stmt.close();
        }
    }

    @Override
    public void setReadWriteMode(String readWriteMode) {
    }

    private static <T extends Document> T castAsT(NodeDocument doc) {
        return (T)doc;
    }

    private Lock getAndLock(String key) {
        Lock l = (Lock)this.locks.get((Object)key);
        l.lock();
        return l;
    }

    @CheckForNull
    private static NodeDocument unwrap(@Nonnull NodeDocument doc) {
        return doc == NodeDocument.NULL ? null : doc;
    }

    @Nonnull
    private static NodeDocument wrap(@CheckForNull NodeDocument doc) {
        return doc == null ? NodeDocument.NULL : doc;
    }

    @Nonnull
    private static String idOf(@Nonnull Document doc) {
        String id = doc.getId();
        if (id == null) {
            throw new IllegalArgumentException("non-null ID expected");
        }
        return id;
    }

    private static long modcountOf(@Nonnull Document doc) {
        Number n = doc.getModCount();
        return n != null ? n.longValue() : -1L;
    }

    @Nonnull
    private NodeDocument addToCache(final @Nonnull NodeDocument doc) {
        if (doc == NodeDocument.NULL) {
            throw new IllegalArgumentException("doc must not be NULL document");
        }
        doc.seal();
        try {
            StringValue key = new StringValue(RDBDocumentStore.idOf(doc));
            while (true) {
                NodeDocument cached;
                if ((cached = (NodeDocument)this.nodesCache.get((Object)key, (Callable)new Callable<NodeDocument>(){

                    @Override
                    public NodeDocument call() {
                        return doc;
                    }
                })) != NodeDocument.NULL) {
                    return cached;
                }
                this.nodesCache.invalidate((Object)key);
            }
        }
        catch (ExecutionException e) {
            throw new IllegalStateException(e);
        }
    }

    @Nonnull
    private void applyToCache(@Nonnull NodeDocument oldDoc, @Nonnull NodeDocument newDoc) {
        NodeDocument cached = this.addToCache(newDoc);
        if (cached == newDoc) {
            return;
        }
        if (oldDoc == null) {
            return;
        }
        StringValue key = new StringValue(RDBDocumentStore.idOf(newDoc));
        if (Objects.equal((Object)cached.getModCount(), (Object)oldDoc.getModCount())) {
            this.nodesCache.put((Object)key, (Object)newDoc);
        } else {
            this.nodesCache.invalidate((Object)key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T extends Document> void addToCache(Collection<T> collection, T doc) {
        if (collection == Collection.NODES) {
            Lock lock = this.getAndLock(RDBDocumentStore.idOf(doc));
            try {
                this.addToCache((NodeDocument)doc);
            }
            finally {
                lock.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T extends Document> T runThroughCache(Collection<T> collection, RDBRow row, long now) {
        if (collection != Collection.NODES) {
            return this.SR.fromRow(collection, row);
        }
        String id = row.getId();
        StringValue cacheKey = new StringValue(id);
        NodeDocument inCache = (NodeDocument)this.nodesCache.getIfPresent((Object)cacheKey);
        Long modCount = row.getModcount();
        if (inCache != null && inCache != NodeDocument.NULL) {
            Number cachedModCount = inCache.getModCount();
            if (cachedModCount == null) {
                throw new IllegalStateException("Missing _modCount");
            }
            if (modCount <= cachedModCount.longValue()) {
                inCache.markUpToDate(now);
                return RDBDocumentStore.castAsT(inCache);
            }
        }
        NodeDocument fresh = (NodeDocument)this.SR.fromRow(collection, row);
        fresh.seal();
        Lock lock = this.getAndLock(id);
        try {
            inCache = (NodeDocument)this.nodesCache.getIfPresent((Object)cacheKey);
            if (inCache != null && inCache != NodeDocument.NULL) {
                Number cachedModCount = inCache.getModCount();
                if (cachedModCount == null) {
                    throw new IllegalStateException("Missing _modCount");
                }
                if (modCount > cachedModCount.longValue()) {
                    this.nodesCache.put((Object)cacheKey, (Object)fresh);
                } else {
                    fresh = inCache;
                }
            } else {
                this.nodesCache.put((Object)cacheKey, (Object)fresh);
            }
        }
        finally {
            lock.unlock();
        }
        return RDBDocumentStore.castAsT(fresh);
    }

    private boolean hasChangesToCollisions(UpdateOp update) {
        for (Map.Entry<UpdateOp.Key, UpdateOp.Operation> e : ((UpdateOp)Preconditions.checkNotNull((Object)update)).getChanges().entrySet()) {
            UpdateOp.Key k = e.getKey();
            UpdateOp.Operation op = e.getValue();
            if (op.type != UpdateOp.Operation.Type.SET_MAP_ENTRY || !"_collisions".equals(k.getName())) continue;
            return true;
        }
        return false;
    }

    protected static enum DB {
        DEFAULT("default"){}
        ,
        H2("H2"){

            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                RDBDocumentStore.versionCheck(md, 1, 4, this.description);
            }
        }
        ,
        POSTGRES("PostgreSQL"){

            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                RDBDocumentStore.versionCheck(md, 9, 3, this.description);
            }

            @Override
            public String getTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varchar(512) not null primary key, MODIFIED bigint, HASBINARY smallint, DELETEDONCE smallint, MODCOUNT bigint, CMODCOUNT bigint, DSIZE bigint, DATA varchar(16384), BDATA bytea)";
            }
        }
        ,
        DB2("DB2"){

            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                RDBDocumentStore.versionCheck(md, 10, 5, this.description);
            }
        }
        ,
        ORACLE("Oracle"){

            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                RDBDocumentStore.versionCheck(md, 12, 1, this.description);
            }

            @Override
            public String getInitializationStatement() {
                return "ALTER SESSION SET NLS_SORT='BINARY'";
            }

            @Override
            public String getTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varchar(512) not null primary key, MODIFIED number, HASBINARY number, DELETEDONCE number, MODCOUNT number, CMODCOUNT number, DSIZE number, DATA varchar(4000), BDATA blob)";
            }
        }
        ,
        MYSQL("MySQL"){

            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                RDBDocumentStore.versionCheck(md, 5, 5, this.description);
            }

            @Override
            public boolean isPrimaryColumnByteEncoded() {
                return true;
            }

            @Override
            public String getTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varbinary(512) not null primary key, MODIFIED bigint, HASBINARY smallint, DELETEDONCE smallint, MODCOUNT bigint, CMODCOUNT bigint, DSIZE bigint, DATA varchar(16000), BDATA longblob)";
            }

            @Override
            public FETCHFIRSTSYNTAX getFetchFirstSyntax() {
                return FETCHFIRSTSYNTAX.LIMIT;
            }

            @Override
            public String getConcatQueryString(int dataOctetLimit, int dataLength) {
                return "CONCAT(DATA, ?)";
            }
        }
        ,
        MSSQL("Microsoft SQL Server"){

            @Override
            public void checkVersion(DatabaseMetaData md) throws SQLException {
                RDBDocumentStore.versionCheck(md, 11, 0, this.description);
            }

            @Override
            public boolean isPrimaryColumnByteEncoded() {
                return true;
            }

            @Override
            public String getTableCreationStatement(String tableName) {
                return "create table " + tableName + " (ID varbinary(512) not null primary key, MODIFIED bigint, HASBINARY smallint, DELETEDONCE smallint, MODCOUNT bigint, CMODCOUNT bigint, DSIZE bigint, DATA nvarchar(4000), BDATA varbinary(max))";
            }

            @Override
            public FETCHFIRSTSYNTAX getFetchFirstSyntax() {
                return FETCHFIRSTSYNTAX.TOP;
            }

            @Override
            public String getConcatQueryString(int dataOctetLimit, int dataLength) {
                return "CASE WHEN LEN(DATA) <= " + (dataOctetLimit - dataLength) + " THEN (DATA + CAST(? AS nvarchar(" + dataOctetLimit + "))) ELSE (DATA + CAST(DATA AS nvarchar(max))) END";
            }

            @Override
            public String getGreatestQueryString(String column) {
                return "(select MAX(mod) from (VALUES (" + column + "), (?)) AS ALLMOD(mod))";
            }
        };

        protected String description;

        public void checkVersion(DatabaseMetaData md) throws SQLException {
            LOG.info("Unknown database type: " + md.getDatabaseProductName());
        }

        public boolean isPrimaryColumnByteEncoded() {
            return false;
        }

        public boolean allowsCaseInSelect() {
            return true;
        }

        public FETCHFIRSTSYNTAX getFetchFirstSyntax() {
            return FETCHFIRSTSYNTAX.FETCHFIRST;
        }

        public String getConcatQueryString(int dataOctetLimit, int dataLength) {
            return "DATA || CAST(? AS varchar(" + dataOctetLimit + "))";
        }

        public String getGreatestQueryString(String column) {
            return "GREATEST(" + column + ", ?)";
        }

        @Nonnull
        public String getInitializationStatement() {
            return "";
        }

        public String getTableCreationStatement(String tableName) {
            return "create table " + tableName + " (ID varchar(512) not null primary key, MODIFIED bigint, HASBINARY smallint, DELETEDONCE smallint, MODCOUNT bigint, CMODCOUNT bigint, DSIZE bigint, DATA varchar(16384), BDATA blob(" + 0x40000000 + "))";
        }

        private DB(String description) {
            this.description = description;
        }

        public String toString() {
            return this.description;
        }

        @Nonnull
        public static DB getValue(String desc) {
            for (DB db : DB.values()) {
                if (db.description.equals(desc)) {
                    return db;
                }
                if (db != DB2 || !desc.startsWith("DB2/")) continue;
                return db;
            }
            LOG.error("DB type " + desc + " unknown, trying default settings");
            DB.DEFAULT.description = desc + " - using default settings";
            return DEFAULT;
        }
    }

    static enum FETCHFIRSTSYNTAX {
        FETCHFIRST,
        LIMIT,
        TOP;

    }
}

