/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.segment;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.UUID;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.segment.RecordId;
import org.apache.jackrabbit.oak.plugins.segment.SegmentId;
import org.apache.jackrabbit.oak.plugins.segment.SegmentTracker;

public class CompactionMap {
    private final int compressInterval;
    private final SegmentTracker tracker;
    private Map<RecordId, RecordId> recent = Maps.newHashMap();
    private long[] msbs = new long[0];
    private long[] lsbs = new long[0];
    private short[] beforeOffsets = new short[0];
    private int[] entryIndex = new int[0];
    private short[] afterOffsets = new short[0];
    private int[] afterSegmentIds = new int[0];
    private long[] amsbs = new long[0];
    private long[] alsbs = new long[0];
    private long prevWeight;
    private CompactionMap prev;

    CompactionMap(int compressInterval, SegmentTracker tracker) {
        this.compressInterval = compressInterval;
        this.tracker = tracker;
    }

    boolean wasCompactedTo(RecordId before, RecordId after) {
        return this.recursiveWasCompactedTo(before, after);
    }

    private boolean recursiveWasCompactedTo(RecordId before, RecordId after) {
        RecordId potentialAfter = CompactionMap.recursiveGet(this, before);
        if (potentialAfter == null) {
            return false;
        }
        if (after.equals(potentialAfter)) {
            return true;
        }
        return this.recursiveWasCompactedTo(potentialAfter, after);
    }

    private static RecordId recursiveGet(CompactionMap map, RecordId before) {
        RecordId after = map.get(before);
        if (after != null) {
            return after;
        }
        if (map.prev != null) {
            return CompactionMap.recursiveGet(map.prev, before);
        }
        return null;
    }

    public boolean wasCompacted(UUID id) {
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        return CompactionMap.wasCompacted(this, msb, lsb);
    }

    private static boolean wasCompacted(CompactionMap map, long msb, long lsb) {
        int find = map.findEntry(msb, lsb);
        if (find != -1) {
            return true;
        }
        if (map.prev != null) {
            return CompactionMap.wasCompacted(map.prev, msb, lsb);
        }
        return false;
    }

    public RecordId get(RecordId before) {
        RecordId after = this.recent.get(before);
        if (after != null) {
            return after;
        }
        if (this.msbs.length == 0) {
            return null;
        }
        SegmentId segmentId = before.getSegmentId();
        long msb = segmentId.getMostSignificantBits();
        long lsb = segmentId.getLeastSignificantBits();
        int offset = before.getOffset();
        int entry = this.findEntry(msb, lsb);
        if (entry != -1) {
            int index = this.entryIndex[entry];
            int limit = this.entryIndex[entry + 1];
            for (int i = index; i < limit; ++i) {
                int o = CompactionMap.decode(this.beforeOffsets[i]);
                if (o == offset) {
                    return new RecordId(this.asSegmentId(i), CompactionMap.decode(this.afterOffsets[i]));
                }
                if (o <= offset) continue;
                return null;
            }
        }
        return null;
    }

    private static int decode(short offset) {
        return (offset & 0xFFFF) << 2;
    }

    private static short encode(int offset) {
        return (short)(offset >> 2);
    }

    private SegmentId asSegmentId(int index) {
        int idx = this.afterSegmentIds[index];
        return new SegmentId(this.tracker, this.amsbs[idx], this.alsbs[idx]);
    }

    private static UUID asUUID(SegmentId id) {
        return new UUID(id.getMostSignificantBits(), id.getLeastSignificantBits());
    }

    void put(RecordId before, RecordId after) {
        if (this.get(before) != null) {
            throw new IllegalArgumentException();
        }
        this.recent.put(before, after);
        if (this.recent.size() >= this.compressInterval) {
            this.compress();
        }
    }

    void compress() {
        if (this.recent.isEmpty()) {
            return;
        }
        TreeSet uuids = Sets.newTreeSet();
        TreeMap mapping = Maps.newTreeMap();
        for (Map.Entry<RecordId, RecordId> entry : this.recent.entrySet()) {
            RecordId before = entry.getKey();
            SegmentId id = before.getSegmentId();
            UUID uuid = new UUID(id.getMostSignificantBits(), id.getLeastSignificantBits());
            uuids.add(uuid);
            Map map = (Map)mapping.get(uuid);
            if (map == null) {
                map = Maps.newTreeMap();
                mapping.put(uuid, map);
            }
            map.put(before.getOffset(), entry.getValue());
        }
        for (int i = 0; i < this.msbs.length; ++i) {
            uuids.add(new UUID(this.msbs[i], this.lsbs[i]));
        }
        long[] newmsbs = new long[uuids.size()];
        long[] newlsbs = new long[uuids.size()];
        int[] newEntryIndex = new int[uuids.size() + 1];
        int newEntries = this.beforeOffsets.length + this.recent.size();
        short[] newBeforeOffsets = new short[newEntries];
        short[] newAfterOffsets = new short[newEntries];
        int[] newAfterSegmentIds = new int[newEntries];
        HashMap newAfterSegments = Maps.newHashMap();
        int newIndex = 0;
        int newEntry = 0;
        int oldEntry = 0;
        for (UUID uUID : uuids) {
            newmsbs[newEntry] = uUID.getMostSignificantBits();
            newlsbs[newEntry] = uUID.getLeastSignificantBits();
            Map newsegment = (Map)mapping.get(uUID);
            if (newsegment == null) {
                newsegment = Maps.newTreeMap();
            }
            if (oldEntry < this.msbs.length && this.msbs[oldEntry] == newmsbs[newEntry] && this.lsbs[oldEntry] == newlsbs[newEntry]) {
                int index = this.entryIndex[oldEntry];
                int limit = this.entryIndex[oldEntry + 1];
                for (int i = index; i < limit; ++i) {
                    newsegment.put(CompactionMap.decode(this.beforeOffsets[i]), new RecordId(this.asSegmentId(i), CompactionMap.decode(this.afterOffsets[i])));
                }
                ++oldEntry;
            }
            newEntryIndex[newEntry++] = newIndex;
            for (Map.Entry entry : newsegment.entrySet()) {
                int key = (Integer)entry.getKey();
                RecordId id = (RecordId)entry.getValue();
                newBeforeOffsets[newIndex] = CompactionMap.encode(key);
                newAfterOffsets[newIndex] = CompactionMap.encode(id.getOffset());
                UUID aUUID = CompactionMap.asUUID(id.getSegmentId());
                int aSIdx = -1;
                if (newAfterSegments.containsKey(aUUID)) {
                    aSIdx = (Integer)newAfterSegments.get(aUUID);
                } else {
                    aSIdx = newAfterSegments.size();
                    newAfterSegments.put(aUUID, aSIdx);
                }
                newAfterSegmentIds[newIndex] = aSIdx;
                ++newIndex;
            }
        }
        newEntryIndex[newEntry] = newIndex;
        this.msbs = newmsbs;
        this.lsbs = newlsbs;
        this.entryIndex = newEntryIndex;
        this.beforeOffsets = newBeforeOffsets;
        this.afterOffsets = newAfterOffsets;
        this.afterSegmentIds = newAfterSegmentIds;
        this.amsbs = new long[newAfterSegments.size()];
        this.alsbs = new long[newAfterSegments.size()];
        for (Map.Entry entry : newAfterSegments.entrySet()) {
            this.amsbs[((Integer)entry.getValue()).intValue()] = ((UUID)entry.getKey()).getMostSignificantBits();
            this.alsbs[((Integer)entry.getValue()).intValue()] = ((UUID)entry.getKey()).getLeastSignificantBits();
        }
        this.recent = Maps.newHashMap();
    }

    private final int findEntry(long msb, long lsb) {
        int lowIndex = 0;
        int highIndex = this.msbs.length - 1;
        float lowValue = -9.223372E18f;
        float highValue = 9.223372E18f;
        float targetValue = msb;
        while (lowIndex <= highIndex) {
            long m;
            int guessIndex = lowIndex;
            float valueRange = highValue - lowValue;
            if (valueRange >= 1.0f) {
                guessIndex += Math.round((float)(highIndex - lowIndex) * (targetValue - lowValue) / valueRange);
            }
            if (msb < (m = this.msbs[guessIndex])) {
                highIndex = guessIndex - 1;
                highValue = m;
                continue;
            }
            if (msb > m) {
                lowIndex = guessIndex + 1;
                lowValue = m;
                continue;
            }
            long l = this.lsbs[guessIndex];
            if (lsb < l) {
                highIndex = guessIndex - 1;
                highValue = m;
                continue;
            }
            if (lsb > l) {
                highIndex = guessIndex + 1;
                highValue = m;
                continue;
            }
            return guessIndex;
        }
        return -1;
    }

    void merge(CompactionMap prev) {
        this.prev = prev;
        this.prevWeight = prev.getEstimatedWeight();
    }

    public String getCompactionStats() {
        StringBuilder sb = new StringBuilder();
        CompactionMap cm = this;
        while (cm != null) {
            sb.append("[");
            sb.append(CompactionMap.getCompactionStats(cm));
            sb.append("], ");
            cm = cm.prev;
        }
        return sb.toString();
    }

    private static String getCompactionStats(CompactionMap cm) {
        StringBuilder sb = new StringBuilder();
        sb.append("Estimated Weight: ");
        sb.append(IOUtils.humanReadableByteCount(CompactionMap.getEstimatedWeight(cm)));
        sb.append(", Records: ");
        sb.append(cm.afterOffsets.length);
        sb.append(", Segments: ");
        sb.append(cm.amsbs.length);
        return sb.toString();
    }

    public long getEstimatedWeight() {
        long total = 0L;
        CompactionMap cm = this;
        while (cm != null) {
            total += CompactionMap.getEstimatedWeight(cm);
            cm = cm.prev;
        }
        return total;
    }

    public long getLastMergeWeight() {
        return this.prevWeight;
    }

    private static long getEstimatedWeight(CompactionMap cm) {
        long total = 168L;
        total += (long)(24 + cm.msbs.length * 8);
        total += (long)(24 + cm.lsbs.length * 8);
        total += (long)(24 + cm.beforeOffsets.length * 2);
        total += (long)(24 + cm.entryIndex.length * 4);
        total += (long)(24 + cm.afterOffsets.length * 2);
        total += (long)(24 + cm.afterSegmentIds.length * 4);
        total += (long)(24 + cm.amsbs.length * 8);
        return total += (long)(24 + cm.alsbs.length * 8);
    }
}

