/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.segment;

import com.google.common.base.Preconditions;
import java.util.Formatter;
import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.plugins.segment.ListRecord;
import org.apache.jackrabbit.oak.plugins.segment.MapEntry;
import org.apache.jackrabbit.oak.plugins.segment.MapRecord;
import org.apache.jackrabbit.oak.plugins.segment.PropertyTemplate;
import org.apache.jackrabbit.oak.plugins.segment.RecordId;
import org.apache.jackrabbit.oak.plugins.segment.RecordIdSet;
import org.apache.jackrabbit.oak.plugins.segment.Segment;
import org.apache.jackrabbit.oak.plugins.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.plugins.segment.SegmentVersion;
import org.apache.jackrabbit.oak.plugins.segment.Template;
import org.apache.jackrabbit.oak.spi.state.ChildNodeEntry;
import org.apache.jackrabbit.oak.spi.state.NodeState;

public class RecordUsageAnalyser {
    private final RecordIdSet seenIds = new RecordIdSet();
    private long mapSize;
    private long listSize;
    private long valueSize;
    private long templateSize;
    private long nodeSize;
    private long mapCount;
    private long listCount;
    private long propertyCount;
    private long smallBlobCount;
    private long mediumBlobCount;
    private long longBlobCount;
    private long externalBlobCount;
    private long smallStringCount;
    private long mediumStringCount;
    private long longStringCount;
    private long templateCount;
    private long nodeCount;

    public long getMapSize() {
        return this.mapSize;
    }

    public long getListSize() {
        return this.listSize;
    }

    public long getValueSize() {
        return this.valueSize;
    }

    public long getTemplateSize() {
        return this.templateSize;
    }

    public long getNodeSize() {
        return this.nodeSize;
    }

    public long getMapCount() {
        return this.mapCount;
    }

    public long getListCount() {
        return this.listCount;
    }

    public long getPropertyCount() {
        return this.propertyCount;
    }

    public long getSmallBlobCount() {
        return this.smallBlobCount;
    }

    public long getMediumBlobCount() {
        return this.mediumBlobCount;
    }

    public long getLongBlobCount() {
        return this.longBlobCount;
    }

    public long getExternalBlobCount() {
        return this.externalBlobCount;
    }

    public long getSmallStringCount() {
        return this.smallStringCount;
    }

    public long getMediumStringCount() {
        return this.mediumStringCount;
    }

    public long getLongStringCount() {
        return this.longStringCount;
    }

    public long getTemplateCount() {
        return this.templateCount;
    }

    public long getNodeCount() {
        return this.nodeCount;
    }

    public void analyseNode(RecordId nodeId) {
        if (this.seenIds.addIfNotPresent(nodeId)) {
            ++this.nodeCount;
            Segment segment = nodeId.getSegment();
            int offset = nodeId.getOffset();
            RecordId templateId = segment.readRecordId(offset);
            this.analyseTemplate(templateId);
            Template template = segment.readTemplate(templateId);
            if (template.getChildName() == "") {
                RecordId childMapId = segment.readRecordId(offset + 3);
                MapRecord childMap = segment.readMap(childMapId);
                this.analyseMap(childMapId, childMap);
                for (ChildNodeEntry childNodeEntry : childMap.getEntries()) {
                    NodeState child = childNodeEntry.getNodeState();
                    if (!(child instanceof SegmentNodeState)) continue;
                    RecordId childId = ((SegmentNodeState)child).getRecordId();
                    this.analyseNode(childId);
                }
            } else if (template.getChildName() != Template.ZERO_CHILD_NODES) {
                RecordId childId = segment.readRecordId(offset + 3);
                this.analyseNode(childId);
            }
            int ids = template.getChildName() == Template.ZERO_CHILD_NODES ? 1 : 2;
            this.nodeSize += (long)(ids * 3);
            PropertyTemplate[] propertyTemplates = template.getPropertyTemplates();
            if (segment.getSegmentVersion().onOrAfter(SegmentVersion.V_11)) {
                if (propertyTemplates.length > 0) {
                    this.nodeSize += 3L;
                    RecordId id = segment.readRecordId(offset + ids * 3);
                    ListRecord listRecord = new ListRecord(id, propertyTemplates.length);
                    for (int i = 0; i < propertyTemplates.length; ++i) {
                        RecordId propertyId = listRecord.getEntry(i);
                        this.analyseProperty(propertyId, propertyTemplates[i]);
                    }
                    this.analyseList(id, propertyTemplates.length);
                }
            } else {
                for (PropertyTemplate propertyTemplate : propertyTemplates) {
                    this.nodeSize += 3L;
                    RecordId propertyId = segment.readRecordId(offset + ids++ * 3);
                    this.analyseProperty(propertyId, propertyTemplate);
                }
            }
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        Formatter formatter = new Formatter(sb);
        formatter.format("%s in maps (%s leaf and branch records)%n", FileUtils.byteCountToDisplaySize((long)this.mapSize), this.mapCount);
        formatter.format("%s in lists (%s list and bucket records)%n", FileUtils.byteCountToDisplaySize((long)this.listSize), this.listCount);
        formatter.format("%s in values (value and block records of %s properties, %s/%s/%s/%s small/medium/long/external blobs, %s/%s/%s small/medium/long strings)%n", FileUtils.byteCountToDisplaySize((long)this.valueSize), this.propertyCount, this.smallBlobCount, this.mediumBlobCount, this.longBlobCount, this.externalBlobCount, this.smallStringCount, this.mediumStringCount, this.longStringCount);
        formatter.format("%s in templates (%s template records)%n", FileUtils.byteCountToDisplaySize((long)this.templateSize), this.templateCount);
        formatter.format("%s in nodes (%s node records)%n", FileUtils.byteCountToDisplaySize((long)this.nodeSize), this.nodeCount);
        return sb.toString();
    }

    private void analyseTemplate(RecordId templateId) {
        if (this.seenIds.addIfNotPresent(templateId)) {
            ++this.templateCount;
            Segment segment = templateId.getSegment();
            int size = 0;
            int offset = templateId.getOffset();
            int head = segment.readInt(offset + size);
            boolean hasPrimaryType = (head & Integer.MIN_VALUE) != 0;
            boolean hasMixinTypes = (head & 0x40000000) != 0;
            boolean zeroChildNodes = (head & 0x20000000) != 0;
            boolean manyChildNodes = (head & 0x10000000) != 0;
            int mixinCount = head >> 18 & 0x3FF;
            int propertyCount = head & 0x3FFFF;
            size += 4;
            if (hasPrimaryType) {
                RecordId primaryId = segment.readRecordId(offset + size);
                this.analyseString(primaryId);
                size += 3;
            }
            if (hasMixinTypes) {
                for (int i = 0; i < mixinCount; ++i) {
                    RecordId mixinId = segment.readRecordId(offset + size);
                    this.analyseString(mixinId);
                    size += 3;
                }
            }
            if (!zeroChildNodes && !manyChildNodes) {
                RecordId childNameId = segment.readRecordId(offset + size);
                this.analyseString(childNameId);
                size += 3;
            }
            if (segment.getSegmentVersion().onOrAfter(SegmentVersion.V_11)) {
                if (propertyCount > 0) {
                    RecordId listId = segment.readRecordId(offset + size);
                    size += 3;
                    ListRecord propertyNames = new ListRecord(listId, propertyCount);
                    for (int i = 0; i < propertyCount; ++i) {
                        RecordId propertyNameId = propertyNames.getEntry(i);
                        ++size;
                        this.analyseString(propertyNameId);
                    }
                    this.analyseList(listId, propertyCount);
                }
            } else {
                for (int i = 0; i < propertyCount; ++i) {
                    RecordId propertyNameId = segment.readRecordId(offset + size);
                    size += 3;
                    ++size;
                    this.analyseString(propertyNameId);
                }
            }
            this.templateSize += (long)size;
        }
    }

    private void analyseMap(RecordId mapId, MapRecord map) {
        if (this.seenIds.addIfNotPresent(mapId)) {
            ++this.mapCount;
            if (map.isDiff()) {
                this.analyseDiff(mapId, map);
            } else if (map.isLeaf()) {
                this.analyseLeaf(map);
            } else {
                this.analyseBranch(map);
            }
        }
    }

    private void analyseDiff(RecordId mapId, MapRecord map) {
        this.mapSize += 4L;
        this.mapSize += 4L;
        this.mapSize += 3L;
        this.mapSize += 3L;
        this.mapSize += 3L;
        RecordId baseId = mapId.getSegment().readRecordId(mapId.getOffset() + 8 + 6);
        this.analyseMap(baseId, new MapRecord(baseId));
    }

    private void analyseLeaf(MapRecord map) {
        this.mapSize += 4L;
        this.mapSize += (long)(map.size() * 4);
        for (MapEntry entry : map.getEntries()) {
            this.mapSize += 6L;
            this.analyseString(entry.getKey());
        }
    }

    private void analyseBranch(MapRecord map) {
        this.mapSize += 4L;
        this.mapSize += 4L;
        for (MapRecord bucket : map.getBuckets()) {
            if (bucket == null) continue;
            this.mapSize += 3L;
            this.analyseMap(bucket.getRecordId(), bucket);
        }
    }

    private void analyseProperty(RecordId propertyId, PropertyTemplate template) {
        if (!this.seenIds.contains(propertyId)) {
            ++this.propertyCount;
            Segment segment = propertyId.getSegment();
            int offset = propertyId.getOffset();
            Type<?> type = template.getType();
            if (type.isArray()) {
                this.seenIds.addIfNotPresent(propertyId);
                int size = segment.readInt(offset);
                this.valueSize += 4L;
                if (size > 0) {
                    RecordId listId = segment.readRecordId(offset + 4);
                    this.valueSize += 3L;
                    for (RecordId valueId : new ListRecord(listId, size).getEntries()) {
                        this.analyseValue(valueId, type.getBaseType());
                    }
                    this.analyseList(listId, size);
                }
            } else {
                this.analyseValue(propertyId, type);
            }
        }
    }

    private void analyseValue(RecordId valueId, Type<?> type) {
        Preconditions.checkArgument((!type.isArray() ? 1 : 0) != 0);
        if (type == Type.BINARY) {
            this.analyseBlob(valueId);
        } else {
            this.analyseString(valueId);
        }
    }

    private void analyseBlob(RecordId blobId) {
        if (this.seenIds.addIfNotPresent(blobId)) {
            int offset;
            Segment segment = blobId.getSegment();
            byte head = segment.readByte(offset = blobId.getOffset());
            if ((head & 0x80) == 0) {
                this.valueSize += (long)(1 + head);
                ++this.smallBlobCount;
            } else if ((head & 0xC0) == 128) {
                int length = (segment.readShort(offset) & 0x3FFF) + 128;
                this.valueSize += (long)(2 + length);
                ++this.mediumBlobCount;
            } else if ((head & 0xE0) == 192) {
                long length = (segment.readLong(offset) & 0x1FFFFFFFFFFFFFFFL) + 16512L;
                int size = (int)((length + 4096L - 1L) / 4096L);
                RecordId listId = segment.readRecordId(offset + 8);
                this.analyseList(listId, size);
                this.valueSize += 11L + length;
                ++this.longBlobCount;
            } else if ((head & 0xF0) == 224) {
                int length = (head & 0xF) << 8 | segment.readByte(offset + 1) & 0xFF;
                this.valueSize += (long)(2 + length);
                ++this.externalBlobCount;
            } else {
                throw new IllegalStateException(String.format("Unexpected value record type: %02x", head & 0xFF));
            }
        }
    }

    private void analyseString(RecordId stringId) {
        if (this.seenIds.addIfNotPresent(stringId)) {
            int offset;
            Segment segment = stringId.getSegment();
            long length = segment.readLength(offset = stringId.getOffset());
            if (length < 128L) {
                this.valueSize += 1L + length;
                ++this.smallStringCount;
            } else if (length < 16512L) {
                this.valueSize += 2L + length;
                ++this.mediumStringCount;
            } else if (length < Integer.MAX_VALUE) {
                int size = (int)((length + 4096L - 1L) / 4096L);
                RecordId listId = segment.readRecordId(offset + 8);
                this.analyseList(listId, size);
                this.valueSize += 11L + length;
                ++this.longStringCount;
            } else {
                throw new IllegalStateException("String is too long: " + length);
            }
        }
    }

    private void analyseList(RecordId listId, int size) {
        if (this.seenIds.addIfNotPresent(listId)) {
            ++this.listCount;
            this.listSize += (long)(RecordUsageAnalyser.noOfListSlots(size) * 3);
        }
    }

    private static int noOfListSlots(int size) {
        if (size <= 255) {
            return size;
        }
        int fullBuckets = size / 255;
        if (size % 255 > 1) {
            return size + RecordUsageAnalyser.noOfListSlots(fullBuckets + 1);
        }
        return size + RecordUsageAnalyser.noOfListSlots(fullBuckets);
    }
}

