/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.document.ClusterNodeInfo;
import org.apache.jackrabbit.oak.plugins.document.ClusterNodeInfoDocument;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.JournalEntry;
import org.apache.jackrabbit.oak.plugins.document.MissingLastRevSeeker;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.UnsavedModifications;
import org.apache.jackrabbit.oak.plugins.document.util.MapFactory;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LastRevRecoveryAgent {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final DocumentNodeStore nodeStore;
    private final MissingLastRevSeeker missingLastRevUtil;

    public LastRevRecoveryAgent(DocumentNodeStore nodeStore, MissingLastRevSeeker seeker) {
        this.nodeStore = nodeStore;
        this.missingLastRevUtil = seeker;
    }

    public LastRevRecoveryAgent(DocumentNodeStore nodeStore) {
        this(nodeStore, new MissingLastRevSeeker(nodeStore.getDocumentStore()));
    }

    public int recover(int clusterId) {
        ClusterNodeInfoDocument nodeInfo = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
        long leaseTime = ClusterNodeInfo.DEFAULT_LEASE_DURATION_MILLIS;
        long asyncDelay = this.nodeStore.getAsyncDelay();
        if (nodeInfo != null && this.isRecoveryNeeded(nodeInfo)) {
            long leaseEnd = nodeInfo.getLeaseEndTime();
            NodeDocument root = this.missingLastRevUtil.getRoot();
            Revision lastRev = root.getLastRev().get(clusterId);
            long startTime = lastRev != null ? lastRev.getTimestamp() : leaseEnd - leaseTime - asyncDelay;
            this.log.info("Recovering candidates modified after: [{}] for clusterId [{}]", (Object)Utils.timestampToString(startTime), (Object)clusterId);
            return this.recoverCandidates(clusterId, startTime);
        }
        this.log.debug("No recovery needed for clusterId {}", (Object)clusterId);
        return 0;
    }

    public int recover(Iterator<NodeDocument> suspects, int clusterId) {
        return this.recover(suspects, clusterId, false);
    }

    public int recover(Iterator<NodeDocument> suspects, int clusterId, boolean dryRun) {
        UnsavedModifications unsaved = new UnsavedModifications();
        UnsavedModifications unsavedParents = new UnsavedModifications();
        ConcurrentMap<String, Revision> knownLastRevOrModification = MapFactory.getInstance().create();
        final DocumentStore docStore = this.nodeStore.getDocumentStore();
        final JournalEntry changes = Collection.JOURNAL.newDocument(docStore);
        long count = 0L;
        while (suspects.hasNext()) {
            NodeDocument doc = suspects.next();
            if (++count % 100000L == 0L) {
                this.log.info("Scanned {} suspects so far...", (Object)count);
            }
            Revision currentLastRev = doc.getLastRev().get(clusterId);
            Revision lastModifiedRev = this.determineLastModification(doc, clusterId);
            Revision lastRevForParents = Utils.max(lastModifiedRev, currentLastRev);
            if (lastRevForParents != null) {
                knownLastRevOrModification.put(doc.getPath(), lastRevForParents);
            }
            if (lastRevForParents == null) continue;
            String path = doc.getPath();
            changes.modified(path);
            while (!PathUtils.denotesRoot(path)) {
                path = PathUtils.getParentPath(path);
                unsavedParents.put(path, lastRevForParents);
            }
        }
        for (String parentPath : unsavedParents.getPaths()) {
            Revision calcLastRev = unsavedParents.get(parentPath);
            Revision knownLastRev = (Revision)knownLastRevOrModification.get(parentPath);
            if (knownLastRev == null) {
                String id = Utils.getIdFromPath(parentPath);
                NodeDocument doc = docStore.find(Collection.NODES, id);
                if (doc != null) {
                    Revision lastRev = doc.getLastRev().get(clusterId);
                    Revision lastMod = this.determineLastModification(doc, clusterId);
                    knownLastRev = Utils.max(lastRev, lastMod);
                } else {
                    this.log.warn("Unable to find document: {}", (Object)id);
                    continue;
                }
            }
            if (knownLastRev != null && calcLastRev.compareRevisionTime(knownLastRev) <= 0) continue;
            unsaved.put(parentPath, calcLastRev);
        }
        final Revision lastRootRev = unsaved.get("/");
        int size = unsaved.getPaths().size();
        String updates = unsaved.toString();
        if (dryRun) {
            this.log.info("Dry run of lastRev recovery identified [{}] documents for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        } else {
            unsaved.persist(this.nodeStore, new UnsavedModifications.Snapshot(){

                @Override
                public void acquiring(Revision mostRecent) {
                    if (lastRootRev == null) {
                        return;
                    }
                    String id = JournalEntry.asId(lastRootRev);
                    JournalEntry existingEntry = docStore.find(Collection.JOURNAL, id);
                    if (existingEntry != null) {
                        return;
                    }
                    docStore.create(Collection.JOURNAL, Collections.singletonList(changes.asUpdateOp(lastRootRev)));
                }
            }, new ReentrantLock());
            this.log.info("Updated lastRev of [{}] documents while performing lastRev recovery for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        }
        return size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int recoverCandidates(int clusterId, long startTime) {
        int myClusterId = this.nodeStore.getClusterId();
        boolean lockAcquired = this.missingLastRevUtil.acquireRecoveryLock(clusterId, myClusterId);
        if (!lockAcquired) {
            this.log.info("Last revision recovery already being performed by some other node. Would not attempt recovery");
            return 0;
        }
        Iterable<NodeDocument> suspects = this.missingLastRevUtil.getCandidates(startTime);
        this.log.info("Performing Last Revision Recovery for clusterNodeId {}", (Object)clusterId);
        try {
            int n = this.recover(suspects.iterator(), clusterId);
            return n;
        }
        finally {
            Utils.closeIfCloseable(suspects);
            this.missingLastRevUtil.releaseRecoveryLock(clusterId);
            this.nodeStore.signalClusterStateChange();
        }
    }

    @CheckForNull
    private Revision determineLastModification(NodeDocument doc, int clusterId) {
        ClusterPredicate cp = new ClusterPredicate(clusterId);
        Revision lastModified = null;
        block0: for (String property : Sets.filter(doc.keySet(), Utils.PROPERTY_OR_DELETED)) {
            SortedMap<Revision, String> valueMap = doc.getLocalMap(property);
            for (Map.Entry entry : Maps.filterKeys(valueMap, (Predicate)cp).entrySet()) {
                Revision rev = (Revision)entry.getKey();
                if (!doc.isCommitted(rev)) continue;
                lastModified = Utils.max(lastModified, doc.getCommitRevision(rev));
                continue block0;
            }
        }
        return lastModified;
    }

    public boolean isRecoveryNeeded() {
        return this.missingLastRevUtil.isRecoveryNeeded(this.nodeStore.getClock().getTime());
    }

    public void performRecoveryIfNeeded() {
        if (this.isRecoveryNeeded()) {
            List<Integer> clusterIds = this.getRecoveryCandidateNodes();
            this.log.info("ClusterNodeId [{}] starting Last Revision Recovery for clusterNodeId(s) {}", (Object)this.nodeStore.getClusterId(), clusterIds);
            for (int clusterId : clusterIds) {
                this.recover(clusterId);
            }
        }
    }

    public List<Integer> getRecoveryCandidateNodes() {
        Iterable<ClusterNodeInfoDocument> clusters = this.missingLastRevUtil.getAllClusters();
        ArrayList candidateClusterNodes = Lists.newArrayList();
        ArrayList beingRecoveredRightNow = Lists.newArrayList();
        for (ClusterNodeInfoDocument nodeInfo : clusters) {
            String id = nodeInfo.getId();
            if (nodeInfo.isBeingRecovered()) {
                Long recoveredBy = (Long)nodeInfo.get("recoveryBy");
                beingRecoveredRightNow.add(nodeInfo == null ? id : String.format("%s (by %d)", id, recoveredBy));
                continue;
            }
            if (!this.isRecoveryNeeded(nodeInfo)) continue;
            candidateClusterNodes.add(Integer.valueOf(id));
        }
        if (!beingRecoveredRightNow.isEmpty()) {
            this.log.info("Active cluster nodes already in the process of being recovered: " + beingRecoveredRightNow);
        }
        return candidateClusterNodes;
    }

    private boolean isRecoveryNeeded(@Nonnull ClusterNodeInfoDocument nodeInfo) {
        return nodeInfo.isActive() && this.nodeStore.getClock().getTime() > nodeInfo.getLeaseEndTime() && !nodeInfo.isBeingRecovered();
    }

    private static class ClusterPredicate
    implements Predicate<Revision> {
        private final int clusterId;

        private ClusterPredicate(int clusterId) {
            this.clusterId = clusterId;
        }

        public boolean apply(Revision input) {
            return this.clusterId == input.getClusterId();
        }
    }
}

