/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.StandardSystemProperty;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterators;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.UnmodifiableIterator;
import java.io.Closeable;
import java.io.IOException;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.commons.sort.StringSort;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.NodeDocumentIdComparator;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionVector;
import org.apache.jackrabbit.oak.plugins.document.UpdateOp;
import org.apache.jackrabbit.oak.plugins.document.VersionGCSupport;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VersionGarbageCollector {
    private static final int DELETE_BATCH_SIZE = 450;
    private static final int PROGRESS_BATCH_SIZE = 10000;
    private static final UpdateOp.Key KEY_MODIFIED = new UpdateOp.Key("_modified", null);
    private final DocumentNodeStore nodeStore;
    private final DocumentStore ds;
    private final VersionGCSupport versionStore;
    private int overflowToDiskThreshold = 100000;
    private static final Logger log = LoggerFactory.getLogger(VersionGarbageCollector.class);
    private static final Set<NodeDocument.SplitDocType> GC_TYPES = EnumSet.of(NodeDocument.SplitDocType.DEFAULT_LEAF, NodeDocument.SplitDocType.COMMIT_ROOT_ONLY);

    VersionGarbageCollector(DocumentNodeStore nodeStore, VersionGCSupport gcSupport) {
        this.nodeStore = nodeStore;
        this.versionStore = gcSupport;
        this.ds = nodeStore.getDocumentStore();
    }

    public VersionGCStats gc(long maxRevisionAge, TimeUnit unit) throws IOException {
        long maxRevisionAgeInMillis = unit.toMillis(maxRevisionAge);
        Stopwatch sw = Stopwatch.createStarted();
        VersionGCStats stats = new VersionGCStats();
        long oldestRevTimeStamp = this.nodeStore.getClock().getTime() - maxRevisionAgeInMillis;
        RevisionVector headRevision = this.nodeStore.getHeadRevision();
        log.info("Starting revision garbage collection. Revisions older than [{}] will be removed", (Object)Utils.timestampToString(oldestRevTimeStamp));
        Revision checkpoint = this.nodeStore.getCheckpoints().getOldestRevisionToKeep();
        if (checkpoint != null && checkpoint.getTimestamp() < oldestRevTimeStamp) {
            log.info("Ignoring revision garbage collection because a valid checkpoint [{}] was found, which is older than [{}].", (Object)checkpoint.toReadableString(), (Object)Utils.timestampToString(oldestRevTimeStamp));
            stats.ignoredGCDueToCheckPoint = true;
            return stats;
        }
        this.collectDeletedDocuments(stats, headRevision, oldestRevTimeStamp);
        this.collectSplitDocuments(stats, oldestRevTimeStamp);
        sw.stop();
        log.info("Revision garbage collection finished in {}. {}", (Object)sw, (Object)stats);
        return stats;
    }

    public void setOverflowToDiskThreshold(int overflowToDiskThreshold) {
        this.overflowToDiskThreshold = overflowToDiskThreshold;
    }

    private void collectSplitDocuments(VersionGCStats stats, long oldestRevTimeStamp) {
        stats.collectAndDeleteSplitDocs.start();
        this.versionStore.deleteSplitDocuments(GC_TYPES, oldestRevTimeStamp, stats);
        stats.collectAndDeleteSplitDocs.stop();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void collectDeletedDocuments(VersionGCStats stats, RevisionVector headRevision, long oldestRevTimeStamp) throws IOException {
        int docsTraversed = 0;
        DeletedDocsGC gc = new DeletedDocsGC(headRevision);
        try {
            stats.collectDeletedDocs.start();
            Iterable<NodeDocument> itr = this.versionStore.getPossiblyDeletedDocs(oldestRevTimeStamp);
            try {
                for (NodeDocument doc : itr) {
                    if (++docsTraversed % 10000 == 0) {
                        log.info("Iterated through {} documents so far. {} found to be deleted", (Object)docsTraversed, (Object)gc.getNumDocuments());
                    }
                    gc.possiblyDeleted(doc);
                }
            }
            finally {
                Utils.closeIfCloseable(itr);
            }
            stats.collectDeletedDocs.stop();
            if (gc.getNumDocuments() == 0L) {
                return;
            }
            stats.deleteDeletedDocs.start();
            gc.removeDocuments(stats);
            this.nodeStore.invalidateDocChildrenCache();
            stats.deleteDeletedDocs.stop();
        }
        finally {
            gc.close();
        }
    }

    @Nonnull
    private StringSort newStringSort() {
        return new StringSort(this.overflowToDiskThreshold, NodeDocumentIdComparator.INSTANCE);
    }

    private class DeletedDocsGC
    implements Closeable {
        private final RevisionVector headRevision;
        private final StringSort docIdsToDelete;
        private final StringSort prevDocIdsToDelete;
        private final Set<String> exclude;
        private boolean sorted;

        public DeletedDocsGC(RevisionVector headRevision) {
            this.docIdsToDelete = VersionGarbageCollector.this.newStringSort();
            this.prevDocIdsToDelete = VersionGarbageCollector.this.newStringSort();
            this.exclude = Sets.newHashSet();
            this.sorted = false;
            this.headRevision = (RevisionVector)Preconditions.checkNotNull((Object)headRevision);
        }

        long getNumDocuments() {
            return this.docIdsToDelete.getSize();
        }

        void possiblyDeleted(NodeDocument doc) throws IOException {
            if (doc.getNodeAtRevision(VersionGarbageCollector.this.nodeStore, this.headRevision, null) == null) {
                String id = doc.getId() + "/" + doc.getModified();
                this.addDocument(id);
                for (NodeDocument prevDoc : ImmutableList.copyOf(doc.getAllPreviousDocs())) {
                    this.addPreviousDocument(prevDoc.getId());
                }
            }
        }

        void removeDocuments(VersionGCStats stats) throws IOException {
            stats.deletedDocGCCount += this.removeDeletedDocuments();
            stats.splitDocGCCount += this.removeDeletedPreviousDocuments();
        }

        @Override
        public void close() {
            try {
                this.docIdsToDelete.close();
            }
            catch (IOException e) {
                log.warn("Failed to close docIdsToDelete", (Throwable)e);
            }
            try {
                this.prevDocIdsToDelete.close();
            }
            catch (IOException e) {
                log.warn("Failed to close prevDocIdsToDelete", (Throwable)e);
            }
        }

        private void addDocument(String id) throws IOException {
            this.docIdsToDelete.add(id);
        }

        private long getNumPreviousDocuments() {
            return this.prevDocIdsToDelete.getSize() - (long)this.exclude.size();
        }

        private void addPreviousDocument(String id) throws IOException {
            this.prevDocIdsToDelete.add(id);
        }

        private Iterator<String> getDocIdsToDelete() throws IOException {
            this.ensureSorted();
            return this.docIdsToDelete.getIds();
        }

        private void concurrentModification(NodeDocument doc) {
            for (NodeDocument prevDoc : ImmutableList.copyOf(doc.getAllPreviousDocs())) {
                this.exclude.add(prevDoc.getId());
            }
        }

        private Iterator<String> getPrevDocIdsToDelete() throws IOException {
            this.ensureSorted();
            return Iterators.filter(this.prevDocIdsToDelete.getIds(), (Predicate)new Predicate<String>(){

                public boolean apply(String input) {
                    return !DeletedDocsGC.this.exclude.contains(input);
                }
            });
        }

        private int removeDeletedDocuments() throws IOException {
            Iterator<String> docIdsToDelete = this.getDocIdsToDelete();
            log.info("Proceeding to delete [{}] documents", (Object)this.getNumDocuments());
            UnmodifiableIterator idListItr = Iterators.partition(docIdsToDelete, (int)450);
            int deletedCount = 0;
            int lastLoggedCount = 0;
            int recreatedCount = 0;
            while (idListItr.hasNext()) {
                int nRemoved;
                LinkedHashMap deletionBatch = Maps.newLinkedHashMap();
                for (String s : (List)idListItr.next()) {
                    int idx = s.lastIndexOf(47);
                    String id = s.substring(0, idx);
                    long modified = -1L;
                    try {
                        modified = Long.parseLong(s.substring(idx + 1));
                    }
                    catch (NumberFormatException e) {
                        log.warn("Invalid _modified {} for {}", (Object)s.substring(idx + 1), (Object)id);
                    }
                    deletionBatch.put(id, Collections.singletonMap(KEY_MODIFIED, UpdateOp.Condition.newEqualsCondition(modified)));
                }
                if (log.isDebugEnabled()) {
                    StringBuilder sb = new StringBuilder("Performing batch deletion of documents with following ids. \n");
                    Joiner.on((String)StandardSystemProperty.LINE_SEPARATOR.value()).appendTo(sb, deletionBatch.keySet());
                    log.debug(sb.toString());
                }
                if ((nRemoved = VersionGarbageCollector.this.ds.remove(Collection.NODES, deletionBatch)) < deletionBatch.size()) {
                    for (String id : deletionBatch.keySet()) {
                        NodeDocument d = VersionGarbageCollector.this.ds.find(Collection.NODES, id);
                        if (d == null) continue;
                        this.concurrentModification(d);
                    }
                    recreatedCount += deletionBatch.size() - nRemoved;
                }
                log.debug("Deleted [{}] documents so far", (Object)(deletedCount += nRemoved));
                if (deletedCount + recreatedCount - lastLoggedCount < 10000) continue;
                lastLoggedCount = deletedCount + recreatedCount;
                double progress = (double)lastLoggedCount * 1.0 / (double)this.getNumDocuments() * 100.0;
                String msg = String.format("Deleted %d (%1.2f%%) documents so far", deletedCount, progress);
                log.info(msg);
            }
            return deletedCount;
        }

        private int removeDeletedPreviousDocuments() throws IOException {
            log.info("Proceeding to delete [{}] previous documents", (Object)this.getNumPreviousDocuments());
            int deletedCount = 0;
            int lastLoggedCount = 0;
            UnmodifiableIterator idListItr = Iterators.partition(this.getPrevDocIdsToDelete(), (int)450);
            while (idListItr.hasNext()) {
                List deletionBatch = (List)idListItr.next();
                deletedCount += deletionBatch.size();
                if (log.isDebugEnabled()) {
                    StringBuilder sb = new StringBuilder("Performing batch deletion of previous documents with following ids. \n");
                    Joiner.on((String)StandardSystemProperty.LINE_SEPARATOR.value()).appendTo(sb, (Iterable)deletionBatch);
                    log.debug(sb.toString());
                }
                VersionGarbageCollector.this.ds.remove(Collection.NODES, deletionBatch);
                log.debug("Deleted [{}] previous documents so far", (Object)deletedCount);
                if (deletedCount - lastLoggedCount < 10000) continue;
                lastLoggedCount = deletedCount;
                double progress = (double)deletedCount * 1.0 / (double)(this.prevDocIdsToDelete.getSize() - (long)this.exclude.size()) * 100.0;
                String msg = String.format("Deleted %d (%1.2f%%) previous documents so far", deletedCount, progress);
                log.info(msg);
            }
            return deletedCount;
        }

        private void ensureSorted() throws IOException {
            if (!this.sorted) {
                this.docIdsToDelete.sort();
                this.prevDocIdsToDelete.sort();
                this.sorted = true;
            }
        }
    }

    public static class VersionGCStats {
        boolean ignoredGCDueToCheckPoint;
        int deletedDocGCCount;
        int splitDocGCCount;
        int intermediateSplitDocGCCount;
        final Stopwatch collectDeletedDocs = Stopwatch.createUnstarted();
        final Stopwatch deleteDeletedDocs = Stopwatch.createUnstarted();
        final Stopwatch collectAndDeleteSplitDocs = Stopwatch.createUnstarted();

        public String toString() {
            return "VersionGCStats{ignoredGCDueToCheckPoint=" + this.ignoredGCDueToCheckPoint + ", deletedDocGCCount=" + this.deletedDocGCCount + ", splitDocGCCount=" + this.splitDocGCCount + ", intermediateSplitDocGCCount=" + this.intermediateSplitDocGCCount + ", timeToCollectDeletedDocs=" + this.collectDeletedDocs + ", timeTakenToDeleteDeletedDocs=" + this.deleteDeletedDocs + ", timeTakenToCollectAndDeleteSplitDocs=" + this.collectAndDeleteSplitDocs + '}';
        }
    }
}

