/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.security.authentication.token;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.jcr.AccessDeniedException;
import javax.jcr.Credentials;
import javax.jcr.RepositoryException;
import javax.jcr.SimpleCredentials;
import org.apache.jackrabbit.api.security.authentication.token.TokenCredentials;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.jackrabbit.api.security.user.UserManager;
import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Root;
import org.apache.jackrabbit.oak.api.Tree;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.namepath.NamePathMapper;
import org.apache.jackrabbit.oak.plugins.identifier.IdentifierManager;
import org.apache.jackrabbit.oak.plugins.name.NamespaceConstants;
import org.apache.jackrabbit.oak.security.authentication.token.CommitMarker;
import org.apache.jackrabbit.oak.security.authentication.token.TokenConstants;
import org.apache.jackrabbit.oak.spi.security.ConfigurationParameters;
import org.apache.jackrabbit.oak.spi.security.authentication.ImpersonationCredentials;
import org.apache.jackrabbit.oak.spi.security.authentication.token.TokenInfo;
import org.apache.jackrabbit.oak.spi.security.authentication.token.TokenProvider;
import org.apache.jackrabbit.oak.spi.security.user.UserConfiguration;
import org.apache.jackrabbit.oak.spi.security.user.util.PasswordUtil;
import org.apache.jackrabbit.oak.util.NodeUtil;
import org.apache.jackrabbit.oak.util.TreeUtil;
import org.apache.jackrabbit.util.ISO8601;
import org.apache.jackrabbit.util.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TokenProviderImpl
implements TokenProvider,
TokenConstants {
    private static final Logger log = LoggerFactory.getLogger(TokenProviderImpl.class);
    private static final long DEFAULT_TOKEN_EXPIRATION = 0x6DDD00L;
    private static final int DEFAULT_KEY_SIZE = 8;
    private static final char DELIM = '_';
    private final Root root;
    private final ConfigurationParameters options;
    private final long tokenExpiration;
    private final UserManager userManager;
    private final IdentifierManager identifierManager;

    TokenProviderImpl(Root root, ConfigurationParameters options, UserConfiguration userConfiguration) {
        this.root = root;
        this.options = options;
        this.tokenExpiration = options.getConfigValue("tokenExpiration", 0x6DDD00L);
        this.userManager = userConfiguration.getUserManager(root, NamePathMapper.DEFAULT);
        this.identifierManager = new IdentifierManager(root);
    }

    @Override
    public boolean doCreateToken(@Nonnull Credentials credentials) {
        SimpleCredentials sc = TokenProviderImpl.extractSimpleCredentials(credentials);
        if (sc == null) {
            return false;
        }
        Object attr = sc.getAttribute(".token");
        return attr != null && "".equals(attr.toString());
    }

    @Override
    @CheckForNull
    public TokenInfo createToken(@Nonnull Credentials credentials) {
        SimpleCredentials sc = TokenProviderImpl.extractSimpleCredentials(credentials);
        TokenInfo tokenInfo = null;
        if (sc != null) {
            String[] attrNames = sc.getAttributeNames();
            HashMap<String, String> attributes = new HashMap<String, String>(attrNames.length);
            for (String attrName : sc.getAttributeNames()) {
                attributes.put(attrName, sc.getAttribute(attrName).toString());
            }
            tokenInfo = this.createToken(sc.getUserID(), attributes);
            if (tokenInfo != null) {
                sc.setAttribute(".token", (Object)tokenInfo.getToken());
            }
        }
        return tokenInfo;
    }

    @Override
    public TokenInfo createToken(@Nonnull String userId, @Nonnull Map<String, ?> attributes) {
        String error = "Failed to create login token. {}";
        User user = this.getUser(userId);
        NodeUtil tokenParent = this.getTokenParent(user);
        if (tokenParent != null && user != null) {
            try {
                TokenInfo tokenInfo;
                String id = user.getID();
                long creationTime = new Date().getTime();
                long exp = attributes.containsKey("tokenExpiration") ? Long.parseLong(attributes.get("tokenExpiration").toString()) : this.tokenExpiration;
                long expTime = TokenProviderImpl.createExpirationTime(creationTime, exp);
                String uuid = UUID.randomUUID().toString();
                try {
                    String tokenName = TokenProviderImpl.generateTokenName(creationTime);
                    tokenInfo = this.createTokenNode(tokenParent, tokenName, expTime, uuid, id, attributes);
                    this.root.commit(CommitMarker.asCommitAttributes());
                }
                catch (CommitFailedException e) {
                    log.debug("Failed to create token node. Using random name as fallback.");
                    this.root.refresh();
                    tokenInfo = this.createTokenNode(tokenParent, UUID.randomUUID().toString(), expTime, uuid, id, attributes);
                    this.root.commit(CommitMarker.asCommitAttributes());
                }
                return tokenInfo;
            }
            catch (NoSuchAlgorithmException e) {
                log.error(error, (Object)e.getMessage());
            }
            catch (UnsupportedEncodingException e) {
                log.error(error, (Object)e.getMessage());
            }
            catch (CommitFailedException e) {
                log.warn(error, (Object)e.getMessage());
            }
            catch (RepositoryException e) {
                log.warn(error, (Object)e.getMessage());
            }
        } else {
            log.warn("Unable to get/create token store for user " + userId);
        }
        return null;
    }

    @Override
    public TokenInfo getTokenInfo(@Nonnull String token) {
        int pos = token.indexOf(95);
        String nodeId = pos == -1 ? token : token.substring(0, pos);
        Tree tokenTree = this.identifierManager.getTree(nodeId);
        String userId = this.getUserId(tokenTree);
        if (userId == null || !TokenProviderImpl.isValidTokenTree(tokenTree)) {
            return null;
        }
        return new TokenInfoImpl(new NodeUtil(tokenTree), token, userId);
    }

    private static long createExpirationTime(long creationTime, long tokenExpiration) {
        return creationTime + tokenExpiration;
    }

    private static long getExpirationTime(NodeUtil tokenNode, long defaultValue) {
        return tokenNode.getLong("rep:token.exp", defaultValue);
    }

    @CheckForNull
    private static SimpleCredentials extractSimpleCredentials(Credentials credentials) {
        Credentials base;
        if (credentials instanceof SimpleCredentials) {
            return (SimpleCredentials)credentials;
        }
        if (credentials instanceof ImpersonationCredentials && (base = ((ImpersonationCredentials)credentials).getBaseCredentials()) instanceof SimpleCredentials) {
            return (SimpleCredentials)base;
        }
        return null;
    }

    @Nonnull
    private static String generateKey(int size) {
        SecureRandom random = new SecureRandom();
        byte[] key = new byte[size];
        random.nextBytes(key);
        StringBuilder res = new StringBuilder(key.length * 2);
        for (byte b : key) {
            res.append(Text.hexTable[b >> 4 & 0xF]);
            res.append(Text.hexTable[b & 0xF]);
        }
        return res.toString();
    }

    @Nonnull
    private static String getKeyValue(@Nonnull String key, @Nonnull String userId) {
        return key + userId;
    }

    private static boolean isValidTokenTree(@CheckForNull Tree tokenTree) {
        if (tokenTree == null || !tokenTree.exists()) {
            return false;
        }
        return ".tokens".equals(tokenTree.getParent().getName()) && "rep:Token".equals(TreeUtil.getPrimaryTypeName(tokenTree));
    }

    @Nonnull
    private static String generateTokenName(long creationTime) {
        Calendar creation = GregorianCalendar.getInstance();
        creation.setTimeInMillis(creationTime);
        return Text.replace(ISO8601.format(creation), ":", ".");
    }

    @CheckForNull
    private Tree getTokenTree(@Nonnull TokenInfo tokenInfo) {
        if (tokenInfo instanceof TokenInfoImpl) {
            return this.root.getTree(((TokenInfoImpl)tokenInfo).tokenPath);
        }
        return null;
    }

    @CheckForNull
    private String getUserId(@CheckForNull Tree tokenTree) {
        if (tokenTree != null && tokenTree.exists()) {
            try {
                String userPath = Text.getRelativeParent(tokenTree.getPath(), 2);
                Authorizable authorizable = this.userManager.getAuthorizableByPath(userPath);
                if (authorizable != null && !authorizable.isGroup() && !((User)authorizable).isDisabled()) {
                    return authorizable.getID();
                }
            }
            catch (RepositoryException e) {
                log.debug("Cannot determine userID from token: {}", (Object)e.getMessage());
            }
        }
        return null;
    }

    @CheckForNull
    private User getUser(@Nonnull String userId) {
        try {
            Authorizable user = this.userManager.getAuthorizable(userId);
            if (user != null && !user.isGroup()) {
                return (User)user;
            }
            log.debug("Cannot create login token: No corresponding node for User " + userId + '.');
        }
        catch (RepositoryException e) {
            log.debug("Error while accessing user " + userId + '.', (Throwable)e);
        }
        return null;
    }

    @CheckForNull
    private NodeUtil getTokenParent(@CheckForNull User user) {
        NodeUtil tokenParent;
        block4: {
            if (user == null) {
                return null;
            }
            tokenParent = null;
            String parentPath = null;
            try {
                String userPath = user.getPath();
                parentPath = userPath + '/' + ".tokens";
                NodeUtil userNode = new NodeUtil(this.root.getTree(userPath));
                tokenParent = userNode.getOrAddChild(".tokens", "rep:Unstructured");
                this.root.commit();
            }
            catch (RepositoryException e) {
                log.debug("Error while creating token node {}", (Object)e.getMessage());
            }
            catch (CommitFailedException e) {
                log.debug("Conflict while creating token store -> retrying {}", (Object)e.getMessage());
                this.root.refresh();
                Tree parentTree = this.root.getTree(parentPath);
                if (!parentTree.exists()) break block4;
                tokenParent = new NodeUtil(parentTree);
            }
        }
        return tokenParent;
    }

    private TokenInfo createTokenNode(@Nonnull NodeUtil parent, @Nonnull String tokenName, long expTime, @Nonnull String uuid, @Nonnull String id, Map<String, ?> attributes) throws AccessDeniedException, UnsupportedEncodingException, NoSuchAlgorithmException {
        NodeUtil tokenNode = parent.addChild(tokenName, "rep:Token");
        tokenNode.setString("jcr:uuid", uuid);
        String key = TokenProviderImpl.generateKey(this.options.getConfigValue("tokenLength", 8));
        String nodeId = IdentifierManager.getIdentifier(tokenNode.getTree());
        String token = nodeId + '_' + key;
        String keyHash = PasswordUtil.buildPasswordHash(TokenProviderImpl.getKeyValue(key, id), this.options);
        tokenNode.setString("rep:token.key", keyHash);
        tokenNode.setDate("rep:token.exp", expTime);
        for (String name : attributes.keySet()) {
            if (RESERVED_ATTRIBUTES.contains(name)) continue;
            String attr = attributes.get(name).toString();
            tokenNode.setString(name, attr);
        }
        return new TokenInfoImpl(tokenNode, token, id);
    }

    private final class TokenInfoImpl
    implements TokenInfo {
        private final String token;
        private final String tokenPath;
        private final String userId;
        private final long expirationTime;
        private final String key;
        private final Map<String, String> mandatoryAttributes;
        private final Map<String, String> publicAttributes;

        private TokenInfoImpl(NodeUtil tokenNode, String token, String userId) {
            this.token = token;
            this.tokenPath = tokenNode.getTree().getPath();
            this.userId = userId;
            this.expirationTime = TokenProviderImpl.getExpirationTime(tokenNode, Long.MIN_VALUE);
            this.key = tokenNode.getString("rep:token.key", null);
            this.mandatoryAttributes = new HashMap<String, String>();
            this.publicAttributes = new HashMap<String, String>();
            for (PropertyState propertyState : tokenNode.getTree().getProperties()) {
                String name = propertyState.getName();
                String value = propertyState.getValue(Type.STRING);
                if (TokenConstants.RESERVED_ATTRIBUTES.contains(name)) continue;
                if (this.isMandatoryAttribute(name)) {
                    this.mandatoryAttributes.put(name, value);
                    continue;
                }
                if (!this.isInfoAttribute(name)) continue;
                this.publicAttributes.put(name, value);
            }
        }

        @Override
        @Nonnull
        public String getUserId() {
            return this.userId;
        }

        @Override
        @Nonnull
        public String getToken() {
            return this.token;
        }

        @Override
        public boolean isExpired(long loginTime) {
            return this.expirationTime < loginTime;
        }

        @Override
        public boolean resetExpiration(long loginTime) {
            Tree tokenTree;
            if (TokenProviderImpl.this.options.getConfigValue("tokenRefresh", true).booleanValue() && (tokenTree = TokenProviderImpl.this.getTokenTree(this)) != null && tokenTree.exists()) {
                NodeUtil tokenNode = new NodeUtil(tokenTree);
                if (this.isExpired(loginTime)) {
                    log.debug("Attempt to reset an expired token.");
                    return false;
                }
                if (this.expirationTime - loginTime <= TokenProviderImpl.this.tokenExpiration / 2L) {
                    try {
                        long expTime = TokenProviderImpl.createExpirationTime(loginTime, TokenProviderImpl.this.tokenExpiration);
                        tokenNode.setDate("rep:token.exp", expTime);
                        TokenProviderImpl.this.root.commit(CommitMarker.asCommitAttributes());
                        log.debug("Successfully reset token expiration time.");
                        return true;
                    }
                    catch (CommitFailedException e) {
                        log.debug("Failed to reset token expiration {}", (Object)e.getMessage());
                        TokenProviderImpl.this.root.refresh();
                    }
                }
            }
            return false;
        }

        @Override
        public boolean remove() {
            Tree tokenTree = TokenProviderImpl.this.getTokenTree(this);
            if (tokenTree != null && tokenTree.exists()) {
                try {
                    if (tokenTree.remove()) {
                        TokenProviderImpl.this.root.commit(CommitMarker.asCommitAttributes());
                        return true;
                    }
                }
                catch (CommitFailedException e) {
                    log.debug("Error while removing expired token {}", (Object)e.getMessage());
                }
            }
            return false;
        }

        @Override
        public boolean matches(TokenCredentials tokenCredentials) {
            String tk = tokenCredentials.getToken();
            int pos = tk.lastIndexOf(95);
            if (pos > -1) {
                tk = tk.substring(pos + 1);
            }
            if (this.key == null || !PasswordUtil.isSame(this.key, TokenProviderImpl.getKeyValue(tk, this.userId))) {
                return false;
            }
            for (Map.Entry<String, String> mandatory : this.mandatoryAttributes.entrySet()) {
                String name = mandatory.getKey();
                String expectedValue = mandatory.getValue();
                if (expectedValue.equals(tokenCredentials.getAttribute(name))) continue;
                return false;
            }
            List<String> attrNames = Arrays.asList(tokenCredentials.getAttributeNames());
            for (Map.Entry<String, String> attr : this.publicAttributes.entrySet()) {
                String name = attr.getKey();
                if (attrNames.contains(name)) continue;
                tokenCredentials.setAttribute(name, attr.getValue());
            }
            return true;
        }

        @Override
        @Nonnull
        public Map<String, String> getPrivateAttributes() {
            return Collections.unmodifiableMap(this.mandatoryAttributes);
        }

        @Override
        @Nonnull
        public Map<String, String> getPublicAttributes() {
            return Collections.unmodifiableMap(this.publicAttributes);
        }

        private boolean isMandatoryAttribute(String attributeName) {
            return attributeName != null && attributeName.startsWith(".token");
        }

        private boolean isInfoAttribute(String attributeName) {
            String prefix = Text.getNamespacePrefix(attributeName);
            return !NamespaceConstants.RESERVED_PREFIXES.contains(prefix);
        }
    }
}

