/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.cloud.api.collections;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.store.OutputStreamIndexOutput;
import org.apache.solr.SolrTestCaseJ4;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.core.PluginInfo;
import org.apache.solr.core.SolrResourceLoader;
import org.apache.solr.core.backup.repository.BackupRepository;
import org.apache.solr.core.backup.repository.BackupRepositoryFactory;
import org.apache.solr.core.backup.repository.DelegatingBackupRepository;
import org.junit.Test;

public abstract class AbstractBackupRepositoryTest
extends SolrTestCaseJ4 {
    protected abstract Class<? extends BackupRepository> getRepositoryClass();

    protected abstract BackupRepository getRepository();

    protected abstract URI getBaseUri() throws URISyntaxException;

    protected NamedList<Object> getBaseBackupRepositoryConfiguration() {
        return new NamedList();
    }

    @Test
    public void testCanReadProvidedConfigValues() throws Exception {
        NamedList<Object> config = this.getBaseBackupRepositoryConfiguration();
        config.add("configKey1", (Object)"configVal1");
        config.add("configKey2", (Object)"configVal2");
        config.add("location", (Object)"foo");
        try (BackupRepository repo = this.getRepository();){
            repo.init(config);
            AbstractBackupRepositoryTest.assertEquals((Object)"configVal1", (Object)repo.getConfigProperty("configKey1"));
            AbstractBackupRepositoryTest.assertEquals((Object)"configVal2", (Object)repo.getConfigProperty("configKey2"));
        }
    }

    @Test
    public void testCanChooseDefaultOrOverrideLocationValue() throws Exception {
        NamedList<Object> config = this.getBaseBackupRepositoryConfiguration();
        int locationIndex = config.indexOf("location", 0);
        if (locationIndex == -1) {
            config.add("location", (Object)"someLocation");
        } else {
            config.setVal(locationIndex, (Object)"someLocation");
        }
        try (BackupRepository repo = this.getRepository();){
            repo.init(config);
            AbstractBackupRepositoryTest.assertEquals((Object)"someLocation", (Object)repo.getBackupLocation(null));
            AbstractBackupRepositoryTest.assertEquals((Object)"someOverridingLocation", (Object)repo.getBackupLocation("someOverridingLocation"));
        }
    }

    @Test
    public void testCanDetermineWhetherFilesAndDirectoriesExist() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            URI emptyDirUri = repo.resolve(this.getBaseUri(), new String[]{"emptyDir/"});
            URI nonEmptyDirUri = repo.resolve(this.getBaseUri(), new String[]{"nonEmptyDir/"});
            URI nestedFileUri = repo.resolve(nonEmptyDirUri, new String[]{"file.txt"});
            repo.createDirectory(emptyDirUri);
            repo.createDirectory(nonEmptyDirUri);
            this.addFile(repo, nestedFileUri);
            AbstractBackupRepositoryTest.assertTrue((boolean)repo.exists(emptyDirUri));
            AbstractBackupRepositoryTest.assertTrue((boolean)repo.exists(nonEmptyDirUri));
            AbstractBackupRepositoryTest.assertTrue((boolean)repo.exists(nestedFileUri));
            URI nonexistedDirUri = repo.resolve(this.getBaseUri(), new String[]{"nonexistentDir/"});
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(nonexistedDirUri));
        }
    }

    @Test
    public void testCanDistinguishBetweenFilesAndDirectories() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            URI emptyDirUri = repo.resolve(this.getBaseUri(), new String[]{"emptyDir/"});
            URI nonEmptyDirUri = repo.resolve(this.getBaseUri(), new String[]{"nonEmptyDir/"});
            URI nestedFileUri = repo.resolve(nonEmptyDirUri, new String[]{"file.txt"});
            repo.createDirectory(emptyDirUri);
            repo.createDirectory(nonEmptyDirUri);
            this.addFile(repo, nestedFileUri);
            AbstractBackupRepositoryTest.assertEquals((Object)BackupRepository.PathType.DIRECTORY, (Object)repo.getPathType(emptyDirUri));
            AbstractBackupRepositoryTest.assertEquals((Object)BackupRepository.PathType.DIRECTORY, (Object)repo.getPathType(nonEmptyDirUri));
            AbstractBackupRepositoryTest.assertEquals((Object)BackupRepository.PathType.FILE, (Object)repo.getPathType(nestedFileUri));
        }
    }

    @Test
    public void testArbitraryFileDataCanBeStoredAndRetrieved() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            URI fileUri = repo.resolve(this.getBaseUri(), new String[]{"file.txt"});
            byte[] storedBytes = new byte[]{104, 101, 108, 108, 111};
            try (OutputStream os = repo.createOutput(fileUri);){
                os.write(storedBytes);
            }
            int expectedNumBytes = storedBytes.length;
            byte[] retrievedBytes = new byte[expectedNumBytes];
            try (IndexInput is = repo.openInput(this.getBaseUri(), "file.txt", new IOContext(IOContext.Context.READ));){
                AbstractBackupRepositoryTest.assertEquals((long)expectedNumBytes, (long)is.length());
                is.readBytes(retrievedBytes, 0, expectedNumBytes);
            }
            AbstractBackupRepositoryTest.assertArrayEquals((byte[])storedBytes, (byte[])retrievedBytes);
        }
    }

    @Test
    public void testCanDeleteEmptyOrFullDirectories() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            URI emptyDirUri = repo.resolve(this.getBaseUri(), new String[]{"emptyDir"});
            URI nonEmptyDirUri = repo.resolve(this.getBaseUri(), new String[]{"nonEmptyDir"});
            URI fileUri = repo.resolve(nonEmptyDirUri, new String[]{"file.txt"});
            repo.createDirectory(emptyDirUri);
            repo.createDirectory(nonEmptyDirUri);
            this.addFile(repo, fileUri);
            repo.deleteDirectory(emptyDirUri);
            repo.deleteDirectory(nonEmptyDirUri);
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(emptyDirUri));
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(nonEmptyDirUri));
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(fileUri));
            URI level1DeeplyNestedUri = repo.resolve(this.getBaseUri(), new String[]{"nest1/"});
            URI level2DeeplyNestedUri = repo.resolve(level1DeeplyNestedUri, new String[]{"nest2/"});
            URI level3DeeplyNestedUri = repo.resolve(level2DeeplyNestedUri, new String[]{"nest3/"});
            URI level4DeeplyNestedUri = repo.resolve(level3DeeplyNestedUri, new String[]{"nest4/"});
            repo.createDirectory(level1DeeplyNestedUri);
            repo.createDirectory(level2DeeplyNestedUri);
            repo.createDirectory(level3DeeplyNestedUri);
            repo.createDirectory(level4DeeplyNestedUri);
            repo.deleteDirectory(level3DeeplyNestedUri);
            AbstractBackupRepositoryTest.assertTrue((boolean)repo.exists(level1DeeplyNestedUri));
            AbstractBackupRepositoryTest.assertTrue((boolean)repo.exists(level2DeeplyNestedUri));
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(level3DeeplyNestedUri));
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(level4DeeplyNestedUri));
        }
    }

    @Test
    public void testDirectoryCreationFailsIfParentDoesntExist() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            URI nonExistentParentUri = repo.resolve(this.getBaseUri(), new String[]{"nonExistentParent"});
            URI nestedUri = repo.resolve(nonExistentParentUri, new String[]{"childDirectoryToCreate"});
            repo.createDirectory(nestedUri);
        }
    }

    @Test
    public void testCanDeleteIndividualFiles() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            URI file1Uri = repo.resolve(this.getBaseUri(), new String[]{"file1.txt"});
            URI file2Uri = repo.resolve(this.getBaseUri(), new String[]{"file2.txt"});
            URI file3Uri = repo.resolve(this.getBaseUri(), new String[]{"file3.txt"});
            this.addFile(repo, file1Uri);
            this.addFile(repo, file2Uri);
            this.addFile(repo, file3Uri);
            URI nonexistentFileUri = repo.resolve(this.getBaseUri(), new String[]{"file4.txt"});
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(nonexistentFileUri));
            repo.delete(this.getBaseUri(), List.of("file4.txt"));
            repo.delete(this.getBaseUri(), List.of("file1.txt"));
            repo.delete(this.getBaseUri(), List.of("file2.txt", "file3.txt"));
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(file1Uri));
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(file2Uri));
            AbstractBackupRepositoryTest.assertFalse((boolean)repo.exists(file3Uri));
        }
    }

    @Test
    public void testCanListFullOrEmptyDirectories() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            URI rootUri = repo.resolve(this.getBaseUri(), new String[]{"containsOtherDirs"});
            URI otherDir1Uri = repo.resolve(rootUri, new String[]{"otherDir1"});
            URI otherDir2Uri = repo.resolve(rootUri, new String[]{"otherDir2"});
            URI otherDir3Uri = repo.resolve(rootUri, new String[]{"otherDir3"});
            URI file1Uri = repo.resolve(otherDir3Uri, new String[]{"file1.txt"});
            URI file2Uri = repo.resolve(otherDir3Uri, new String[]{"file2.txt"});
            repo.createDirectory(rootUri);
            repo.createDirectory(otherDir1Uri);
            repo.createDirectory(otherDir2Uri);
            repo.createDirectory(otherDir3Uri);
            this.addFile(repo, file1Uri);
            this.addFile(repo, file2Uri);
            List<String> rootChildren = List.of(repo.listAll(rootUri));
            AbstractBackupRepositoryTest.assertEquals((long)3L, (long)rootChildren.size());
            AbstractBackupRepositoryTest.assertTrue((boolean)rootChildren.contains("otherDir1"));
            AbstractBackupRepositoryTest.assertTrue((boolean)rootChildren.contains("otherDir2"));
            AbstractBackupRepositoryTest.assertTrue((boolean)rootChildren.contains("otherDir3"));
            String[] otherDir2Children = repo.listAll(otherDir2Uri);
            AbstractBackupRepositoryTest.assertEquals((long)0L, (long)otherDir2Children.length);
            List<String> otherDir3Children = List.of(repo.listAll(otherDir3Uri));
            AbstractBackupRepositoryTest.assertEquals((long)2L, (long)otherDir3Children.size());
            AbstractBackupRepositoryTest.assertTrue((boolean)otherDir3Children.contains("file1.txt"));
            AbstractBackupRepositoryTest.assertTrue((boolean)otherDir3Children.contains("file2.txt"));
        }
    }

    @Test
    public void testDirectoryExistsWithDirectoryUri() throws Exception {
        try (BackupRepository repo = this.getRepository();){
            repo.createDirectory(this.getBaseUri());
            AbstractBackupRepositoryTest.assertTrue((boolean)repo.exists(repo.createDirectoryURI(this.getBaseUri().toString())));
        }
    }

    @Test
    public void testCanDisableChecksumVerification() throws Exception {
        this.getRepository();
        String repoName = "repo";
        String filterRepoName = "filterRepo";
        PluginInfo[] plugins = new PluginInfo[]{this.getPluginInfo(repoName, false), this.getNoChecksumFilterPluginInfo(filterRepoName, true, repoName)};
        Collections.shuffle(Arrays.asList(plugins), AbstractBackupRepositoryTest.random());
        File sourceDir = AbstractBackupRepositoryTest.createTempDir().toFile();
        String fileName = "source-file";
        String content = "content";
        try (FileOutputStream os = FileUtils.openOutputStream((File)new File(sourceDir, fileName));
             OutputStreamIndexOutput io = new OutputStreamIndexOutput("", "", (OutputStream)os, 8);){
            byte[] bytes = content.getBytes(StandardCharsets.UTF_8);
            io.writeBytes(bytes, bytes.length);
            CodecUtil.writeBEInt((DataOutput)io, (int)-1071082520);
            CodecUtil.writeBEInt((DataOutput)io, (int)0);
            CodecUtil.writeBELong((DataOutput)io, (long)1L);
        }
        BackupRepositoryFactory repoFactory = new BackupRepositoryFactory(plugins);
        try (SolrResourceLoader resourceLoader = new SolrResourceLoader(sourceDir.toPath());){
            AbstractBackupRepositoryTest.expectThrows(CorruptIndexException.class, () -> this.copyFileToRepo(sourceDir, fileName, repoName, repoFactory, resourceLoader));
            File destinationDir = AbstractBackupRepositoryTest.createTempDir().toFile();
            AbstractBackupRepositoryTest.expectThrows(CorruptIndexException.class, () -> this.copyFileToDir(sourceDir, fileName, destinationDir, repoName, repoFactory, resourceLoader));
            NoChecksumDelegatingBackupRepository.checksumVerifiedAlternatively = false;
            this.copyFileToRepo(sourceDir, fileName, filterRepoName, repoFactory, resourceLoader);
            AbstractBackupRepositoryTest.assertTrue((boolean)NoChecksumDelegatingBackupRepository.checksumVerifiedAlternatively);
            NoChecksumDelegatingBackupRepository.checksumVerifiedAlternatively = false;
            this.copyFileToDir(sourceDir, fileName, destinationDir, filterRepoName, repoFactory, resourceLoader);
            AbstractBackupRepositoryTest.assertTrue((boolean)NoChecksumDelegatingBackupRepository.checksumVerifiedAlternatively);
        }
    }

    private void copyFileToRepo(File dir, String fileName, String repoName, BackupRepositoryFactory repoFactory, SolrResourceLoader resourceLoader) throws IOException, URISyntaxException {
        try (BackupRepository repo = repoFactory.newInstance(resourceLoader, repoName);
             NIOFSDirectory directory = new NIOFSDirectory(dir.toPath());){
            URI destinationDir = repo.resolve(this.getBaseUri(), new String[]{"destination-folder"});
            repo.copyIndexFileFrom((Directory)directory, fileName, destinationDir, fileName);
        }
    }

    private void copyFileToDir(File sourceDir, String fileName, File destinationDir, String repoName, BackupRepositoryFactory repoFactory, SolrResourceLoader resourceLoader) throws IOException {
        try (BackupRepository repo = repoFactory.newInstance(resourceLoader, repoName);
             NIOFSDirectory sourceDirectory = new NIOFSDirectory(sourceDir.toPath());
             NIOFSDirectory destinationDirectory = new NIOFSDirectory(destinationDir.toPath());){
            repo.copyIndexFileFrom((Directory)sourceDirectory, fileName, (Directory)destinationDirectory, fileName);
        }
    }

    protected PluginInfo getPluginInfo(String pluginName, boolean isDefault) {
        return this.getPluginInfo(pluginName, isDefault, Map.of());
    }

    protected PluginInfo getPluginInfo(String pluginName, boolean isDefault, Map<String, String> attributes) {
        HashMap<String, String> attrs = new HashMap<String, String>();
        attrs.put("name", pluginName);
        attrs.put("class", this.getRepositoryClass().getName());
        attrs.put("default", Boolean.toString(isDefault));
        attrs.putAll(attributes);
        return new PluginInfo("repository", attrs, this.getBaseBackupRepositoryConfiguration(), null);
    }

    private PluginInfo getNoChecksumFilterPluginInfo(String pluginName, boolean isDefault, String delegateName) {
        HashMap<String, String> attrs = new HashMap<String, String>();
        attrs.put("name", pluginName);
        attrs.put("class", NoChecksumDelegatingBackupRepository.class.getName());
        attrs.put("default", Boolean.toString(isDefault));
        NamedList args = new NamedList();
        args.add("delegateRepoName", (Object)delegateName);
        return new PluginInfo("repository", attrs, args, null);
    }

    private void addFile(BackupRepository repo, URI file) throws IOException {
        try (OutputStream os = repo.createOutput(file);){
            os.write(100);
            os.write(101);
            os.write(102);
        }
    }

    public static class NoChecksumDelegatingBackupRepository
    extends DelegatingBackupRepository {
        static volatile boolean checksumVerifiedAlternatively;

        protected NamedList<?> getDelegateInitArgs(NamedList<?> initArgs) {
            NamedList newInitArgs = new NamedList(initArgs.size() + 1);
            newInitArgs.add("verifyChecksum", (Object)Boolean.FALSE.toString());
            newInitArgs.addAll(initArgs);
            return newInitArgs;
        }

        public void copyIndexFileFrom(Directory sourceDir, String sourceFileName, Directory destDir, String destFileName) throws IOException {
            this.verifyChecksum(sourceDir, sourceFileName);
            super.copyIndexFileFrom(FilterDirectory.unwrap((Directory)sourceDir), sourceFileName, destDir, destFileName);
        }

        public void copyIndexFileFrom(Directory sourceDir, String sourceFileName, URI destDir, String destFileName) throws IOException {
            this.verifyChecksum(sourceDir, sourceFileName);
            super.copyIndexFileFrom(FilterDirectory.unwrap((Directory)sourceDir), sourceFileName, destDir, destFileName);
        }

        private void verifyChecksum(Directory sourceDir, String sourceFileName) {
            checksumVerifiedAlternatively = true;
        }
    }
}

