/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.mllib.feature

import org.apache.spark.annotation.DeveloperApi
import org.apache.spark.api.java.JavaRDD
import org.apache.spark.mllib.linalg.Vector
import org.apache.spark.rdd.RDD

/**
 * :: DeveloperApi ::
 * Trait for transformation of a vector
 */
@DeveloperApi
trait VectorTransformer extends Serializable {

  /**
   * Applies transformation on a vector.
   *
   * @param vector vector to be transformed.
   * @return transformed vector.
   */
  def transform(vector: Vector): Vector

  /**
   * Applies transformation on an RDD[Vector].
   *
   * @param data RDD[Vector] to be transformed.
   * @return transformed RDD[Vector].
   */
  def transform(data: RDD[Vector]): RDD[Vector] = {
    // Later in #1498 , all RDD objects are sent via broadcasting instead of akka.
    // So it should be no longer necessary to explicitly broadcast `this` object.
    data.map(x => this.transform(x))
  }

  /**
   * Applies transformation on an JavaRDD[Vector].
   *
   * @param data JavaRDD[Vector] to be transformed.
   * @return transformed JavaRDD[Vector].
   */
  def transform(data: JavaRDD[Vector]): JavaRDD[Vector] = {
    transform(data.rdd)
  }

}
