/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.NoOffsetForPartitionException;
import org.apache.kafka.clients.consumer.OffsetOutOfRangeException;
import org.apache.kafka.clients.consumer.OffsetResetStrategy;
import org.apache.kafka.clients.consumer.internals.ConsumerNetworkClient;
import org.apache.kafka.clients.consumer.internals.RequestFuture;
import org.apache.kafka.clients.consumer.internals.RequestFutureAdapter;
import org.apache.kafka.clients.consumer.internals.RequestFutureListener;
import org.apache.kafka.clients.consumer.internals.StaleMetadataException;
import org.apache.kafka.clients.consumer.internals.SubscriptionState;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.InvalidMetadataException;
import org.apache.kafka.common.errors.InvalidTopicException;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Count;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Rate;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.LogEntry;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.requests.FetchRequest;
import org.apache.kafka.common.requests.FetchResponse;
import org.apache.kafka.common.requests.ListOffsetRequest;
import org.apache.kafka.common.requests.ListOffsetResponse;
import org.apache.kafka.common.requests.MetadataRequest;
import org.apache.kafka.common.requests.MetadataResponse;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Fetcher<K, V> {
    private static final Logger log = LoggerFactory.getLogger(Fetcher.class);
    private final ConsumerNetworkClient client;
    private final Time time;
    private final int minBytes;
    private final int maxWaitMs;
    private final int fetchSize;
    private final long retryBackoffMs;
    private final int maxPollRecords;
    private final boolean checkCrcs;
    private final Metadata metadata;
    private final FetchManagerMetrics sensors;
    private final SubscriptionState subscriptions;
    private final List<PartitionRecords<K, V>> records;
    private final Deserializer<K> keyDeserializer;
    private final Deserializer<V> valueDeserializer;
    private final Map<TopicPartition, Long> offsetOutOfRangePartitions;
    private final Set<String> unauthorizedTopics;
    private final Map<TopicPartition, Long> recordTooLargePartitions;

    public Fetcher(ConsumerNetworkClient client, int minBytes, int maxWaitMs, int fetchSize, int maxPollRecords, boolean checkCrcs, Deserializer<K> keyDeserializer, Deserializer<V> valueDeserializer, Metadata metadata, SubscriptionState subscriptions, Metrics metrics, String metricGrpPrefix, Time time, long retryBackoffMs) {
        this.time = time;
        this.client = client;
        this.metadata = metadata;
        this.subscriptions = subscriptions;
        this.minBytes = minBytes;
        this.maxWaitMs = maxWaitMs;
        this.fetchSize = fetchSize;
        this.maxPollRecords = maxPollRecords;
        this.checkCrcs = checkCrcs;
        this.keyDeserializer = keyDeserializer;
        this.valueDeserializer = valueDeserializer;
        this.records = new LinkedList<PartitionRecords<K, V>>();
        this.offsetOutOfRangePartitions = new HashMap<TopicPartition, Long>();
        this.unauthorizedTopics = new HashSet<String>();
        this.recordTooLargePartitions = new HashMap<TopicPartition, Long>();
        this.sensors = new FetchManagerMetrics(metrics, metricGrpPrefix);
        this.retryBackoffMs = retryBackoffMs;
    }

    public void sendFetches() {
        for (Map.Entry<Node, FetchRequest> fetchEntry : this.createFetchRequests().entrySet()) {
            final FetchRequest fetch2 = fetchEntry.getValue();
            this.client.send(fetchEntry.getKey(), ApiKeys.FETCH, fetch2).addListener(new RequestFutureListener<ClientResponse>(){

                @Override
                public void onSuccess(ClientResponse response) {
                    Fetcher.this.handleFetchResponse(response, fetch2);
                }

                @Override
                public void onFailure(RuntimeException e) {
                    log.debug("Fetch failed", (Throwable)e);
                }
            });
        }
    }

    public void updateFetchPositions(Set<TopicPartition> partitions) {
        for (TopicPartition tp : partitions) {
            if (!this.subscriptions.isAssigned(tp) || this.subscriptions.isFetchable(tp)) continue;
            if (this.subscriptions.isOffsetResetNeeded(tp)) {
                this.resetOffset(tp);
                continue;
            }
            if (this.subscriptions.committed(tp) == null) {
                this.subscriptions.needOffsetReset(tp);
                this.resetOffset(tp);
                continue;
            }
            long committed = this.subscriptions.committed(tp).offset();
            log.debug("Resetting offset for partition {} to the committed offset {}", (Object)tp, (Object)committed);
            this.subscriptions.seek(tp, committed);
        }
    }

    public Map<String, List<PartitionInfo>> getAllTopicMetadata(long timeout) {
        return this.getTopicMetadata(MetadataRequest.allTopics(), timeout);
    }

    public Map<String, List<PartitionInfo>> getTopicMetadata(MetadataRequest request, long timeout) {
        if (!request.isAllTopics() && request.topics().isEmpty()) {
            return Collections.emptyMap();
        }
        long start2 = this.time.milliseconds();
        long remaining = timeout;
        do {
            long elapsed;
            RequestFuture<ClientResponse> future = this.sendMetadataRequest(request);
            this.client.poll(future, remaining);
            if (future.failed() && !future.isRetriable()) {
                throw future.exception();
            }
            if (future.succeeded()) {
                MetadataResponse response = new MetadataResponse(future.value().responseBody());
                Cluster cluster = response.cluster();
                Set<String> unauthorizedTopics = cluster.unauthorizedTopics();
                if (!unauthorizedTopics.isEmpty()) {
                    throw new TopicAuthorizationException(unauthorizedTopics);
                }
                boolean shouldRetry = false;
                Map<String, Errors> errors = response.errors();
                if (!errors.isEmpty()) {
                    log.debug("Topic metadata fetch included errors: {}", errors);
                    for (Map.Entry<String, Errors> errorEntry : errors.entrySet()) {
                        String topic = errorEntry.getKey();
                        Errors error = errorEntry.getValue();
                        if (error == Errors.INVALID_TOPIC_EXCEPTION) {
                            throw new InvalidTopicException("Topic '" + topic + "' is invalid");
                        }
                        if (error == Errors.UNKNOWN_TOPIC_OR_PARTITION) continue;
                        if (error.exception() instanceof RetriableException) {
                            shouldRetry = true;
                            continue;
                        }
                        throw new KafkaException("Unexpected error fetching metadata for topic " + topic, error.exception());
                    }
                }
                if (!shouldRetry) {
                    HashMap<String, List<PartitionInfo>> topicsPartitionInfos = new HashMap<String, List<PartitionInfo>>();
                    for (String topic : cluster.topics()) {
                        topicsPartitionInfos.put(topic, cluster.availablePartitionsForTopic(topic));
                    }
                    return topicsPartitionInfos;
                }
            }
            if ((remaining = timeout - (elapsed = this.time.milliseconds() - start2)) <= 0L) continue;
            long backoff2 = Math.min(remaining, this.retryBackoffMs);
            this.time.sleep(backoff2);
            remaining -= backoff2;
        } while (remaining > 0L);
        throw new TimeoutException("Timeout expired while fetching topic metadata");
    }

    private RequestFuture<ClientResponse> sendMetadataRequest(MetadataRequest request) {
        Node node = this.client.leastLoadedNode();
        if (node == null) {
            return RequestFuture.noBrokersAvailable();
        }
        return this.client.send(node, ApiKeys.METADATA, request);
    }

    private void resetOffset(TopicPartition partition) {
        long timestamp2;
        OffsetResetStrategy strategy = this.subscriptions.resetStrategy(partition);
        if (strategy == OffsetResetStrategy.EARLIEST) {
            timestamp2 = -2L;
        } else if (strategy == OffsetResetStrategy.LATEST) {
            timestamp2 = -1L;
        } else {
            throw new NoOffsetForPartitionException(partition);
        }
        log.debug("Resetting offset for partition {} to {} offset.", (Object)partition, (Object)strategy.name().toLowerCase(Locale.ROOT));
        long offset = this.listOffset(partition, timestamp2);
        if (this.subscriptions.isAssigned(partition)) {
            this.subscriptions.seek(partition, offset);
        }
    }

    private long listOffset(TopicPartition partition, long timestamp2) {
        while (true) {
            RequestFuture<Long> future = this.sendListOffsetRequest(partition, timestamp2);
            this.client.poll(future);
            if (future.succeeded()) {
                return future.value();
            }
            if (!future.isRetriable()) {
                throw future.exception();
            }
            if (future.exception() instanceof InvalidMetadataException) {
                this.client.awaitMetadataUpdate();
                continue;
            }
            this.time.sleep(this.retryBackoffMs);
        }
    }

    private void throwIfOffsetOutOfRange() throws OffsetOutOfRangeException {
        HashMap<TopicPartition, Long> currentOutOfRangePartitions = new HashMap<TopicPartition, Long>();
        for (Map.Entry<TopicPartition, Long> entry2 : this.offsetOutOfRangePartitions.entrySet()) {
            if (!this.subscriptions.isFetchable(entry2.getKey())) {
                log.debug("Ignoring fetched records for {} since it is no longer fetchable", (Object)entry2.getKey());
                continue;
            }
            Long position = this.subscriptions.position(entry2.getKey());
            if (position == null || !entry2.getValue().equals(position)) continue;
            currentOutOfRangePartitions.put(entry2.getKey(), entry2.getValue());
        }
        this.offsetOutOfRangePartitions.clear();
        if (!currentOutOfRangePartitions.isEmpty()) {
            throw new OffsetOutOfRangeException(currentOutOfRangePartitions);
        }
    }

    private void throwIfUnauthorizedTopics() throws TopicAuthorizationException {
        if (!this.unauthorizedTopics.isEmpty()) {
            HashSet<String> topics = new HashSet<String>(this.unauthorizedTopics);
            this.unauthorizedTopics.clear();
            throw new TopicAuthorizationException(topics);
        }
    }

    private void throwIfRecordTooLarge() throws RecordTooLargeException {
        HashMap<TopicPartition, Long> copiedRecordTooLargePartitions = new HashMap<TopicPartition, Long>(this.recordTooLargePartitions);
        this.recordTooLargePartitions.clear();
        if (!copiedRecordTooLargePartitions.isEmpty()) {
            throw new RecordTooLargeException("There are some messages at [Partition=Offset]: " + copiedRecordTooLargePartitions + " whose size is larger than the fetch size " + this.fetchSize + " and hence cannot be ever returned." + " Increase the fetch size, or decrease the maximum message size the broker will allow.", copiedRecordTooLargePartitions);
        }
    }

    public Map<TopicPartition, List<ConsumerRecord<K, V>>> fetchedRecords() {
        PartitionRecords<K, V> part;
        if (this.subscriptions.partitionAssignmentNeeded()) {
            return Collections.emptyMap();
        }
        HashMap<TopicPartition, List<ConsumerRecord<K, V>>> drained = new HashMap<TopicPartition, List<ConsumerRecord<K, V>>>();
        this.throwIfOffsetOutOfRange();
        this.throwIfUnauthorizedTopics();
        this.throwIfRecordTooLarge();
        Iterator<PartitionRecords<K, V>> iterator = this.records.iterator();
        for (int maxRecords = this.maxPollRecords; iterator.hasNext() && maxRecords > 0; maxRecords -= this.append(drained, part, maxRecords)) {
            part = iterator.next();
            if (!((PartitionRecords)part).isConsumed()) continue;
            iterator.remove();
        }
        return drained;
    }

    private int append(Map<TopicPartition, List<ConsumerRecord<K, V>>> drained, PartitionRecords<K, V> part, int maxRecords) {
        if (!this.subscriptions.isAssigned(part.partition)) {
            log.debug("Not returning fetched records for partition {} since it is no longer assigned", (Object)part.partition);
        } else {
            long position = this.subscriptions.position(part.partition);
            if (!this.subscriptions.isFetchable(part.partition)) {
                log.debug("Not returning fetched records for assigned partition {} since it is no longer fetchable", (Object)part.partition);
            } else {
                if (part.fetchOffset == position) {
                    List partRecords = ((PartitionRecords)part).take(maxRecords);
                    long nextOffset = ((ConsumerRecord)partRecords.get(partRecords.size() - 1)).offset() + 1L;
                    log.trace("Returning fetched records at offset {} for assigned partition {} and update position to {}", new Object[]{position, part.partition, nextOffset});
                    List records = drained.get(part.partition);
                    if (records == null) {
                        records = partRecords;
                        drained.put(part.partition, records);
                    } else {
                        records.addAll(partRecords);
                    }
                    this.subscriptions.position(part.partition, nextOffset);
                    return partRecords.size();
                }
                log.debug("Ignoring fetched records for {} at offset {} since the current position is {}", new Object[]{part.partition, part.fetchOffset, position});
            }
        }
        ((PartitionRecords)part).discard();
        return 0;
    }

    private RequestFuture<Long> sendListOffsetRequest(final TopicPartition topicPartition, long timestamp2) {
        HashMap<TopicPartition, ListOffsetRequest.PartitionData> partitions = new HashMap<TopicPartition, ListOffsetRequest.PartitionData>(1);
        partitions.put(topicPartition, new ListOffsetRequest.PartitionData(timestamp2, 1));
        PartitionInfo info = this.metadata.fetch().partition(topicPartition);
        if (info == null) {
            this.metadata.add(topicPartition.topic());
            log.debug("Partition {} is unknown for fetching offset, wait for metadata refresh", (Object)topicPartition);
            return RequestFuture.staleMetadata();
        }
        if (info.leader() == null) {
            log.debug("Leader for partition {} unavailable for fetching offset, wait for metadata refresh", (Object)topicPartition);
            return RequestFuture.leaderNotAvailable();
        }
        Node node = info.leader();
        ListOffsetRequest request = new ListOffsetRequest(-1, partitions);
        return this.client.send(node, ApiKeys.LIST_OFFSETS, request).compose(new RequestFutureAdapter<ClientResponse, Long>(){

            @Override
            public void onSuccess(ClientResponse response, RequestFuture<Long> future) {
                Fetcher.this.handleListOffsetResponse(topicPartition, response, future);
            }
        });
    }

    private void handleListOffsetResponse(TopicPartition topicPartition, ClientResponse clientResponse, RequestFuture<Long> future) {
        ListOffsetResponse lor = new ListOffsetResponse(clientResponse.responseBody());
        short errorCode = lor.responseData().get((Object)topicPartition).errorCode;
        if (errorCode == Errors.NONE.code()) {
            List<Long> offsets = lor.responseData().get((Object)topicPartition).offsets;
            if (offsets.size() != 1) {
                throw new IllegalStateException("This should not happen.");
            }
            long offset = offsets.get(0);
            log.debug("Fetched offset {} for partition {}", (Object)offset, (Object)topicPartition);
            future.complete(offset);
        } else if (errorCode == Errors.NOT_LEADER_FOR_PARTITION.code() || errorCode == Errors.UNKNOWN_TOPIC_OR_PARTITION.code()) {
            log.debug("Attempt to fetch offsets for partition {} failed due to obsolete leadership information, retrying.", (Object)topicPartition);
            future.raise(Errors.forCode(errorCode));
        } else {
            log.warn("Attempt to fetch offsets for partition {} failed due to: {}", (Object)topicPartition, (Object)Errors.forCode(errorCode).message());
            future.raise(new StaleMetadataException());
        }
    }

    private Set<TopicPartition> fetchablePartitions() {
        Set<TopicPartition> fetchable = this.subscriptions.fetchablePartitions();
        if (this.records.isEmpty()) {
            return fetchable;
        }
        for (PartitionRecords<K, V> partitionRecords : this.records) {
            fetchable.remove(partitionRecords.partition);
        }
        return fetchable;
    }

    private Map<Node, FetchRequest> createFetchRequests() {
        Cluster cluster = this.metadata.fetch();
        HashMap fetchable = new HashMap();
        for (TopicPartition partition : this.fetchablePartitions()) {
            Node node = cluster.leaderFor(partition);
            if (node == null) {
                this.metadata.requestUpdate();
                continue;
            }
            if (this.client.pendingRequestCount(node) != 0) continue;
            HashMap<TopicPartition, FetchRequest.PartitionData> fetch2 = (HashMap<TopicPartition, FetchRequest.PartitionData>)fetchable.get(node);
            if (fetch2 == null) {
                fetch2 = new HashMap<TopicPartition, FetchRequest.PartitionData>();
                fetchable.put(node, fetch2);
            }
            long position = this.subscriptions.position(partition);
            fetch2.put(partition, new FetchRequest.PartitionData(position, this.fetchSize));
            log.trace("Added fetch request for partition {} at offset {}", (Object)partition, (Object)position);
        }
        HashMap<Node, FetchRequest> requests = new HashMap<Node, FetchRequest>();
        for (Map.Entry entry2 : fetchable.entrySet()) {
            Node node = (Node)entry2.getKey();
            FetchRequest fetch3 = new FetchRequest(this.maxWaitMs, this.minBytes, (Map)entry2.getValue());
            requests.put(node, fetch3);
        }
        return requests;
    }

    private void handleFetchResponse(ClientResponse resp, FetchRequest request) {
        int totalBytes = 0;
        int totalCount = 0;
        FetchResponse response = new FetchResponse(resp.responseBody());
        for (Map.Entry<TopicPartition, FetchResponse.PartitionData> entry2 : response.responseData().entrySet()) {
            long fetchOffset;
            TopicPartition tp = entry2.getKey();
            FetchResponse.PartitionData partition = entry2.getValue();
            if (!this.subscriptions.isFetchable(tp)) {
                log.debug("Ignoring fetched records for partition {} since it is no longer fetchable", (Object)tp);
                continue;
            }
            if (partition.errorCode == Errors.NONE.code()) {
                fetchOffset = request.fetchData().get((Object)tp).offset;
                Long position = this.subscriptions.position(tp);
                if (position == null || position != fetchOffset) {
                    log.debug("Discarding fetch response for partition {} since its offset {} does not match the expected offset {}", new Object[]{tp, fetchOffset, position});
                    continue;
                }
                int bytes = 0;
                ByteBuffer buffer = partition.recordSet;
                MemoryRecords records = MemoryRecords.readableRecords(buffer);
                ArrayList parsed = new ArrayList();
                boolean skippedRecords = false;
                for (LogEntry logEntry : records) {
                    if (logEntry.offset() >= position) {
                        parsed.add(this.parseRecord(tp, logEntry));
                        bytes += logEntry.size();
                        continue;
                    }
                    skippedRecords = true;
                }
                if (!parsed.isEmpty()) {
                    log.trace("Adding fetched record for partition {} with offset {} to buffered record list", (Object)tp, (Object)position);
                    ConsumerRecord record = (ConsumerRecord)parsed.get(parsed.size() - 1);
                    this.records.add(new PartitionRecords(fetchOffset, tp, parsed));
                    this.sensors.recordsFetchLag.record(partition.highWatermark - record.offset());
                } else if (buffer.limit() > 0 && !skippedRecords) {
                    this.recordTooLargePartitions.put(tp, fetchOffset);
                }
                this.sensors.recordTopicFetchMetrics(tp.topic(), bytes, parsed.size());
                totalBytes += bytes;
                totalCount += parsed.size();
                continue;
            }
            if (partition.errorCode == Errors.NOT_LEADER_FOR_PARTITION.code() || partition.errorCode == Errors.UNKNOWN_TOPIC_OR_PARTITION.code()) {
                this.metadata.requestUpdate();
                continue;
            }
            if (partition.errorCode == Errors.OFFSET_OUT_OF_RANGE.code()) {
                fetchOffset = request.fetchData().get((Object)tp).offset;
                if (this.subscriptions.hasDefaultOffsetResetPolicy()) {
                    this.subscriptions.needOffsetReset(tp);
                } else {
                    this.offsetOutOfRangePartitions.put(tp, fetchOffset);
                }
                log.info("Fetch offset {} is out of range, resetting offset", (Object)fetchOffset);
                continue;
            }
            if (partition.errorCode == Errors.TOPIC_AUTHORIZATION_FAILED.code()) {
                log.warn("Not authorized to read from topic {}.", (Object)tp.topic());
                this.unauthorizedTopics.add(tp.topic());
                continue;
            }
            if (partition.errorCode == Errors.UNKNOWN.code()) {
                log.warn("Unknown error fetching data for topic-partition {}", (Object)tp);
                continue;
            }
            throw new IllegalStateException("Unexpected error code " + partition.errorCode + " while fetching data");
        }
        this.sensors.bytesFetched.record(totalBytes);
        this.sensors.recordsFetched.record(totalCount);
        this.sensors.fetchThrottleTimeSensor.record(response.getThrottleTime());
        this.sensors.fetchLatency.record(resp.requestLatencyMs());
    }

    private ConsumerRecord<K, V> parseRecord(TopicPartition partition, LogEntry logEntry) {
        try {
            if (this.checkCrcs) {
                logEntry.record().ensureValid();
            }
            long offset = logEntry.offset();
            long timestamp2 = logEntry.record().timestamp();
            TimestampType timestampType2 = logEntry.record().timestampType();
            ByteBuffer keyBytes = logEntry.record().key();
            byte[] keyByteArray = keyBytes == null ? null : Utils.toArray(keyBytes);
            Object key = keyBytes == null ? null : (Object)this.keyDeserializer.deserialize(partition.topic(), keyByteArray);
            ByteBuffer valueBytes = logEntry.record().value();
            byte[] valueByteArray = valueBytes == null ? null : Utils.toArray(valueBytes);
            Object value2 = valueBytes == null ? null : (Object)this.valueDeserializer.deserialize(partition.topic(), valueByteArray);
            return new ConsumerRecord<Object, Object>(partition.topic(), partition.partition(), offset, timestamp2, timestampType2, logEntry.record().checksum(), keyByteArray == null ? -1 : keyByteArray.length, valueByteArray == null ? -1 : valueByteArray.length, key, value2);
        }
        catch (KafkaException e) {
            throw e;
        }
        catch (RuntimeException e) {
            throw new KafkaException("Error deserializing key/value for partition " + partition + " at offset " + logEntry.offset(), e);
        }
    }

    private class FetchManagerMetrics {
        public final Metrics metrics;
        public final String metricGrpName;
        public final Sensor bytesFetched;
        public final Sensor recordsFetched;
        public final Sensor fetchLatency;
        public final Sensor recordsFetchLag;
        public final Sensor fetchThrottleTimeSensor;

        public FetchManagerMetrics(Metrics metrics, String metricGrpPrefix) {
            this.metrics = metrics;
            this.metricGrpName = metricGrpPrefix + "-fetch-manager-metrics";
            this.bytesFetched = metrics.sensor("bytes-fetched");
            this.bytesFetched.add(metrics.metricName("fetch-size-avg", this.metricGrpName, "The average number of bytes fetched per request"), new Avg());
            this.bytesFetched.add(metrics.metricName("fetch-size-max", this.metricGrpName, "The maximum number of bytes fetched per request"), new Max());
            this.bytesFetched.add(metrics.metricName("bytes-consumed-rate", this.metricGrpName, "The average number of bytes consumed per second"), new Rate());
            this.recordsFetched = metrics.sensor("records-fetched");
            this.recordsFetched.add(metrics.metricName("records-per-request-avg", this.metricGrpName, "The average number of records in each request"), new Avg());
            this.recordsFetched.add(metrics.metricName("records-consumed-rate", this.metricGrpName, "The average number of records consumed per second"), new Rate());
            this.fetchLatency = metrics.sensor("fetch-latency");
            this.fetchLatency.add(metrics.metricName("fetch-latency-avg", this.metricGrpName, "The average time taken for a fetch request."), new Avg());
            this.fetchLatency.add(metrics.metricName("fetch-latency-max", this.metricGrpName, "The max time taken for any fetch request."), new Max());
            this.fetchLatency.add(metrics.metricName("fetch-rate", this.metricGrpName, "The number of fetch requests per second."), new Rate(new Count()));
            this.recordsFetchLag = metrics.sensor("records-lag");
            this.recordsFetchLag.add(metrics.metricName("records-lag-max", this.metricGrpName, "The maximum lag in terms of number of records for any partition in this window"), new Max());
            this.fetchThrottleTimeSensor = metrics.sensor("fetch-throttle-time");
            this.fetchThrottleTimeSensor.add(metrics.metricName("fetch-throttle-time-avg", this.metricGrpName, "The average throttle time in ms"), new Avg());
            this.fetchThrottleTimeSensor.add(metrics.metricName("fetch-throttle-time-max", this.metricGrpName, "The maximum throttle time in ms"), new Max());
        }

        public void recordTopicFetchMetrics(String topic, int bytes, int records) {
            HashMap<String, String> metricTags = new HashMap<String, String>();
            metricTags.put("topic", topic.replace(".", "_"));
            String name = "topic." + topic + ".bytes-fetched";
            Sensor bytesFetched = this.metrics.getSensor(name);
            if (bytesFetched == null) {
                bytesFetched = this.metrics.sensor(name);
                bytesFetched.add(this.metrics.metricName("fetch-size-avg", this.metricGrpName, "The average number of bytes fetched per request for topic " + topic, metricTags), new Avg());
                bytesFetched.add(this.metrics.metricName("fetch-size-max", this.metricGrpName, "The maximum number of bytes fetched per request for topic " + topic, metricTags), new Max());
                bytesFetched.add(this.metrics.metricName("bytes-consumed-rate", this.metricGrpName, "The average number of bytes consumed per second for topic " + topic, metricTags), new Rate());
            }
            bytesFetched.record(bytes);
            name = "topic." + topic + ".records-fetched";
            Sensor recordsFetched = this.metrics.getSensor(name);
            if (recordsFetched == null) {
                recordsFetched = this.metrics.sensor(name);
                recordsFetched.add(this.metrics.metricName("records-per-request-avg", this.metricGrpName, "The average number of records in each request for topic " + topic, metricTags), new Avg());
                recordsFetched.add(this.metrics.metricName("records-consumed-rate", this.metricGrpName, "The average number of records consumed per second for topic " + topic, metricTags), new Rate());
            }
            recordsFetched.record(records);
        }
    }

    private static class PartitionRecords<K, V> {
        public long fetchOffset;
        public TopicPartition partition;
        public List<ConsumerRecord<K, V>> records;

        public PartitionRecords(long fetchOffset, TopicPartition partition, List<ConsumerRecord<K, V>> records) {
            this.fetchOffset = fetchOffset;
            this.partition = partition;
            this.records = records;
        }

        private boolean isConsumed() {
            return this.records == null || this.records.isEmpty();
        }

        private void discard() {
            this.records = null;
        }

        private List<ConsumerRecord<K, V>> take(int n) {
            if (this.records == null) {
                return Collections.emptyList();
            }
            if (n >= this.records.size()) {
                List<ConsumerRecord<K, V>> res = this.records;
                this.records = null;
                return res;
            }
            ArrayList<ConsumerRecord<K, V>> res = new ArrayList<ConsumerRecord<K, V>>(n);
            Iterator<ConsumerRecord<K, V>> iterator = this.records.iterator();
            for (int i = 0; i < n; ++i) {
                res.add(iterator.next());
                iterator.remove();
            }
            if (iterator.hasNext()) {
                this.fetchOffset = iterator.next().offset();
            }
            return res;
        }
    }
}

