/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tephra.hbase.txprune;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableExistsException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HConnection;
import org.apache.hadoop.hbase.client.HConnectionManager;
import org.apache.hadoop.hbase.client.HTableInterface;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.tephra.hbase.coprocessor.TransactionProcessor;
import org.apache.tephra.hbase.txprune.DataJanitorState;
import org.apache.tephra.hbase.txprune.TimeRegions;
import org.apache.tephra.txprune.TransactionPruningPlugin;
import org.apache.tephra.util.TxUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HBaseTransactionPruningPlugin
implements TransactionPruningPlugin {
    public static final Logger LOG = LoggerFactory.getLogger(HBaseTransactionPruningPlugin.class);
    protected Configuration conf;
    protected HBaseAdmin hBaseAdmin;
    protected HConnection connection;
    protected DataJanitorState dataJanitorState;

    public void initialize(Configuration conf) throws IOException {
        this.conf = conf;
        this.hBaseAdmin = new HBaseAdmin(conf);
        this.connection = HConnectionManager.createConnection((Configuration)conf);
        final TableName stateTable = TableName.valueOf((String)conf.get("data.tx.prune.state.table", "tephra.state"));
        LOG.info("Initializing plugin with state table {}:{}", (Object)stateTable.getNamespaceAsString(), (Object)stateTable.getNameAsString());
        this.createPruneTable(stateTable);
        this.dataJanitorState = new DataJanitorState(new DataJanitorState.TableSupplier(){

            @Override
            public HTableInterface get() throws IOException {
                return HBaseTransactionPruningPlugin.this.connection.getTable(stateTable);
            }
        });
    }

    public long fetchPruneUpperBound(long time, long inactiveTransactionBound) throws IOException {
        LOG.debug("Fetching prune upper bound for time {} and inactive transaction bound {}", (Object)time, (Object)inactiveTransactionBound);
        if (time < 0L || inactiveTransactionBound < 0L) {
            return -1L;
        }
        SortedSet<byte[]> transactionalRegions = this.getTransactionalRegions();
        if (!transactionalRegions.isEmpty()) {
            LOG.debug("Saving {} transactional regions for time {}", (Object)transactionalRegions.size(), (Object)time);
            this.dataJanitorState.saveRegionsForTime(time, transactionalRegions);
            LOG.debug("Saving inactive transaction bound {} for time {}", (Object)inactiveTransactionBound, (Object)time);
            this.dataJanitorState.saveInactiveTransactionBoundForTime(time, inactiveTransactionBound);
        }
        return this.computePruneUpperBound(new TimeRegions(time, transactionalRegions));
    }

    public void pruneComplete(long time, long maxPrunedInvalid) throws IOException {
        LOG.debug("Prune complete for time {} and prune upper bound {}", (Object)time, (Object)maxPrunedInvalid);
        if (time < 0L || maxPrunedInvalid < 0L) {
            return;
        }
        TimeRegions regionsToExclude = this.dataJanitorState.getRegionsOnOrBeforeTime(time);
        if (regionsToExclude != null) {
            LOG.debug("Deleting prune upper bounds smaller than {} for stale regions", (Object)maxPrunedInvalid);
            this.dataJanitorState.deletePruneUpperBounds(maxPrunedInvalid, regionsToExclude.getRegions());
        } else {
            LOG.warn("Cannot find saved regions on or before time {}", (Object)time);
        }
        long pruneTime = TxUtils.getTimestamp((long)maxPrunedInvalid);
        LOG.debug("Deleting regions recorded before time {}", (Object)pruneTime);
        this.dataJanitorState.deleteAllRegionsOnOrBeforeTime(pruneTime);
        LOG.debug("Deleting inactive transaction bounds recorded on or before time {}", (Object)pruneTime);
        this.dataJanitorState.deleteInactiveTransactionBoundsOnOrBeforeTime(pruneTime);
        LOG.debug("Deleting empty regions recorded on or before time {}", (Object)pruneTime);
        this.dataJanitorState.deleteEmptyRegionsOnOrBeforeTime(pruneTime);
    }

    public void destroy() {
        LOG.info("Stopping plugin...");
        try {
            this.connection.close();
        }
        catch (IOException e) {
            LOG.error("Got exception while closing HConnection", (Throwable)e);
        }
        try {
            this.hBaseAdmin.close();
        }
        catch (IOException e) {
            LOG.error("Got exception while closing HBase admin", (Throwable)e);
        }
    }

    protected void createPruneTable(TableName stateTable) throws IOException {
        try {
            if (this.hBaseAdmin.tableExists(stateTable)) {
                LOG.debug("Not creating pruneStateTable {}:{} since it already exists.", (Object)stateTable.getNamespaceAsString(), (Object)stateTable.getNameAsString());
                return;
            }
            HTableDescriptor htd = new HTableDescriptor(stateTable);
            htd.addFamily(new HColumnDescriptor(DataJanitorState.FAMILY).setMaxVersions(1));
            this.hBaseAdmin.createTable(htd);
            LOG.info("Created pruneTable {}:{}", (Object)stateTable.getNamespaceAsString(), (Object)stateTable.getNameAsString());
        }
        catch (TableExistsException ex) {
            LOG.debug("Not creating pruneStateTable {}:{} since it already exists.", new Object[]{stateTable.getNamespaceAsString(), stateTable.getNameAsString(), ex});
        }
    }

    protected boolean isTransactionalTable(HTableDescriptor tableDescriptor) {
        return tableDescriptor.hasCoprocessor(TransactionProcessor.class.getName());
    }

    protected SortedSet<byte[]> getTransactionalRegions() throws IOException {
        TreeSet<byte[]> regions = new TreeSet<byte[]>(Bytes.BYTES_COMPARATOR);
        HTableDescriptor[] tableDescriptors = this.hBaseAdmin.listTables();
        LOG.debug("Got {} tables to process", (Object)(tableDescriptors == null ? 0 : tableDescriptors.length));
        if (tableDescriptors != null) {
            for (HTableDescriptor tableDescriptor : tableDescriptors) {
                if (this.isTransactionalTable(tableDescriptor)) {
                    List tableRegions = this.hBaseAdmin.getTableRegions(tableDescriptor.getTableName());
                    LOG.debug("Regions for table {}: {}", (Object)tableDescriptor.getTableName(), (Object)tableRegions);
                    if (tableRegions == null) continue;
                    for (HRegionInfo region : tableRegions) {
                        regions.add(region.getRegionName());
                    }
                    continue;
                }
                LOG.debug("{} is not a transactional table", (Object)tableDescriptor.getTableName());
            }
        }
        return regions;
    }

    private long computePruneUpperBound(TimeRegions timeRegions) throws IOException {
        Set<TableName> existingTables = this.getTableNamesForRegions(timeRegions.getRegions());
        LOG.debug("Tables for time {} = {}", (Object)timeRegions.getTime(), existingTables);
        do {
            LOG.debug("Computing prune upper bound for {}", (Object)timeRegions);
            SortedSet<byte[]> transactionalRegions = timeRegions.getRegions();
            long time = timeRegions.getTime();
            long inactiveTransactionBound = this.dataJanitorState.getInactiveTransactionBoundForTime(time);
            LOG.debug("Got inactive transaction bound {}", (Object)inactiveTransactionBound);
            if (inactiveTransactionBound == -1L) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Ignoring regions for time {} as no inactiveTransactionBound was found for that time, and hence the data must be incomplete", (Object)time);
                continue;
            }
            transactionalRegions = this.filterDeletedTableRegions(existingTables, transactionalRegions);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Transactional regions after removing the regions of non-existing tables = {}", (Object)Iterables.transform(transactionalRegions, TimeRegions.BYTE_ARR_TO_STRING_FN));
            }
            Map<byte[], Long> pruneUpperBoundRegions = this.dataJanitorState.getPruneUpperBoundForRegions(transactionalRegions);
            this.logPruneUpperBoundRegions(pruneUpperBoundRegions);
            pruneUpperBoundRegions = this.handleEmptyRegions(inactiveTransactionBound, transactionalRegions, pruneUpperBoundRegions);
            if (!transactionalRegions.isEmpty() && pruneUpperBoundRegions.size() == transactionalRegions.size()) {
                Long minPruneUpperBoundRegions = Collections.min(pruneUpperBoundRegions.values());
                long pruneUpperBound = Math.min(inactiveTransactionBound, minPruneUpperBoundRegions);
                LOG.debug("Found prune upper bound {} for time {}", (Object)pruneUpperBound, (Object)time);
                return pruneUpperBound;
            }
            if (LOG.isDebugEnabled()) {
                Sets.SetView difference = Sets.difference(transactionalRegions, pruneUpperBoundRegions.keySet());
                LOG.debug("Ignoring regions for time {} because the following regions did not record a pruneUpperBound: {}", (Object)time, (Object)Iterables.transform((Iterable)difference, TimeRegions.BYTE_ARR_TO_STRING_FN));
            }
            timeRegions = this.dataJanitorState.getRegionsOnOrBeforeTime(time - 1L);
        } while (timeRegions != null);
        return -1L;
    }

    private SortedSet<byte[]> filterDeletedTableRegions(final Set<TableName> existingTables, SortedSet<byte[]> transactionalRegions) {
        return Sets.filter(transactionalRegions, (Predicate)new Predicate<byte[]>(){

            public boolean apply(byte[] region) {
                return existingTables.contains(HRegionInfo.getTable((byte[])region));
            }
        });
    }

    private Set<TableName> getTableNamesForRegions(Set<byte[]> regions) {
        HashSet<TableName> tableNames = new HashSet<TableName>(regions.size());
        for (byte[] region : regions) {
            tableNames.add(HRegionInfo.getTable((byte[])region));
        }
        return tableNames;
    }

    private Map<byte[], Long> handleEmptyRegions(long inactiveTransactionBound, SortedSet<byte[]> transactionalRegions, Map<byte[], Long> pruneUpperBoundRegions) throws IOException {
        long inactiveTransactionBoundTime = TxUtils.getTimestamp((long)inactiveTransactionBound);
        SortedSet<byte[]> emptyRegions = this.dataJanitorState.getEmptyRegionsAfterTime(inactiveTransactionBoundTime, transactionalRegions);
        LOG.debug("Got empty transactional regions for inactive transaction bound time {}: {}", (Object)inactiveTransactionBoundTime, (Object)Iterables.transform(emptyRegions, TimeRegions.BYTE_ARR_TO_STRING_FN));
        TreeMap<byte[], Long> pubWithEmptyRegions = new TreeMap<byte[], Long>(Bytes.BYTES_COMPARATOR);
        pubWithEmptyRegions.putAll(pruneUpperBoundRegions);
        for (byte[] emptyRegion : emptyRegions) {
            if (pruneUpperBoundRegions.containsKey(emptyRegion)) continue;
            pubWithEmptyRegions.put(emptyRegion, inactiveTransactionBound);
        }
        return Collections.unmodifiableMap(pubWithEmptyRegions);
    }

    private void logPruneUpperBoundRegions(Map<byte[], Long> pruneUpperBoundRegions) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Got region - prune upper bound map: {}", (Object)Iterables.transform(pruneUpperBoundRegions.entrySet(), (Function)new Function<Map.Entry<byte[], Long>, Map.Entry<String, Long>>(){

                public Map.Entry<String, Long> apply(Map.Entry<byte[], Long> input) {
                    String regionName = (String)TimeRegions.BYTE_ARR_TO_STRING_FN.apply((Object)input.getKey());
                    return Maps.immutableEntry((Object)regionName, (Object)input.getValue());
                }
            }));
        }
    }
}

