/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.crypto.sodium;

import java.nio.charset.StandardCharsets;
import jnr.ffi.Pointer;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.crypto.sodium.Allocated;
import org.apache.tuweni.crypto.sodium.Sodium;
import org.apache.tuweni.crypto.sodium.SodiumException;
import org.jetbrains.annotations.Nullable;

public final class PasswordHash {
    public static Bytes hash(String password, int length, Salt salt) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.getBytes(StandardCharsets.UTF_8), length, salt));
    }

    public static Bytes hash(Bytes password, int length, Salt salt) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.toArrayUnsafe(), length, salt));
    }

    public static byte[] hash(byte[] password, int length, Salt salt) {
        return PasswordHash.hash(password, length, salt, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), Algorithm.recommended());
    }

    public static Bytes hash(String password, int length, Salt salt, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.getBytes(StandardCharsets.UTF_8), length, salt, algorithm));
    }

    public static Bytes hash(Bytes password, int length, Salt salt, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.toArrayUnsafe(), length, salt, algorithm));
    }

    public static byte[] hash(byte[] password, int length, Salt salt, Algorithm algorithm) {
        return PasswordHash.hash(password, length, salt, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    public static Bytes hashInteractive(String password, int length, Salt salt) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.getBytes(StandardCharsets.UTF_8), length, salt, Algorithm.recommended()));
    }

    public static Bytes hashInteractive(Bytes password, int length, Salt salt) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.toArrayUnsafe(), length, salt, Algorithm.recommended()));
    }

    public static byte[] hashInteractive(byte[] password, int length, Salt salt) {
        return PasswordHash.hash(password, length, salt, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), Algorithm.recommended());
    }

    public static Bytes hashInteractive(String password, int length, Salt salt, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.getBytes(StandardCharsets.UTF_8), length, salt, algorithm));
    }

    public static Bytes hashInteractive(Bytes password, int length, Salt salt, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.toArrayUnsafe(), length, salt, algorithm));
    }

    public static byte[] hashInteractive(byte[] password, int length, Salt salt, Algorithm algorithm) {
        return PasswordHash.hash(password, length, salt, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    public static Bytes hashSensitive(String password, int length, Salt salt) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.getBytes(StandardCharsets.UTF_8), length, salt, Algorithm.recommended()));
    }

    public static Bytes hashSensitive(Bytes password, int length, Salt salt) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.toArrayUnsafe(), length, salt, Algorithm.recommended()));
    }

    public static byte[] hashSensitive(byte[] password, int length, Salt salt) {
        return PasswordHash.hash(password, length, salt, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), Algorithm.recommended());
    }

    public static Bytes hashSensitive(String password, int length, Salt salt, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.getBytes(StandardCharsets.UTF_8), length, salt, algorithm));
    }

    public static Bytes hashSensitive(Bytes password, int length, Salt salt, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.toArrayUnsafe(), length, salt, algorithm));
    }

    public static byte[] hashSensitive(byte[] password, int length, Salt salt, Algorithm algorithm) {
        return PasswordHash.hash(password, length, salt, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    public static Bytes hash(String password, int length, Salt salt, long opsLimit, long memLimit, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.getBytes(StandardCharsets.UTF_8), length, salt, opsLimit, memLimit, algorithm));
    }

    public static Bytes hash(Bytes password, int length, Salt salt, long opsLimit, long memLimit, Algorithm algorithm) {
        return Bytes.wrap((byte[])PasswordHash.hash(password.toArrayUnsafe(), length, salt, opsLimit, memLimit, algorithm));
    }

    public static byte[] hash(byte[] password, int length, Salt salt, long opsLimit, long memLimit, Algorithm algorithm) {
        PasswordHash.assertHashLength(length);
        PasswordHash.assertOpsLimit(opsLimit);
        PasswordHash.assertMemLimit(memLimit);
        if (opsLimit < algorithm.minOps) {
            throw new IllegalArgumentException("opsLimit " + opsLimit + " too low for specified algorithm");
        }
        if (!algorithm.isSupported()) {
            throw new UnsupportedOperationException(algorithm.name() + " is not supported by the currently loaded sodium native library");
        }
        byte[] out = new byte[length];
        int rc = Sodium.crypto_pwhash(out, length, password, password.length, salt.value.pointer(), opsLimit, memLimit, algorithm.id);
        if (rc != 0) {
            throw new SodiumException("crypto_pwhash: failed with result " + rc);
        }
        return out;
    }

    public static int minHashLength() {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_12)) {
            return 16;
        }
        long len = Sodium.crypto_pwhash_bytes_min();
        if (len > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_pwhash_bytes_min: " + len + " is too large");
        }
        return (int)len;
    }

    public static int maxHashLength() {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_12)) {
            return Integer.MAX_VALUE;
        }
        long len = Sodium.crypto_pwhash_bytes_max();
        if (len > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }
        return (int)len;
    }

    private static void assertHashLength(int length) {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_12)) {
            if (length < 16) {
                throw new IllegalArgumentException("length out of range");
            }
            return;
        }
        if ((long)length < Sodium.crypto_pwhash_bytes_min() || (long)length > Sodium.crypto_pwhash_bytes_max()) {
            throw new IllegalArgumentException("length out of range");
        }
    }

    public static String hash(String password) {
        return PasswordHash.hash(password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit());
    }

    public static String hashInteractive(String password) {
        return PasswordHash.hash(password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit());
    }

    public static String hashSensitive(String password) {
        return PasswordHash.hash(password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit());
    }

    public static String hash(String password, long opsLimit, long memLimit) {
        int i;
        PasswordHash.assertOpsLimit(opsLimit);
        PasswordHash.assertMemLimit(memLimit);
        byte[] out = new byte[PasswordHash.hashStringLength()];
        byte[] pwBytes = password.getBytes(StandardCharsets.UTF_8);
        int rc = Sodium.crypto_pwhash_str(out, pwBytes, pwBytes.length, opsLimit, memLimit);
        if (rc != 0) {
            throw new SodiumException("crypto_pwhash_str: failed with result " + rc);
        }
        for (i = 0; i < out.length && out[i] != 0; ++i) {
        }
        return new String(out, 0, i, StandardCharsets.UTF_8);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean verify(String hash, String password) {
        int hashLength;
        byte[] hashBytes = hash.getBytes(StandardCharsets.UTF_8);
        if (hashBytes.length >= (hashLength = PasswordHash.hashStringLength())) {
            return false;
        }
        Pointer str = Sodium.malloc(hashLength);
        try {
            str.put(0L, hashBytes, 0, hashBytes.length);
            str.putByte((long)hashBytes.length, (byte)0);
            byte[] pwBytes = password.getBytes(StandardCharsets.UTF_8);
            boolean bl = Sodium.crypto_pwhash_str_verify(str, pwBytes, pwBytes.length) == 0;
            return bl;
        }
        finally {
            Sodium.sodium_free(str);
        }
    }

    private static void assertCheckRehashAvailable() {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_14)) {
            throw new UnsupportedOperationException("Sodium re-hash checking is not available (requires sodium native library version >= 10.0.14)");
        }
    }

    public static VerificationResult checkHash(String hash, String password) {
        return PasswordHash.checkHash(hash, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit());
    }

    public static VerificationResult checkHashForInteractive(String hash, String password) {
        return PasswordHash.checkHash(hash, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit());
    }

    public static VerificationResult checkHashForSensitive(String hash, String password) {
        return PasswordHash.checkHash(hash, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static VerificationResult checkHash(String hash, String password, long opsLimit, long memLimit) {
        PasswordHash.assertCheckRehashAvailable();
        PasswordHash.assertOpsLimit(opsLimit);
        PasswordHash.assertMemLimit(memLimit);
        byte[] hashBytes = hash.getBytes(StandardCharsets.UTF_8);
        int hashLength = PasswordHash.hashStringLength();
        if (hashBytes.length >= hashLength) {
            return VerificationResult.FAILED;
        }
        Pointer str = Sodium.malloc(hashLength);
        try {
            str.put(0L, hashBytes, 0, hashBytes.length);
            str.putByte((long)hashBytes.length, (byte)0);
            byte[] pwBytes = password.getBytes(StandardCharsets.UTF_8);
            if (Sodium.crypto_pwhash_str_verify(str, pwBytes, pwBytes.length) != 0) {
                VerificationResult verificationResult = VerificationResult.FAILED;
                return verificationResult;
            }
            int rc = Sodium.crypto_pwhash_str_needs_rehash(str, opsLimit, memLimit);
            if (rc < 0) {
                throw new SodiumException("crypto_pwhash_str_needs_rehash: failed with result " + rc);
            }
            VerificationResult verificationResult = rc == 0 ? VerificationResult.PASSED : VerificationResult.NEEDS_REHASH;
            return verificationResult;
        }
        finally {
            Sodium.sodium_free(str);
        }
    }

    public static boolean needsRehash(String hash) {
        return PasswordHash.needsRehash(hash, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit());
    }

    public static boolean needsRehashForInteractive(String hash) {
        return PasswordHash.needsRehash(hash, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit());
    }

    public static boolean needsRehashForSensitive(String hash) {
        return PasswordHash.needsRehash(hash, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean needsRehash(String hash, long opsLimit, long memLimit) {
        PasswordHash.assertCheckRehashAvailable();
        PasswordHash.assertOpsLimit(opsLimit);
        PasswordHash.assertMemLimit(memLimit);
        byte[] hashBytes = hash.getBytes(StandardCharsets.UTF_8);
        int hashLength = PasswordHash.hashStringLength();
        if (hashBytes.length >= hashLength) {
            throw new IllegalArgumentException("hash is too long");
        }
        Pointer str = Sodium.malloc(hashLength);
        try {
            str.put(0L, hashBytes, 0, hashBytes.length);
            str.putByte((long)hashBytes.length, (byte)0);
            int rc = Sodium.crypto_pwhash_str_needs_rehash(str, opsLimit, memLimit);
            if (rc < 0) {
                throw new SodiumException("crypto_pwhash_str_needs_rehash: failed with result " + rc);
            }
            boolean bl = rc != 0;
            return bl;
        }
        finally {
            Sodium.sodium_free(str);
        }
    }

    private static int hashStringLength() {
        long hashLength = Sodium.crypto_pwhash_strbytes();
        if (hashLength > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_pwhash_strbytes: " + hashLength + " is too large");
        }
        return (int)hashLength;
    }

    public static long minOpsLimit() {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_12)) {
            return 3L;
        }
        return Sodium.crypto_pwhash_opslimit_min();
    }

    public static long interactiveOpsLimit() {
        return Sodium.crypto_pwhash_opslimit_interactive();
    }

    public static long moderateOpsLimit() {
        return Sodium.crypto_pwhash_opslimit_moderate();
    }

    public static long sensitiveOpsLimit() {
        return Sodium.crypto_pwhash_opslimit_sensitive();
    }

    public static long maxOpsLimit() {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_12)) {
            return 0xFFFFFFFFL;
        }
        return Sodium.crypto_pwhash_opslimit_max();
    }

    private static void assertOpsLimit(long opsLimit) {
        if (opsLimit < PasswordHash.minOpsLimit() || opsLimit > PasswordHash.maxOpsLimit()) {
            throw new IllegalArgumentException("opsLimit out of range");
        }
    }

    public static long minMemLimit() {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_12)) {
            return 8192L;
        }
        return Sodium.crypto_pwhash_memlimit_min();
    }

    public static long interactiveMemLimit() {
        return Sodium.crypto_pwhash_memlimit_interactive();
    }

    public static long moderateMemLimit() {
        return Sodium.crypto_pwhash_memlimit_moderate();
    }

    public static long sensitiveMemLimit() {
        return Sodium.crypto_pwhash_memlimit_sensitive();
    }

    public static long maxMemLimit() {
        if (!Sodium.supportsVersion(Sodium.VERSION_10_0_12)) {
            return 4398046510080L;
        }
        return Sodium.crypto_pwhash_memlimit_max();
    }

    private static void assertMemLimit(long memLimit) {
        if (memLimit < PasswordHash.minMemLimit() || memLimit > PasswordHash.maxMemLimit()) {
            throw new IllegalArgumentException("memLimit out of range");
        }
    }

    public static final class Salt {
        final Allocated value;

        private Salt(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        public static Salt fromBytes(Bytes bytes) {
            return Salt.fromBytes(bytes.toArrayUnsafe());
        }

        public static Salt fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_pwhash_saltbytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_pwhash_saltbytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Salt::new);
        }

        public static int length() {
            long saltLength = Sodium.crypto_pwhash_saltbytes();
            if (saltLength > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_pwhash_saltbytes: " + saltLength + " is too large");
            }
            return (int)saltLength;
        }

        public static Salt random() {
            return Sodium.randomBytes(Salt.length(), Salt::new);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Salt)) {
                return false;
            }
            Salt other = (Salt)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }

    public static final class Algorithm {
        private static Algorithm ARGON2I13 = new Algorithm("argon2i13", 1, 3L, true);
        private static Algorithm ARGON2ID13 = new Algorithm("argon2id13", 2, 1L, Sodium.supportsVersion(Sodium.VERSION_10_0_13));
        private final String name;
        private final int id;
        private final long minOps;
        private final boolean supported;

        private Algorithm(String name, int id, long minOps, boolean supported) {
            this.name = name;
            this.id = id;
            this.minOps = minOps;
            this.supported = supported;
        }

        public static Algorithm recommended() {
            return ARGON2ID13.isSupported() ? ARGON2ID13 : ARGON2I13;
        }

        public static Algorithm argon2i13() {
            return ARGON2I13;
        }

        public static Algorithm argon2id13() {
            return ARGON2ID13;
        }

        @Nullable
        static Algorithm fromId(int id) {
            if (Algorithm.ARGON2ID13.id == id) {
                return ARGON2ID13;
            }
            if (Algorithm.ARGON2I13.id == id) {
                return ARGON2I13;
            }
            return null;
        }

        public String name() {
            return this.name;
        }

        int id() {
            return this.id;
        }

        public boolean isSupported() {
            return this.supported;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Algorithm)) {
                return false;
            }
            Algorithm other = (Algorithm)obj;
            return this.id == other.id;
        }

        public int hashCode() {
            return Integer.hashCode(this.id);
        }

        public String toString() {
            return this.name;
        }
    }

    public static enum VerificationResult {
        FAILED,
        PASSED,
        NEEDS_REHASH;


        public boolean passed() {
            return this != FAILED;
        }

        public boolean needsRehash() {
            return this == NEEDS_REHASH;
        }
    }
}

