/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.units.bigints;

import com.google.common.base.Preconditions;
import java.math.BigInteger;
import java.util.Arrays;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.bytes.Bytes48;
import org.apache.tuweni.bytes.MutableBytes;
import org.apache.tuweni.bytes.MutableBytes48;
import org.apache.tuweni.units.bigints.UInt384Value;

public final class UInt384
implements UInt384Value<UInt384> {
    private static final int MAX_CONSTANT = 64;
    private static final BigInteger BI_MAX_CONSTANT = BigInteger.valueOf(64L);
    private static UInt384[] CONSTANTS = new UInt384[65];
    public static final UInt384 MIN_VALUE;
    public static final UInt384 MAX_VALUE;
    public static final UInt384 ZERO;
    public static final UInt384 ONE;
    private static final int INTS_SIZE = 12;
    private static final long LONG_MASK = 0xFFFFFFFFL;
    private static final BigInteger P_2_384;
    private final int[] ints;

    public static UInt384 valueOf(long value) {
        Preconditions.checkArgument((value >= 0L ? 1 : 0) != 0, (Object)"Argument must be positive");
        if (value <= 64L) {
            return CONSTANTS[(int)value];
        }
        return new UInt384(value);
    }

    public static UInt384 valueOf(BigInteger value) {
        Preconditions.checkArgument((value.signum() >= 0 ? 1 : 0) != 0, (Object)"Argument must be positive");
        Preconditions.checkArgument((value.bitLength() <= 384 ? 1 : 0) != 0, (Object)"Argument is too large to represent a UInt384");
        if (value.compareTo(BI_MAX_CONSTANT) <= 0) {
            return CONSTANTS[value.intValue()];
        }
        int[] ints = new int[12];
        for (int i = 11; i >= 0; --i) {
            ints[i] = value.intValue();
            value = value.shiftRight(32);
        }
        return new UInt384(ints);
    }

    public static UInt384 fromBytes(Bytes bytes) {
        return new UInt384(Bytes48.leftPad((Bytes)bytes));
    }

    public static UInt384 fromHexString(String str) {
        return new UInt384(Bytes48.fromHexStringLenient((CharSequence)str));
    }

    private UInt384(Bytes48 bytes) {
        this.ints = new int[12];
        int i = 0;
        int j = 0;
        while (i < 12) {
            this.ints[i] = bytes.getInt(j);
            ++i;
            j += 4;
        }
    }

    private UInt384(long value) {
        this.ints = new int[12];
        this.ints[10] = (int)(value >>> 32 & 0xFFFFFFFFL);
        this.ints[11] = (int)(value & 0xFFFFFFFFL);
    }

    private UInt384(int[] ints) {
        this.ints = ints;
    }

    @Override
    public boolean isZero() {
        if (this == ZERO) {
            return true;
        }
        for (int i = 11; i >= 0; --i) {
            if (this.ints[i] == 0) continue;
            return false;
        }
        return true;
    }

    @Override
    public UInt384 add(UInt384 value) {
        if (value.isZero()) {
            return this;
        }
        if (this.isZero()) {
            return value;
        }
        int[] result = new int[12];
        boolean constant = true;
        long sum = ((long)this.ints[11] & 0xFFFFFFFFL) + ((long)value.ints[11] & 0xFFFFFFFFL);
        result[11] = (int)(sum & 0xFFFFFFFFL);
        if (result[11] < 0 || result[11] > 64) {
            constant = false;
        }
        for (int i = 10; i >= 0; --i) {
            sum = ((long)this.ints[i] & 0xFFFFFFFFL) + ((long)value.ints[i] & 0xFFFFFFFFL) + (sum >>> 32);
            result[i] = (int)(sum & 0xFFFFFFFFL);
            constant &= result[i] == 0;
        }
        if (constant) {
            return CONSTANTS[result[11]];
        }
        return new UInt384(result);
    }

    @Override
    public UInt384 add(long value) {
        if (value == 0L) {
            return this;
        }
        if (value > 0L && this.isZero()) {
            return UInt384.valueOf(value);
        }
        int[] result = new int[12];
        boolean constant = true;
        long sum = ((long)this.ints[11] & 0xFFFFFFFFL) + (value & 0xFFFFFFFFL);
        result[11] = (int)(sum & 0xFFFFFFFFL);
        if (result[11] < 0 || result[11] > 64) {
            constant = false;
        }
        sum = ((long)this.ints[10] & 0xFFFFFFFFL) + (value >>> 32) + (sum >>> 32);
        result[10] = (int)(sum & 0xFFFFFFFFL);
        constant &= result[10] == 0;
        long signExtent = value >> 63 & 0xFFFFFFFFL;
        for (int i = 9; i >= 0; --i) {
            sum = ((long)this.ints[i] & 0xFFFFFFFFL) + signExtent + (sum >>> 32);
            result[i] = (int)(sum & 0xFFFFFFFFL);
            constant &= result[i] == 0;
        }
        if (constant) {
            return CONSTANTS[result[11]];
        }
        return new UInt384(result);
    }

    @Override
    public UInt384 addMod(UInt384 value, UInt384 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("addMod with zero modulus");
        }
        return UInt384.valueOf(this.toBigInteger().add(value.toBigInteger()).mod(modulus.toBigInteger()));
    }

    @Override
    public UInt384 addMod(long value, UInt384 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("addMod with zero modulus");
        }
        return UInt384.valueOf(this.toBigInteger().add(BigInteger.valueOf(value)).mod(modulus.toBigInteger()));
    }

    @Override
    public UInt384 addMod(long value, long modulus) {
        if (modulus == 0L) {
            throw new ArithmeticException("addMod with zero modulus");
        }
        if (modulus < 0L) {
            throw new ArithmeticException("addMod unsigned with negative modulus");
        }
        return UInt384.valueOf(this.toBigInteger().add(BigInteger.valueOf(value)).mod(BigInteger.valueOf(modulus)));
    }

    @Override
    public UInt384 subtract(UInt384 value) {
        if (value.isZero()) {
            return this;
        }
        int[] result = new int[12];
        boolean constant = true;
        long sum = ((long)this.ints[11] & 0xFFFFFFFFL) + ((long)(~value.ints[11]) & 0xFFFFFFFFL) + 1L;
        result[11] = (int)(sum & 0xFFFFFFFFL);
        if (result[11] < 0 || result[11] > 64) {
            constant = false;
        }
        for (int i = 10; i >= 0; --i) {
            sum = ((long)this.ints[i] & 0xFFFFFFFFL) + ((long)(~value.ints[i]) & 0xFFFFFFFFL) + (sum >>> 32);
            result[i] = (int)(sum & 0xFFFFFFFFL);
            constant &= result[i] == 0;
        }
        if (constant) {
            return CONSTANTS[result[11]];
        }
        return new UInt384(result);
    }

    @Override
    public UInt384 subtract(long value) {
        return this.add(-value);
    }

    @Override
    public UInt384 multiply(UInt384 value) {
        if (this.isZero() || value.isZero()) {
            return ZERO;
        }
        if (value.equals(ONE)) {
            return this;
        }
        return UInt384.multiply(this.ints, value.ints);
    }

    private static UInt384 multiply(int[] x, int[] y) {
        int[] result = new int[24];
        long carry = 0L;
        int j = 11;
        int k = 23;
        while (j >= 0) {
            long product = ((long)y[j] & 0xFFFFFFFFL) * ((long)x[11] & 0xFFFFFFFFL) + carry;
            result[k] = (int)product;
            carry = product >>> 32;
            --j;
            --k;
        }
        result[11] = (int)carry;
        for (int i = 10; i >= 0; --i) {
            carry = 0L;
            int j2 = 11;
            int k2 = 12 + i;
            while (j2 >= 0) {
                long product = ((long)y[j2] & 0xFFFFFFFFL) * ((long)x[i] & 0xFFFFFFFFL) + ((long)result[k2] & 0xFFFFFFFFL) + carry;
                result[k2] = (int)product;
                carry = product >>> 32;
                --j2;
                --k2;
            }
            result[i] = (int)carry;
        }
        boolean constant = true;
        for (int i = 12; i < 22; ++i) {
            constant &= result[i] == 0;
        }
        if (constant && result[23] >= 0 && result[23] <= 64) {
            return CONSTANTS[result[23]];
        }
        return new UInt384(Arrays.copyOfRange(result, 12, 24));
    }

    @Override
    public UInt384 multiply(long value) {
        if (value == 0L || this.isZero()) {
            return ZERO;
        }
        if (value == 1L) {
            return this;
        }
        if (value < 0L) {
            throw new ArithmeticException("multiply unsigned by negative");
        }
        UInt384 other = new UInt384(value);
        return UInt384.multiply(this.ints, other.ints);
    }

    @Override
    public UInt384 multiplyMod(UInt384 value, UInt384 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("multiplyMod with zero modulus");
        }
        if (this.isZero() || value.isZero()) {
            return ZERO;
        }
        if (value.equals(ONE)) {
            return this.mod(modulus);
        }
        return UInt384.valueOf(this.toBigInteger().multiply(value.toBigInteger()).mod(modulus.toBigInteger()));
    }

    @Override
    public UInt384 multiplyMod(long value, UInt384 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("multiplyMod with zero modulus");
        }
        if (value == 0L || this.isZero()) {
            return ZERO;
        }
        if (value == 1L) {
            return this.mod(modulus);
        }
        if (value < 0L) {
            throw new ArithmeticException("multiplyMod unsigned by negative");
        }
        return UInt384.valueOf(this.toBigInteger().multiply(BigInteger.valueOf(value)).mod(modulus.toBigInteger()));
    }

    @Override
    public UInt384 multiplyMod(long value, long modulus) {
        if (modulus == 0L) {
            throw new ArithmeticException("multiplyMod with zero modulus");
        }
        if (modulus < 0L) {
            throw new ArithmeticException("multiplyMod unsigned with negative modulus");
        }
        if (value == 0L || this.isZero()) {
            return ZERO;
        }
        if (value == 1L) {
            return this.mod(modulus);
        }
        if (value < 0L) {
            throw new ArithmeticException("multiplyMod unsigned by negative");
        }
        return UInt384.valueOf(this.toBigInteger().multiply(BigInteger.valueOf(value)).mod(BigInteger.valueOf(modulus)));
    }

    @Override
    public UInt384 divide(UInt384 value) {
        if (value.isZero()) {
            throw new ArithmeticException("divide by zero");
        }
        if (value.equals(ONE)) {
            return this;
        }
        return UInt384.valueOf(this.toBigInteger().divide(value.toBigInteger()));
    }

    @Override
    public UInt384 divide(long value) {
        if (value == 0L) {
            throw new ArithmeticException("divide by zero");
        }
        if (value < 0L) {
            throw new ArithmeticException("divide unsigned by negative");
        }
        if (value == 1L) {
            return this;
        }
        if (UInt384.isPowerOf2(value)) {
            return this.shiftRight(UInt384.log2(value));
        }
        return UInt384.valueOf(this.toBigInteger().divide(BigInteger.valueOf(value)));
    }

    @Override
    public UInt384 pow(UInt384 exponent) {
        return UInt384.valueOf(this.toBigInteger().modPow(exponent.toBigInteger(), P_2_384));
    }

    @Override
    public UInt384 pow(long exponent) {
        return UInt384.valueOf(this.toBigInteger().modPow(BigInteger.valueOf(exponent), P_2_384));
    }

    @Override
    public UInt384 mod(UInt384 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("mod by zero");
        }
        return UInt384.valueOf(this.toBigInteger().mod(modulus.toBigInteger()));
    }

    @Override
    public UInt384 mod(long modulus) {
        if (modulus == 0L) {
            throw new ArithmeticException("mod by zero");
        }
        if (modulus < 0L) {
            throw new ArithmeticException("mod by negative");
        }
        if (UInt384.isPowerOf2(modulus)) {
            int log2 = UInt384.log2(modulus);
            int d = log2 / 32;
            int s = log2 % 32;
            assert (d == 0 || d == 1);
            int[] result = new int[12];
            result[11 - d] = this.ints[11 - d] & ~(-1 << s);
            if (d != 0) {
                result[11] = this.ints[11];
            }
            return new UInt384(result);
        }
        return UInt384.valueOf(this.toBigInteger().mod(BigInteger.valueOf(modulus)));
    }

    public UInt384 and(UInt384 value) {
        int[] result = new int[12];
        for (int i = 11; i >= 0; --i) {
            result[i] = this.ints[i] & value.ints[i];
        }
        return new UInt384(result);
    }

    public UInt384 and(Bytes48 bytes) {
        int[] result = new int[12];
        int i = 11;
        int j = 28;
        while (i >= 0) {
            int other = (bytes.get(j) & 0xFF) << 24;
            other |= (bytes.get(j + 1) & 0xFF) << 16;
            other |= (bytes.get(i + 2) & 0xFF) << 8;
            result[i] = this.ints[i] & (other |= bytes.get(i + 3) & 0xFF);
            --i;
            j -= 4;
        }
        return new UInt384(result);
    }

    public UInt384 or(UInt384 value) {
        int[] result = new int[12];
        for (int i = 11; i >= 0; --i) {
            result[i] = this.ints[i] | value.ints[i];
        }
        return new UInt384(result);
    }

    public UInt384 or(Bytes48 bytes) {
        int[] result = new int[12];
        int i = 11;
        int j = 28;
        while (i >= 0) {
            result[i] = this.ints[i] | (bytes.get(j) & 0xFF) << 24;
            int n = i;
            result[n] = result[n] | (bytes.get(j + 1) & 0xFF) << 16;
            int n2 = i;
            result[n2] = result[n2] | (bytes.get(j + 2) & 0xFF) << 8;
            int n3 = i--;
            result[n3] = result[n3] | bytes.get(j + 3) & 0xFF;
            j -= 4;
        }
        return new UInt384(result);
    }

    public UInt384 xor(UInt384 value) {
        int[] result = new int[12];
        for (int i = 11; i >= 0; --i) {
            result[i] = this.ints[i] ^ value.ints[i];
        }
        return new UInt384(result);
    }

    public UInt384 xor(Bytes48 bytes) {
        int[] result = new int[12];
        int i = 11;
        int j = 28;
        while (i >= 0) {
            result[i] = this.ints[i] ^ (bytes.get(j) & 0xFF) << 24;
            int n = i;
            result[n] = result[n] ^ (bytes.get(j + 1) & 0xFF) << 16;
            int n2 = i;
            result[n2] = result[n2] ^ (bytes.get(j + 2) & 0xFF) << 8;
            int n3 = i--;
            result[n3] = result[n3] ^ bytes.get(j + 3) & 0xFF;
            j -= 4;
        }
        return new UInt384(result);
    }

    public UInt384 not() {
        int[] result = new int[12];
        for (int i = 11; i >= 0; --i) {
            result[i] = ~this.ints[i];
        }
        return new UInt384(result);
    }

    public UInt384 shiftRight(int distance) {
        if (distance == 0) {
            return this;
        }
        if (distance >= 384) {
            return ZERO;
        }
        int[] result = new int[12];
        int d = distance / 32;
        int s = distance % 32;
        int resIdx = 12;
        if (s == 0) {
            int i = 12 - d;
            while (i > 0) {
                result[--resIdx] = this.ints[--i];
            }
        } else {
            for (int i = 11 - d; i >= 0; --i) {
                int leftSide = this.ints[i] >>> s;
                int rightSide = i == 0 ? 0 : this.ints[i - 1] << 32 - s;
                result[--resIdx] = leftSide | rightSide;
            }
        }
        return new UInt384(result);
    }

    public UInt384 shiftLeft(int distance) {
        if (distance == 0) {
            return this;
        }
        if (distance >= 384) {
            return ZERO;
        }
        int[] result = new int[12];
        int d = distance / 32;
        int s = distance % 32;
        int resIdx = 0;
        if (s == 0) {
            int i = d;
            while (i < 12) {
                result[resIdx++] = this.ints[i++];
            }
        } else {
            for (int i = d; i < 12; ++i) {
                int leftSide = this.ints[i] << s;
                int rightSide = i == 11 ? 0 : this.ints[i + 1] >>> 32 - s;
                result[resIdx++] = leftSide | rightSide;
            }
        }
        return new UInt384(result);
    }

    public boolean equals(Object object) {
        if (object == this) {
            return true;
        }
        if (!(object instanceof UInt384)) {
            return false;
        }
        UInt384 other = (UInt384)object;
        for (int i = 0; i < 12; ++i) {
            if (this.ints[i] == other.ints[i]) continue;
            return false;
        }
        return true;
    }

    public int hashCode() {
        int result = 1;
        for (int i = 0; i < 12; ++i) {
            result = 31 * result + this.ints[i];
        }
        return result;
    }

    @Override
    public int compareTo(UInt384 other) {
        for (int i = 0; i < 12; ++i) {
            int cmp = Long.compare((long)this.ints[i] & 0xFFFFFFFFL, (long)other.ints[i] & 0xFFFFFFFFL);
            if (cmp == 0) continue;
            return cmp;
        }
        return 0;
    }

    @Override
    public boolean fitsInt() {
        for (int i = 0; i < 11; ++i) {
            if (this.ints[i] == 0) continue;
            return false;
        }
        return this.ints[11] >= 0;
    }

    @Override
    public int intValue() {
        if (!this.fitsInt()) {
            throw new ArithmeticException("Value does not fit a 4 byte int");
        }
        return this.ints[11];
    }

    @Override
    public boolean fitsLong() {
        for (int i = 0; i < 10; ++i) {
            if (this.ints[i] == 0) continue;
            return false;
        }
        return this.ints[10] >= 0;
    }

    @Override
    public long toLong() {
        if (!this.fitsLong()) {
            throw new ArithmeticException("Value does not fit a 8 byte long");
        }
        return (long)this.ints[10] << 32 | (long)this.ints[11] & 0xFFFFFFFFL;
    }

    public String toString() {
        return this.toBigInteger().toString();
    }

    @Override
    public BigInteger toBigInteger() {
        byte[] mag = new byte[48];
        int j = 0;
        for (int i = 0; i < 12; ++i) {
            mag[j++] = (byte)(this.ints[i] >>> 24);
            mag[j++] = (byte)(this.ints[i] >>> 16 & 0xFF);
            mag[j++] = (byte)(this.ints[i] >>> 8 & 0xFF);
            mag[j++] = (byte)(this.ints[i] & 0xFF);
        }
        return new BigInteger(1, mag);
    }

    @Override
    public UInt384 toUInt384() {
        return this;
    }

    @Override
    public Bytes48 toBytes() {
        MutableBytes48 bytes = MutableBytes48.create();
        int i = 0;
        int j = 0;
        while (i < 12) {
            bytes.setInt(j, this.ints[i]);
            ++i;
            j += 4;
        }
        return bytes;
    }

    @Override
    public Bytes toMinimalBytes() {
        int i;
        for (i = 0; i < 12 && this.ints[i] == 0; ++i) {
        }
        if (i == 12) {
            return Bytes.EMPTY;
        }
        int firstIntBytes = 4 - Integer.numberOfLeadingZeros(this.ints[i]) / 8;
        int totalBytes = firstIntBytes + (12 - (i + 1)) * 4;
        MutableBytes bytes = MutableBytes.create((int)totalBytes);
        int j = 0;
        switch (firstIntBytes) {
            case 4: {
                bytes.set(j++, (byte)(this.ints[i] >>> 24));
            }
            case 3: {
                bytes.set(j++, (byte)(this.ints[i] >>> 16 & 0xFF));
            }
            case 2: {
                bytes.set(j++, (byte)(this.ints[i] >>> 8 & 0xFF));
            }
            case 1: {
                bytes.set(j++, (byte)(this.ints[i] & 0xFF));
            }
        }
        ++i;
        while (i < 12) {
            bytes.setInt(j, this.ints[i]);
            ++i;
            j += 4;
        }
        return bytes;
    }

    @Override
    public int numberOfLeadingZeros() {
        for (int i = 0; i < 12; ++i) {
            if (this.ints[i] == 0) continue;
            return i * 32 + Integer.numberOfLeadingZeros(this.ints[i]);
        }
        return 384;
    }

    @Override
    public int bitLength() {
        for (int i = 0; i < 12; ++i) {
            if (this.ints[i] == 0) continue;
            return 384 - i * 32 - Integer.numberOfLeadingZeros(this.ints[i]);
        }
        return 0;
    }

    private static boolean isPowerOf2(long n) {
        assert (n > 0L);
        return (n & n - 1L) == 0L;
    }

    private static int log2(long v) {
        assert (v > 0L);
        return 63 - Long.numberOfLeadingZeros(v);
    }

    static {
        UInt384.CONSTANTS[0] = new UInt384(Bytes48.ZERO);
        for (int i = 1; i <= 64; ++i) {
            UInt384.CONSTANTS[i] = new UInt384(i);
        }
        MIN_VALUE = UInt384.valueOf(0L);
        MAX_VALUE = new UInt384(Bytes48.ZERO.not());
        ZERO = UInt384.valueOf(0L);
        ONE = UInt384.valueOf(1L);
        P_2_384 = BigInteger.valueOf(2L).pow(384);
    }
}

