/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.service;

import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.inject.Inject;
import org.apache.commons.lang3.StringUtils;
import org.apache.zeppelin.common.Message;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.display.AngularObject;
import org.apache.zeppelin.display.AngularObjectRegistry;
import org.apache.zeppelin.interpreter.InterpreterResult;
import org.apache.zeppelin.interpreter.InterpreterSetting;
import org.apache.zeppelin.interpreter.thrift.InterpreterCompletion;
import org.apache.zeppelin.notebook.AuthorizationService;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.Notebook;
import org.apache.zeppelin.notebook.Paragraph;
import org.apache.zeppelin.notebook.exception.CorruptedNoteException;
import org.apache.zeppelin.notebook.exception.NotePathAlreadyExistsException;
import org.apache.zeppelin.notebook.repo.NotebookRepoWithVersionControl;
import org.apache.zeppelin.notebook.scheduler.SchedulerService;
import org.apache.zeppelin.rest.exception.BadRequestException;
import org.apache.zeppelin.rest.exception.ForbiddenException;
import org.apache.zeppelin.rest.exception.NoteNotFoundException;
import org.apache.zeppelin.rest.exception.ParagraphNotFoundException;
import org.apache.zeppelin.scheduler.Job;
import org.apache.zeppelin.service.ServiceCallback;
import org.apache.zeppelin.service.ServiceContext;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.bitbucket.cowwoc.diffmatchpatch.DiffMatchPatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NotebookService {
    private static final Logger LOGGER = LoggerFactory.getLogger(NotebookService.class);
    private static final DateTimeFormatter TRASH_CONFLICT_TIMESTAMP_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss").withZone(ZoneId.systemDefault());
    private final ZeppelinConfiguration zConf;
    private final Notebook notebook;
    private final AuthorizationService authorizationService;
    private final SchedulerService schedulerService;

    @Inject
    public NotebookService(Notebook notebook, AuthorizationService authorizationService, ZeppelinConfiguration zeppelinConfiguration, SchedulerService schedulerService) {
        this.notebook = notebook;
        this.authorizationService = authorizationService;
        this.zConf = zeppelinConfiguration;
        this.schedulerService = schedulerService;
    }

    public String getHomeNote(ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        String noteId = this.notebook.getConf().getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_NOTEBOOK_HOMESCREEN);
        if (StringUtils.isBlank((CharSequence)noteId)) {
            callback.onSuccess(null, context);
        } else {
            this.notebook.processNote(noteId, note -> {
                if (note != null && !this.checkPermission(noteId, Permission.READER, Message.OP.GET_HOME_NOTE, context, callback)) {
                    return null;
                }
                callback.onSuccess(note, context);
                return null;
            });
        }
        return noteId;
    }

    public <T> T getNote(String noteId, ServiceContext context, ServiceCallback<Note> callback, Notebook.NoteProcessor<T> noteProcessor) throws IOException {
        return this.getNote(noteId, false, context, callback, noteProcessor);
    }

    public <T> T getNote(String noteId, boolean reload, ServiceContext context, ServiceCallback<Note> callback, Notebook.NoteProcessor<T> noteProcessor) throws IOException {
        return (T)this.notebook.processNote(noteId, reload, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.READER, Message.OP.GET_NOTE, context, callback)) {
                return null;
            }
            Note newNote = note;
            if (note.isPersonalizedMode()) {
                newNote = note.getUserNote(context.getAutheInfo().getUser());
            }
            callback.onSuccess(newNote, context);
            if (noteProcessor == null) {
                return null;
            }
            return noteProcessor.process(newNote);
        });
    }

    public <T> T getNoteByPath(String notePath, boolean reload, ServiceContext context, ServiceCallback<Note> callback, Notebook.NoteProcessor<T> noteProcessor) throws IOException {
        String noteId = null;
        try {
            noteId = this.notebook.getNoteIdByPath(notePath);
        }
        catch (IOException e) {
            throw new NoteNotFoundException(notePath);
        }
        return this.getNote(noteId, reload, context, callback, noteProcessor);
    }

    public String createNote(String notePath, String defaultInterpreterGroup, boolean addingEmptyParagraph, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (defaultInterpreterGroup == null) {
            defaultInterpreterGroup = this.zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_INTERPRETER_GROUP_DEFAULT);
        }
        try {
            String noteId = this.notebook.createNote(this.normalizeNotePath(notePath), defaultInterpreterGroup, context.getAutheInfo(), false);
            this.notebook.processNote(noteId, note -> {
                if (addingEmptyParagraph) {
                    note.addNewParagraph(context.getAutheInfo());
                }
                this.notebook.saveNote(note, context.getAutheInfo());
                callback.onSuccess(note, context);
                return null;
            });
            return noteId;
        }
        catch (IOException e) {
            callback.onFailure(e, context);
            return null;
        }
    }

    String normalizeNotePath(String notePath) throws IOException {
        if (StringUtils.isBlank((CharSequence)notePath)) {
            notePath = "/Untitled Note";
        }
        if (!notePath.startsWith("/")) {
            notePath = "/" + notePath;
        }
        notePath = notePath.replace("\r", " ").replace("\n", " ");
        if ((notePath = URLDecoder.decode(notePath, StandardCharsets.UTF_8.toString())).endsWith("/")) {
            throw new IOException("Note name shouldn't end with '/'");
        }
        int pos = notePath.lastIndexOf("/");
        if (notePath.length() - pos > 255) {
            throw new IOException("Note name must be less than 255");
        }
        if (notePath.contains("..")) {
            throw new IOException("Note name can not contain '..'");
        }
        return notePath;
    }

    public void removeNote(String noteId, ServiceContext context, ServiceCallback<String> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.OWNER, Message.OP.DEL_NOTE, context, callback)) {
            return;
        }
        if (this.notebook.containsNoteById(noteId)) {
            try {
                this.notebook.removeNote(noteId, context.getAutheInfo());
                callback.onSuccess("Delete note successfully", context);
            }
            catch (CorruptedNoteException e) {
                this.notebook.removeCorruptedNote(noteId, context.getAutheInfo());
                callback.onSuccess("Delete note successfully", context);
            }
        } else {
            callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
        }
    }

    public List<NoteInfo> listNotesInfo(boolean needsReload, ServiceContext context, ServiceCallback<List<NoteInfo>> callback) throws IOException {
        if (needsReload) {
            try {
                this.notebook.reloadAllNotes(context.getAutheInfo());
            }
            catch (IOException e) {
                LOGGER.error("Fail to reload notes from repository", (Throwable)e);
            }
        }
        List notesInfo = this.notebook.getNotesInfo(noteId -> this.authorizationService.isReader(noteId, context.getUserAndRoles()));
        callback.onSuccess(notesInfo, context);
        return notesInfo;
    }

    public void renameNote(String noteId, String newNotePath, boolean isRelative, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.OWNER, Message.OP.NOTE_RENAME, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, readNote -> {
            if (readNote == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
            }
            readNote.setCronSupported(this.notebook.getConf());
            String newNotePathReal = newNotePath;
            if (isRelative && !readNote.getParentPath().equals("/")) {
                newNotePathReal = readNote.getParentPath() + "/" + newNotePath;
            } else if (!newNotePath.startsWith("/")) {
                newNotePathReal = "/" + newNotePath;
            }
            try {
                this.notebook.moveNote(noteId, newNotePathReal, context.getAutheInfo());
                callback.onSuccess(readNote, context);
            }
            catch (NotePathAlreadyExistsException e) {
                callback.onFailure((Exception)((Object)e), context);
            }
            return null;
        });
    }

    public String cloneNote(String noteId, String newNotePath, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        return this.cloneNote(noteId, "", newNotePath, context, callback);
    }

    public String cloneNote(String noteId, String revisionId, String newNotePath, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (StringUtils.isBlank((CharSequence)newNotePath)) {
            newNotePath = "/Cloned Note_" + noteId;
            if (StringUtils.isNotEmpty((CharSequence)revisionId)) {
                newNotePath = newNotePath + "_" + revisionId.substring(0, 7);
            }
        }
        try {
            String newNoteId = this.notebook.cloneNote(noteId, revisionId, this.normalizeNotePath(newNotePath), context.getAutheInfo());
            return (String)this.notebook.processNote(newNoteId, newNote -> {
                callback.onSuccess(newNote, context);
                return newNote.getId();
            });
        }
        catch (IOException e) {
            callback.onFailure(new IOException("Fail to clone note", e), context);
            return null;
        }
    }

    public String importNote(String notePath, String noteJson, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        try {
            String noteId = this.notebook.importNote(noteJson, notePath == null ? notePath : this.normalizeNotePath(notePath), context.getAutheInfo());
            return (String)this.notebook.processNote(noteId, note -> {
                callback.onSuccess(note, context);
                return note.getId();
            });
        }
        catch (IOException e) {
            callback.onFailure(new IOException("Fail to import note: " + e.getMessage(), e), context);
            return null;
        }
    }

    public boolean runParagraph(Note note, String paragraphId, String title, String text, Map<String, Object> params, Map<String, Object> config, String sessionId, boolean failIfDisabled, boolean blocking, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (note == null) {
            return false;
        }
        LOGGER.info("Start to run paragraph: {} of note: {}", (Object)paragraphId, (Object)note.getId());
        if (!this.checkPermission(note.getId(), Permission.RUNNER, Message.OP.RUN_PARAGRAPH, context, callback)) {
            return false;
        }
        Paragraph p = note.getParagraph(paragraphId);
        if (p == null) {
            callback.onFailure((Exception)((Object)new ParagraphNotFoundException(paragraphId)), context);
            return false;
        }
        if (failIfDisabled && !p.isEnabled()) {
            callback.onFailure(new IOException("paragraph is disabled."), context);
            return false;
        }
        p.setText(text);
        p.setTitle(title);
        p.setAuthenticationInfo(context.getAutheInfo());
        if (params != null && !params.isEmpty()) {
            p.settings.setParams(params);
        }
        if (config != null && !config.isEmpty()) {
            p.mergeConfig(config);
        }
        if (note.isPersonalizedMode()) {
            p = p.getUserParagraph(context.getAutheInfo().getUser());
            p.setText(text);
            p.setTitle(title);
            p.setAuthenticationInfo(context.getAutheInfo());
            if (params != null && !params.isEmpty()) {
                p.settings.setParams(params);
            }
            if (config != null && !config.isEmpty()) {
                p.mergeConfig(config);
            }
        }
        try {
            this.notebook.saveNote(note, context.getAutheInfo());
            note.run(p.getId(), sessionId, blocking, context.getAutheInfo().getUser());
            callback.onSuccess(p, context);
            return true;
        }
        catch (Exception ex) {
            LOGGER.error("Exception from run", (Throwable)ex);
            p.setReturn(new InterpreterResult(InterpreterResult.Code.ERROR, ex.getMessage()), (Throwable)ex);
            p.setStatus(Job.Status.ERROR);
            return false;
        }
    }

    public boolean runAllParagraphs(String noteId, List<Map<String, Object>> paragraphs, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.RUNNER, Message.OP.RUN_ALL_PARAGRAPHS, context, callback)) {
            return false;
        }
        return (Boolean)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return false;
            }
            if (paragraphs != null) {
                try {
                    note.setRunning(true);
                    for (Map raw : paragraphs) {
                        String paragraphId;
                        block15: {
                            paragraphId = (String)raw.get("id");
                            if (paragraphId == null) {
                                LOGGER.warn("No id found in paragraph json: {}", (Object)raw);
                                continue;
                            }
                            try {
                                String text = (String)raw.get("paragraph");
                                String title = (String)raw.get("title");
                                Map params = (Map)raw.get("params");
                                Map config = (Map)raw.get("config");
                                if (this.runParagraph(note, paragraphId, title, text, params, config, null, false, true, context, callback)) break block15;
                                Boolean bl = false;
                                return bl;
                            }
                            catch (Exception e) {
                                throw new IOException("Fail to run paragraph json: " + raw, e);
                                return true;
                            }
                        }
                        Paragraph p = note.getParagraph(paragraphId);
                        InterpreterResult result = p.getReturn();
                        if (result != null && result.code() == InterpreterResult.Code.ERROR) {
                            Boolean bl = false;
                            return bl;
                        }
                        if (p.getStatus() != Job.Status.ABORT && !p.isAborted()) continue;
                        Boolean bl = false;
                        return bl;
                    }
                }
                finally {
                    note.setRunning(false);
                }
            }
            try {
                note.runAll(context.getAutheInfo(), true, false, new HashMap());
                return true;
            }
            catch (Exception e) {
                LOGGER.warn("Fail to run note: {}", (Object)note.getName(), (Object)e);
                return false;
            }
        });
    }

    public void cancelParagraph(String noteId, String paragraphId, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.RUNNER, Message.OP.CANCEL_PARAGRAPH, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                throw new NoteNotFoundException(noteId);
            }
            Paragraph p = note.getParagraph(paragraphId);
            if (p == null) {
                throw new ParagraphNotFoundException(paragraphId);
            }
            p.abort();
            callback.onSuccess(p, context);
            return null;
        });
    }

    public void moveParagraph(String noteId, String paragraphId, int newIndex, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.MOVE_PARAGRAPH, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                throw new NoteNotFoundException(noteId);
            }
            if (note.getParagraph(paragraphId) == null) {
                throw new ParagraphNotFoundException(paragraphId);
            }
            if (newIndex >= note.getParagraphCount()) {
                callback.onFailure((Exception)((Object)new BadRequestException("newIndex " + newIndex + " is out of bounds")), context);
                return null;
            }
            note.moveParagraph(paragraphId, newIndex);
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(note.getParagraph(newIndex), context);
            return null;
        });
    }

    public void removeParagraph(String noteId, String paragraphId, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.PARAGRAPH_REMOVE, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                throw new NoteNotFoundException(noteId);
            }
            if (note.getParagraph(paragraphId) == null) {
                throw new ParagraphNotFoundException(paragraphId);
            }
            Paragraph p = note.removeParagraph(context.getAutheInfo().getUser(), paragraphId);
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(p, context);
            return null;
        });
    }

    public Paragraph insertParagraph(String noteId, int index, Map<String, Object> config, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.INSERT_PARAGRAPH, context, callback)) {
            return null;
        }
        return (Paragraph)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                throw new NoteNotFoundException(noteId);
            }
            Paragraph newPara = note.insertNewParagraph(index, context.getAutheInfo());
            newPara.mergeConfig(config);
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(newPara, context);
            return newPara;
        });
    }

    public void restoreNote(String noteId, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.RESTORE_NOTE, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!note.getPath().startsWith("/~Trash")) {
                callback.onFailure(new IOException("Can not restore this note " + note.getPath() + " as it is not in trash folder"), context);
                return null;
            }
            try {
                String destNotePath = note.getPath().replace("/~Trash", "");
                this.notebook.moveNote(noteId, destNotePath, context.getAutheInfo());
                callback.onSuccess(note, context);
            }
            catch (IOException e) {
                callback.onFailure(new IOException("Fail to restore note: " + noteId, e), context);
            }
            return null;
        });
    }

    public void restoreFolder(String folderPath, ServiceContext context, ServiceCallback<Void> callback) throws IOException {
        if (!folderPath.startsWith("/~Trash")) {
            callback.onFailure(new IOException("Can not restore this folder: " + folderPath + " as it is not in trash folder"), context);
            return;
        }
        try {
            String destFolderPath = folderPath.replace("/~Trash", "");
            this.notebook.moveFolder(folderPath, destFolderPath, context.getAutheInfo());
            callback.onSuccess(null, context);
        }
        catch (IOException e) {
            callback.onFailure(new IOException("Fail to restore folder: " + folderPath, e), context);
        }
    }

    public void restoreAll(ServiceContext context, ServiceCallback<Void> callback) throws IOException {
        try {
            this.notebook.restoreAll(context.getAutheInfo());
            callback.onSuccess(null, context);
        }
        catch (IOException e) {
            callback.onFailure(new IOException("Fail to restore all", e), context);
        }
    }

    public void updateParagraph(String noteId, String paragraphId, String title, String text, Map<String, Object> params, Map<String, Object> config, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.COMMIT_PARAGRAPH, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            Paragraph p = note.getParagraph(paragraphId);
            if (p == null) {
                callback.onFailure((Exception)((Object)new ParagraphNotFoundException(paragraphId)), context);
                return null;
            }
            p.settings.setParams(params);
            p.mergeConfig(config);
            p.setTitle(title);
            p.setText(text);
            if (note.isPersonalizedMode()) {
                p = p.getUserParagraph(context.getAutheInfo().getUser());
                p.settings.setParams(params);
                p.mergeConfig(config);
                p.setTitle(title);
                p.setText(text);
            }
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(p, context);
            return null;
        });
    }

    public String getNextSessionParagraphId(String noteId, int maxParagraph, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.PARAGRAPH_CLEAR_OUTPUT, context, callback)) {
            throw new IOException("No privilege to access this note");
        }
        return (String)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                throw new IOException("No such note");
            }
            NotebookService notebookService = this;
            synchronized (notebookService) {
                if (note.getParagraphCount() >= maxParagraph) {
                    boolean removed = false;
                    for (int i = 1; i < note.getParagraphCount(); ++i) {
                        if (!note.getParagraph(i).getStatus().isCompleted()) continue;
                        note.removeParagraph(context.getAutheInfo().getUser(), note.getParagraph(i).getId());
                        removed = true;
                        break;
                    }
                    if (!removed) {
                        throw new IOException("All the paragraphs are not completed, unable to find available paragraph");
                    }
                }
                return note.addNewParagraph(context.getAutheInfo()).getId();
            }
        });
    }

    public void clearParagraphOutput(String noteId, String paragraphId, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.PARAGRAPH_CLEAR_OUTPUT, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            Paragraph returnedParagraph;
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            Paragraph p = note.getParagraph(paragraphId);
            if (p == null) {
                callback.onFailure((Exception)((Object)new ParagraphNotFoundException(paragraphId)), context);
                return null;
            }
            if (note.isPersonalizedMode()) {
                returnedParagraph = note.clearPersonalizedParagraphOutput(paragraphId, context.getAutheInfo().getUser());
            } else {
                note.clearParagraphOutput(paragraphId);
                returnedParagraph = note.getParagraph(paragraphId);
            }
            callback.onSuccess(returnedParagraph, context);
            return null;
        });
    }

    public void clearAllParagraphOutput(String noteId, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.PARAGRAPH_CLEAR_ALL_OUTPUT, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            note.clearAllParagraphOutput();
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(note, context);
            return null;
        });
    }

    public void updateNote(String noteId, String name, Map<String, Object> config, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.NOTE_UPDATE, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.zConf.isZeppelinNotebookCronEnable()) {
                boolean hasCronSettings = false;
                if (config.get("cron") != null) {
                    LOGGER.warn("cron should be null when cron is disabled");
                    hasCronSettings = true;
                }
                if (config.get("cronExecutingUser") != null) {
                    LOGGER.warn("cronExecutingUser should be null when cron is disabled");
                    hasCronSettings = true;
                }
                if (config.get("cronExecutingRoles") != null) {
                    LOGGER.warn("cronExecutingRoles should be null when cron is disabled");
                    hasCronSettings = true;
                }
                if (hasCronSettings) {
                    callback.onFailure(new IllegalArgumentException("Wrong configs"), context);
                    return null;
                }
            } else {
                boolean cronUpdated;
                AuthenticationInfo requestingAuth = new AuthenticationInfo((String)config.get("cronExecutingUser"), (String)config.get("cronExecutingRoles"), null);
                String requestCronUser = requestingAuth.getUser();
                Set requestCronRoles = requestingAuth.getRoles();
                if (!this.authorizationService.hasRunPermission(Collections.singleton(requestCronUser), note.getId())) {
                    LOGGER.error("Wrong cronExecutingUser: {}", (Object)requestCronUser);
                    callback.onFailure(new IllegalArgumentException(requestCronUser), context);
                    return null;
                }
                if (!context.getUserAndRoles().contains(requestCronUser)) {
                    LOGGER.error("Wrong cronExecutingUser: {}", (Object)requestCronUser);
                    callback.onFailure(new IllegalArgumentException(requestCronUser), context);
                    return null;
                }
                if (!context.getUserAndRoles().containsAll(requestCronRoles)) {
                    LOGGER.error("Wrong cronExecutingRoles: {}", (Object)requestCronRoles);
                    callback.onFailure(new IllegalArgumentException(requestCronRoles.toString()), context);
                    return null;
                }
                if (!((Boolean)note.getConfig().get("isZeppelinNotebookCronEnable")).booleanValue() && config.get("cron") != null) {
                    config.remove("cron");
                }
                if (cronUpdated = this.isCronUpdated(config, note.getConfig())) {
                    this.schedulerService.refreshCron(note.getId());
                }
            }
            note.setName(name);
            note.setConfig(config);
            this.notebook.updateNote(note, context.getAutheInfo());
            callback.onSuccess(note, context);
            return null;
        });
    }

    private boolean isCronUpdated(Map<String, Object> configA, Map<String, Object> configB) {
        boolean cronUpdated = false;
        if (configA.get("cron") != null && configB.get("cron") != null && configA.get("cron").equals(configB.get("cron"))) {
            cronUpdated = true;
        } else if (configA.get("cron") != null || configB.get("cron") != null) {
            cronUpdated = true;
        }
        return cronUpdated;
    }

    public void saveNoteForms(String noteId, Map<String, Object> noteParams, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.SAVE_NOTE_FORMS, context, callback)) {
            return;
        }
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            note.setNoteParams(noteParams);
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(note, context);
            return null;
        });
    }

    public void removeNoteForms(String noteId, String formName, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.REMOVE_NOTE_FORMS, context, callback)) {
                return null;
            }
            note.getNoteForms().remove(formName);
            note.getNoteParams().remove(formName);
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(note, context);
            return null;
        });
    }

    public NotebookRepoWithVersionControl.Revision checkpointNote(String noteId, String commitMessage, ServiceContext context, ServiceCallback<NotebookRepoWithVersionControl.Revision> callback) throws IOException {
        NotebookRepoWithVersionControl.Revision revision = (NotebookRepoWithVersionControl.Revision)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.REMOVE_NOTE_FORMS, context, callback)) {
                return null;
            }
            return this.notebook.checkpointNote(noteId, note.getPath(), commitMessage, context.getAutheInfo());
        });
        callback.onSuccess(revision, context);
        return revision;
    }

    public List<NotebookRepoWithVersionControl.Revision> listRevisionHistory(String noteId, ServiceContext context, ServiceCallback<List<NotebookRepoWithVersionControl.Revision>> callback) throws IOException {
        List revisions = (List)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            return this.notebook.listRevisionHistory(noteId, note.getPath(), context.getAutheInfo());
        });
        callback.onSuccess(revisions, context);
        return revisions;
    }

    public void setNoteRevision(String noteId, String revisionId, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.SET_NOTE_REVISION, context, callback)) {
                return null;
            }
            try {
                Note resultNote = this.notebook.setNoteRevision(noteId, note.getPath(), revisionId, context.getAutheInfo());
                callback.onSuccess(resultNote, context);
            }
            catch (Exception e) {
                callback.onFailure(new IOException("Fail to set given note revision", e), context);
            }
            return null;
        });
    }

    public Note getNotebyRevision(String noteId, String revisionId, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        return (Note)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.READER, Message.OP.NOTE_REVISION, context, callback)) {
                return null;
            }
            Note revisionNote = this.notebook.getNoteByRevision(noteId, note.getPath(), revisionId, context.getAutheInfo());
            callback.onSuccess(revisionNote, context);
            return revisionNote;
        });
    }

    public void getNoteByRevisionForCompare(String noteId, String revisionId, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.READER, Message.OP.NOTE_REVISION_FOR_COMPARE, context, callback)) {
                return null;
            }
            Note revisionNote = revisionId.equals("Head") ? note : this.notebook.getNoteByRevision(noteId, note.getPath(), revisionId, context.getAutheInfo());
            callback.onSuccess(revisionNote, context);
            return null;
        });
    }

    public List<InterpreterCompletion> completion(String noteId, String paragraphId, String buffer, int cursor, ServiceContext context, ServiceCallback<List<InterpreterCompletion>> callback) throws IOException {
        return (List)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.COMPLETION, context, callback)) {
                return null;
            }
            try {
                List completions = note.completion(paragraphId, buffer, cursor, context.getAutheInfo());
                callback.onSuccess(completions, context);
                return completions;
            }
            catch (RuntimeException e) {
                callback.onFailure(new IOException("Fail to get completion", e), context);
                return null;
            }
        });
    }

    public void getEditorSetting(String noteId, String paragraphText, ServiceContext context, ServiceCallback<Map<String, Object>> callback) throws IOException {
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
            }
            try {
                Map settings = this.notebook.getInterpreterSettingManager().getEditorSetting(paragraphText, noteId);
                callback.onSuccess(settings, context);
            }
            catch (Exception e) {
                callback.onFailure(new IOException("Fail to getEditorSetting", e), context);
            }
            return null;
        });
    }

    public void updatePersonalizedMode(String noteId, boolean isPersonalized, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        this.notebook.processNote(noteId, note -> {
            if (note == null) {
                callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                return null;
            }
            if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.UPDATE_PERSONALIZED_MODE, context, callback)) {
                return null;
            }
            note.setPersonalizedMode(Boolean.valueOf(isPersonalized));
            this.notebook.saveNote(note, context.getAutheInfo());
            callback.onSuccess(note, context);
            return null;
        });
    }

    public void moveNoteToTrash(String noteId, ServiceContext context, ServiceCallback<Note> callback) throws IOException {
        if (!this.checkPermission(noteId, Permission.OWNER, Message.OP.MOVE_NOTE_TO_TRASH, context, callback)) {
            return;
        }
        String destNotePath = "/~Trash" + (String)this.notebook.getNoteManager().getNotesInfo().get(noteId);
        if (this.notebook.containsNote(destNotePath)) {
            destNotePath = destNotePath + " " + TRASH_CONFLICT_TIMESTAMP_FORMATTER.format(Instant.now());
        }
        String finalDestNotePath = destNotePath;
        try {
            this.notebook.processNote(noteId, note -> {
                if (note == null) {
                    callback.onFailure((Exception)((Object)new NoteNotFoundException(noteId)), context);
                    return null;
                }
                this.notebook.moveNote(noteId, finalDestNotePath, context.getAutheInfo());
                callback.onSuccess(note, context);
                return null;
            });
        }
        catch (CorruptedNoteException e) {
            LOGGER.info("Move corrupted note to trash");
            this.notebook.moveNote(noteId, destNotePath, context.getAutheInfo());
        }
    }

    public void moveFolderToTrash(String folderPath, ServiceContext context, ServiceCallback<Void> callback) throws IOException {
        LOGGER.info("Move folder {} to trash", (Object)folderPath);
        String destFolderPath = "/~Trash/" + folderPath;
        if (this.notebook.containsNote(destFolderPath)) {
            destFolderPath = destFolderPath + " " + TRASH_CONFLICT_TIMESTAMP_FORMATTER.format(Instant.now());
        }
        this.notebook.moveFolder("/" + folderPath, destFolderPath, context.getAutheInfo());
        callback.onSuccess(null, context);
    }

    public void emptyTrash(ServiceContext context, ServiceCallback<Void> callback) throws IOException {
        try {
            this.notebook.emptyTrash(context.getAutheInfo());
            callback.onSuccess(null, context);
        }
        catch (IOException e) {
            callback.onFailure(e, context);
        }
    }

    public List<NoteInfo> removeFolder(String folderPath, ServiceContext context, ServiceCallback<List<NoteInfo>> callback) throws IOException {
        try {
            this.notebook.removeFolder(folderPath, context.getAutheInfo());
            List notesInfo = this.notebook.getNotesInfo(noteId -> this.authorizationService.isReader(noteId, context.getUserAndRoles()));
            callback.onSuccess(notesInfo, context);
            return notesInfo;
        }
        catch (IOException e) {
            callback.onFailure(e, context);
            return null;
        }
    }

    public List<NoteInfo> renameFolder(String folderPath, String newFolderPath, ServiceContext context, ServiceCallback<List<NoteInfo>> callback) throws IOException {
        try {
            this.notebook.moveFolder(this.normalizeNotePath(folderPath), this.normalizeNotePath(newFolderPath), context.getAutheInfo());
            List notesInfo = this.notebook.getNotesInfo(noteId -> this.authorizationService.isReader(noteId, context.getUserAndRoles()));
            callback.onSuccess(notesInfo, context);
            return notesInfo;
        }
        catch (IOException e) {
            callback.onFailure(e, context);
            return null;
        }
    }

    public void spell(String noteId, Message message, ServiceContext context, ServiceCallback<Paragraph> callback) throws IOException {
        try {
            if (!this.checkPermission(noteId, Permission.RUNNER, Message.OP.RUN_PARAGRAPH_USING_SPELL, context, callback)) {
                return;
            }
            String paragraphId = (String)message.get("id");
            if (paragraphId == null) {
                return;
            }
            String text = (String)message.get("paragraph");
            String title = (String)message.get("title");
            Job.Status status = Job.Status.valueOf((String)((String)message.get("status")));
            Map params = (Map)message.get("params");
            Map config = (Map)message.get("config");
            this.notebook.processNote(noteId, note -> {
                Paragraph p = this.setParagraphUsingMessage(note, message, paragraphId, text, title, params, config);
                p.setResult((InterpreterResult)message.get("results"));
                p.setErrorMessage((String)message.get("errorMessage"));
                p.setStatusWithoutNotification(status);
                String dateStarted = (String)message.get("dateStarted");
                String dateFinished = (String)message.get("dateFinished");
                SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSX");
                try {
                    p.setDateStarted(df.parse(dateStarted));
                }
                catch (ParseException e) {
                    LOGGER.error("Failed parse dateStarted", (Throwable)e);
                }
                try {
                    p.setDateFinished(df.parse(dateFinished));
                }
                catch (ParseException e) {
                    LOGGER.error("Failed parse dateFinished", (Throwable)e);
                }
                this.addNewParagraphIfLastParagraphIsExecuted(note, p);
                this.notebook.saveNote(note, context.getAutheInfo());
                callback.onSuccess(p, context);
                return null;
            });
        }
        catch (IOException e) {
            callback.onFailure(new IOException("Fail to run spell", e), context);
        }
    }

    private void addNewParagraphIfLastParagraphIsExecuted(Note note, Paragraph p) {
        boolean isTheLastParagraph = note.isLastParagraph(p.getId());
        if (!StringUtils.isEmpty((CharSequence)p.getText()) && !StringUtils.isEmpty((CharSequence)p.getScriptText()) && isTheLastParagraph) {
            note.addNewParagraph(p.getAuthenticationInfo());
        }
    }

    private Paragraph setParagraphUsingMessage(Note note, Message fromMessage, String paragraphId, String text, String title, Map<String, Object> params, Map<String, Object> config) {
        Paragraph p = note.getParagraph(paragraphId);
        p.setText(text);
        p.setTitle(title);
        AuthenticationInfo subject = new AuthenticationInfo(fromMessage.principal, fromMessage.roles, fromMessage.ticket);
        p.setAuthenticationInfo(subject);
        p.settings.setParams(params);
        p.setConfig(config);
        if (note.isPersonalizedMode()) {
            p = note.getParagraph(paragraphId);
            p.setText(text);
            p.setTitle(title);
            p.setAuthenticationInfo(subject);
            p.settings.setParams(params);
            p.setConfig(config);
        }
        return p;
    }

    public void updateAngularObject(String noteId, String paragraphId, String interpreterGroupId, String varName, Object varValue, ServiceContext context, ServiceCallback<AngularObject> callback) throws IOException {
        String user = context.getAutheInfo().getUser();
        AngularObject ao = null;
        boolean global = false;
        List settings = (List)this.notebook.processNote(noteId, note -> {
            if (note == null) {
                return Collections.emptyList();
            }
            return note.getBindedInterpreterSettings(new ArrayList<String>(context.getUserAndRoles()));
        });
        for (InterpreterSetting setting : settings) {
            if (setting.getInterpreterGroup(user, noteId) == null || !interpreterGroupId.equals(setting.getInterpreterGroup(user, noteId).getId())) continue;
            AngularObjectRegistry angularObjectRegistry = setting.getInterpreterGroup(user, noteId).getAngularObjectRegistry();
            ao = angularObjectRegistry.get(varName, noteId, paragraphId);
            if (ao == null) {
                ao = angularObjectRegistry.get(varName, noteId, null);
                if (ao == null) {
                    ao = angularObjectRegistry.get(varName, null, null);
                    if (ao == null) {
                        LOGGER.warn("Object {} is not binded", (Object)varName);
                        break;
                    }
                    ao.set(varValue, false);
                    global = true;
                    break;
                }
                ao.set(varValue, false);
                global = false;
                break;
            }
            ao.set(varValue, false);
            global = false;
            break;
        }
        callback.onSuccess(ao, context);
    }

    public void patchParagraph(String noteId, String paragraphId, String patchText, ServiceContext context, ServiceCallback<String> callback) throws IOException {
        try {
            if (!this.checkPermission(noteId, Permission.WRITER, Message.OP.PATCH_PARAGRAPH, context, callback)) {
                return;
            }
            this.notebook.processNote(noteId, note -> {
                if (note == null) {
                    return null;
                }
                Paragraph p = note.getParagraph(paragraphId);
                if (p == null) {
                    return null;
                }
                DiffMatchPatch dmp = new DiffMatchPatch();
                LinkedList patches = null;
                try {
                    patches = (LinkedList)dmp.patchFromText(patchText);
                }
                catch (ClassCastException e) {
                    LOGGER.error("Failed to parse patches", (Throwable)e);
                }
                if (patches == null) {
                    return null;
                }
                String paragraphText = p.getText() == null ? "" : p.getText();
                paragraphText = (String)dmp.patchApply(patches, paragraphText)[0];
                p.setText(paragraphText);
                callback.onSuccess(patchText, context);
                return null;
            });
        }
        catch (IOException e) {
            callback.onFailure(new IOException("Fail to patch", e), context);
        }
    }

    private <T> boolean checkPermission(String noteId, Permission permission, Message.OP op, ServiceContext context, ServiceCallback<T> callback) throws IOException {
        boolean isAllowed = false;
        Set allowed = null;
        switch (permission) {
            case READER: {
                isAllowed = this.authorizationService.isReader(noteId, context.getUserAndRoles());
                allowed = this.authorizationService.getReaders(noteId);
                break;
            }
            case WRITER: {
                isAllowed = this.authorizationService.isWriter(noteId, context.getUserAndRoles());
                allowed = this.authorizationService.getWriters(noteId);
                break;
            }
            case RUNNER: {
                isAllowed = this.authorizationService.isRunner(noteId, context.getUserAndRoles());
                allowed = this.authorizationService.getRunners(noteId);
                break;
            }
            case OWNER: {
                isAllowed = this.authorizationService.isOwner(noteId, context.getUserAndRoles());
                allowed = this.authorizationService.getOwners(noteId);
            }
        }
        if (isAllowed) {
            return true;
        }
        String errorMsg = "Insufficient privileges to " + (Object)((Object)permission) + " note.\nAllowed users or roles: " + allowed + "\nBut the user " + context.getAutheInfo().getUser() + " belongs to: " + context.getUserAndRoles();
        callback.onFailure((Exception)((Object)new ForbiddenException(errorMsg)), context);
        return false;
    }

    static enum Permission {
        READER,
        WRITER,
        RUNNER,
        OWNER;

    }
}

