/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteManager;
import org.apache.zeppelin.notebook.exception.NotePathAlreadyExistsException;
import org.apache.zeppelin.notebook.repo.InMemoryNotebookRepo;
import org.apache.zeppelin.notebook.repo.NotebookRepo;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class NoteManagerTest {
    private NoteManager noteManager;
    private ZeppelinConfiguration conf;

    @BeforeEach
    public void setUp() throws IOException {
        this.conf = ZeppelinConfiguration.create();
        this.noteManager = new NoteManager((NotebookRepo)new InMemoryNotebookRepo(), this.conf);
    }

    @Test
    public void testNoteOperations() throws IOException {
        Assertions.assertEquals((int)0, (int)this.noteManager.getNotesInfo().size());
        Note note1 = this.createNote("/prod/my_note1");
        Note note2 = this.createNote("/dev/project_2/my_note2");
        Note note3 = this.createNote("/dev/project_3/my_note3");
        this.noteManager.saveNote(note1);
        this.noteManager.saveNote(note2);
        this.noteManager.saveNote(note3);
        Assertions.assertEquals((int)3, (int)this.noteManager.getNotesInfo().size());
        Assertions.assertEquals((Object)note1, (Object)this.noteManager.processNote(note1.getId(), n -> n));
        Assertions.assertEquals((Object)note2, (Object)this.noteManager.processNote(note2.getId(), n -> n));
        Assertions.assertEquals((Object)note3, (Object)this.noteManager.processNote(note3.getId(), n -> n));
        this.noteManager.moveNote(note1.getId(), "/dev/project_1/my_note1", AuthenticationInfo.ANONYMOUS);
        Assertions.assertEquals((int)3, (int)this.noteManager.getNotesInfo().size());
        Assertions.assertEquals((Object)"/dev/project_1/my_note1", (Object)((Note)this.noteManager.processNote(note1.getId(), n -> n)).getPath());
        this.noteManager.moveFolder("/dev", "/staging", AuthenticationInfo.ANONYMOUS);
        Map notesInfo = this.noteManager.getNotesInfo();
        Assertions.assertEquals((int)3, (int)notesInfo.size());
        Assertions.assertEquals((Object)"/staging/project_1/my_note1", notesInfo.get(note1.getId()));
        Assertions.assertEquals((Object)"/staging/project_2/my_note2", notesInfo.get(note2.getId()));
        Assertions.assertEquals((Object)"/staging/project_3/my_note3", notesInfo.get(note3.getId()));
        this.noteManager.removeNote(note1.getId(), AuthenticationInfo.ANONYMOUS);
        Assertions.assertEquals((int)2, (int)this.noteManager.getNotesInfo().size());
        this.noteManager.removeFolder("/staging", AuthenticationInfo.ANONYMOUS);
        notesInfo = this.noteManager.getNotesInfo();
        Assertions.assertEquals((int)0, (int)notesInfo.size());
    }

    @Test
    public void testAddNoteRejectsDuplicatePath() throws IOException {
        Assertions.assertThrows(NotePathAlreadyExistsException.class, () -> {
            Note note1 = this.createNote("/prod/note");
            Note note2 = this.createNote("/prod/note");
            this.noteManager.addNote(note1, AuthenticationInfo.ANONYMOUS);
            this.noteManager.addNote(note2, AuthenticationInfo.ANONYMOUS);
        }, (String)"Note '/prod/note' existed");
    }

    @Test
    public void testMoveNoteRejectsDuplicatePath() throws IOException {
        Assertions.assertThrows(NotePathAlreadyExistsException.class, () -> {
            Note note1 = this.createNote("/prod/note-1");
            Note note2 = this.createNote("/prod/note-2");
            this.noteManager.addNote(note1, AuthenticationInfo.ANONYMOUS);
            this.noteManager.addNote(note2, AuthenticationInfo.ANONYMOUS);
            this.noteManager.moveNote(note2.getId(), "/prod/note-1", AuthenticationInfo.ANONYMOUS);
        }, (String)"Note '/prod/note-1' existed");
    }

    private Note createNote(String notePath) {
        return new Note(notePath, "test", null, null, null, null, null);
    }

    @Test
    public void testLruCache() throws IOException {
        int cacheThreshold = this.conf.getNoteCacheThreshold();
        for (int i = 0; i < cacheThreshold; ++i) {
            Note note = this.createNote("/prod/note" + i);
            this.noteManager.addNote(note, AuthenticationInfo.ANONYMOUS);
        }
        Assertions.assertEquals((int)cacheThreshold, (int)this.noteManager.getCacheSize());
        Note noteNew = this.createNote("/prod/notenew");
        this.noteManager.addNote(noteNew, AuthenticationInfo.ANONYMOUS);
        Assertions.assertEquals((int)cacheThreshold, (int)this.noteManager.getCacheSize());
        for (int i = 0; i < cacheThreshold; ++i) {
            Note note = this.createNote("/prod/noteDirty" + i);
            note.getLock().readLock().lock();
            this.noteManager.addNote(note, AuthenticationInfo.ANONYMOUS);
        }
        Assertions.assertEquals((int)cacheThreshold, (int)this.noteManager.getCacheSize());
        Note noteNew2 = this.createNote("/prod/notenew2");
        noteNew2.getLock().readLock().lock();
        this.noteManager.addNote(noteNew2, AuthenticationInfo.ANONYMOUS);
        Assertions.assertEquals((int)(cacheThreshold + 1), (int)this.noteManager.getCacheSize());
        Assertions.assertTrue((boolean)this.noteManager.containsNote(noteNew2.getPath()));
        this.noteManager.removeNote(noteNew2.getId(), AuthenticationInfo.ANONYMOUS);
        Assertions.assertFalse((boolean)this.noteManager.containsNote(noteNew2.getPath()));
        Assertions.assertEquals((int)cacheThreshold, (int)this.noteManager.getCacheSize());
        Note noteNew3 = this.createNote("/prod/notenew3");
        this.noteManager.addNote(noteNew3, AuthenticationInfo.ANONYMOUS);
        Assertions.assertEquals((int)cacheThreshold, (int)this.noteManager.getCacheSize());
        Assertions.assertTrue((boolean)this.noteManager.containsNote(noteNew3.getPath()));
    }

    @Test
    public void testConcurrentOperation() throws Exception {
        int threshold = 10;
        int noteNum = 150;
        ConcurrentHashMap<Integer, String> notes = new ConcurrentHashMap<Integer, String>();
        ExecutorService threadPool = Executors.newFixedThreadPool(threshold);
        ConcurrentTaskSaveNote saveNote = new ConcurrentTaskSaveNote(threadPool, noteNum, notes, "/prod/note%s");
        saveNote.exec();
        ConcurrentTaskMoveNote moveNote = new ConcurrentTaskMoveNote(threadPool, noteNum, notes, "/dev/project_%s/my_note%s");
        moveNote.exec();
        ConcurrentTaskMoveFolder moveFolder = new ConcurrentTaskMoveFolder(threadPool, noteNum, notes, "/staging/note_%s/my_note%s");
        moveFolder.exec();
        ConcurrentTaskRemoveNote removeNote = new ConcurrentTaskRemoveNote(threadPool, noteNum, notes, null);
        removeNote.exec();
        threadPool.shutdown();
    }

    class ConcurrentTaskRemoveNote
    extends ConcurrentTask {
        public ConcurrentTaskRemoveNote(ExecutorService threadPool, int noteNum, Map<Integer, String> notes, String pathPattern) {
            super(threadPool, noteNum, notes, pathPattern);
        }

        @Override
        public void run(int index) throws IOException {
            NoteManagerTest.this.noteManager.removeNote((String)((ConcurrentTask)this).notes.get(index), AuthenticationInfo.ANONYMOUS);
            ((ConcurrentTask)this).notes.remove(index);
        }
    }

    class ConcurrentTaskMoveFolder
    extends ConcurrentTask {
        public ConcurrentTaskMoveFolder(ExecutorService threadPool, int noteNum, Map<Integer, String> notes, String pathPattern) {
            super(threadPool, noteNum, notes, pathPattern);
        }

        @Override
        public void run(int index) throws IOException {
            String curPath = "/dev/project_" + index;
            String tarPath = "/staging/note_" + index;
            NoteManagerTest.this.noteManager.moveFolder(curPath, tarPath, AuthenticationInfo.ANONYMOUS);
        }
    }

    class ConcurrentTaskMoveNote
    extends ConcurrentTask {
        public ConcurrentTaskMoveNote(ExecutorService threadPool, int noteNum, Map<Integer, String> notes, String pathPattern) {
            super(threadPool, noteNum, notes, pathPattern);
        }

        @Override
        public void run(int index) throws IOException {
            String tarPath = String.format(((ConcurrentTask)this).pathPattern, index, index);
            NoteManagerTest.this.noteManager.moveNote((String)((ConcurrentTask)this).notes.get(index), tarPath, AuthenticationInfo.ANONYMOUS);
        }
    }

    class ConcurrentTaskSaveNote
    extends ConcurrentTask {
        public ConcurrentTaskSaveNote(ExecutorService threadPool, int noteNum, Map<Integer, String> notes, String pathPattern) {
            super(threadPool, noteNum, notes, pathPattern);
        }

        @Override
        public void run(int index) throws IOException {
            String tarPath = String.format(((ConcurrentTask)this).pathPattern, index, index);
            Note note = NoteManagerTest.this.createNote(tarPath);
            NoteManagerTest.this.noteManager.saveNote(note);
            ((ConcurrentTask)this).notes.put(index, note.getId());
        }
    }

    abstract class ConcurrentTask {
        private ExecutorService threadPool;
        private int noteNum;
        private Map<Integer, String> notes;
        private String pathPattern;

        public ConcurrentTask(ExecutorService threadPool, int noteNum, Map<Integer, String> notes, String pathPattern) {
            this.threadPool = threadPool;
            this.noteNum = noteNum;
            this.notes = notes;
            this.pathPattern = pathPattern;
        }

        public abstract void run(int var1) throws IOException;

        public void exec() throws Exception {
            CountDownLatch latch = new CountDownLatch(this.noteNum);
            int i = 0;
            while (i < this.noteNum) {
                int index = i++;
                this.threadPool.execute(() -> {
                    try {
                        this.run(index);
                        latch.countDown();
                    }
                    catch (IOException e) {
                        e.printStackTrace();
                    }
                });
            }
            Assertions.assertTrue((boolean)latch.await(5L, TimeUnit.SECONDS));
            this.checkPathByPattern();
        }

        private void checkPathByPattern() throws IOException {
            Assertions.assertEquals((int)this.notes.size(), (int)NoteManagerTest.this.noteManager.getNotesInfo().size());
            if (this.notes.isEmpty()) {
                return;
            }
            for (Integer key : this.notes.keySet()) {
                String expectPath = String.format(this.pathPattern, key, key);
                Assertions.assertEquals((Object)expectPath, (Object)((Note)NoteManagerTest.this.noteManager.processNote(this.notes.get(key), n -> n)).getPath());
            }
        }
    }
}

