/**
 * Licensed to Apereo under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Apereo licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License.  You may obtain a
 * copy of the License at the following location:
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apereo.inspektr.audit.support;

import com.fasterxml.jackson.core.util.MinimalPrettyPrinter;
import org.apereo.inspektr.audit.AuditActionContext;
import org.apereo.inspektr.audit.AuditTrailManager;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Abstract AuditTrailManager that turns the AuditActionContext into a printable String.
 *
 * @author Scott Battaglia
 * @since 1.0.1
 */
public abstract class AbstractStringAuditTrailManager implements AuditTrailManager {
    /**
     * what format should the audit log entry use?
     */
    private AuditFormats auditFormat = AuditFormats.DEFAULT;

    /**
     * Use multi-line output by default
     **/
    private boolean useSingleLine = false;

    /**
     * Separator for single line log entries
     */
    private String entrySeparator = ",";

    private List<AuditableFields> auditableFields = new ArrayList<>();

    public void setUseSingleLine(final boolean useSingleLine) {
        this.useSingleLine = useSingleLine;
    }

    @Override
    public void setAuditFormat(final AuditFormats auditFormat) {
        this.auditFormat = auditFormat;
    }

    public void setAuditableFields(final List<AuditableFields> auditableFields) {
        this.auditableFields = auditableFields;
    }

    @Override
    public Set<? extends AuditActionContext> getAuditRecords(final Map<WhereClauseFields, Object> whereClause) {
        return new HashSet<>();
    }

    protected String getEntrySeparator() {
        return this.entrySeparator;
    }

    public void setEntrySeparator(final String separator) {
        this.entrySeparator = separator;
    }

    protected String toString(final AuditActionContext auditActionContext) {
        if (auditFormat == AuditFormats.JSON) {
            final var builder = new StringBuilder();

            try {
                if (this.useSingleLine) {
                    final var writer = this.MAPPER.writer(new MinimalPrettyPrinter());
                    builder.append(writer.writeValueAsString(getMappedAuditActionContext(auditActionContext)));
                } else {
                    builder.append(this.MAPPER.writerWithDefaultPrettyPrinter().writeValueAsString(getMappedAuditActionContext(auditActionContext)));
                    builder.append("\n");
                }
            } catch (final Exception e) {
                throw new IllegalArgumentException(e.getMessage(), e);
            }
            return builder.toString();
        }
        return getAuditLineString(auditActionContext);
    }

    protected String getAuditLineString(final AuditActionContext auditActionContext) {
        final var builder = new StringBuilder();

        if (!useSingleLine) {
            builder.append("Audit trail record BEGIN\n");
            builder.append("=============================================================\n");
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.WHEN)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getWhenActionWasPerformed());
                builder.append(getEntrySeparator());
            } else {
                builder.append("WHEN: ");
                builder.append(auditActionContext.getWhenActionWasPerformed());
                builder.append("\n");
            }
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.WHO)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getPrincipal());
                builder.append(getEntrySeparator());
            } else {
                builder.append("WHO: ");
                builder.append(auditActionContext.getPrincipal());
                builder.append("\n");
            }
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.WHAT)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getResourceOperatedUpon());
                builder.append(getEntrySeparator());
            } else {
                builder.append("WHAT: ");
                builder.append(auditActionContext.getResourceOperatedUpon());
                builder.append("\n");
            }
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.ACTION)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getActionPerformed());
                builder.append(getEntrySeparator());
            } else {
                builder.append("ACTION: ");
                builder.append(auditActionContext.getActionPerformed());
                builder.append("\n");
            }
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.APPLICATION)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getApplicationCode());
                builder.append(getEntrySeparator());
            } else {
                builder.append("APPLICATION: ");
                builder.append(auditActionContext.getApplicationCode());
                builder.append("\n");
            }
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.USER_AGENT)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getUserAgent());
                builder.append(getEntrySeparator());
            } else {
                builder.append("USER-AGENT: ");
                builder.append(auditActionContext.getUserAgent());
                builder.append("\n");
            }
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.CLIENT_IP)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getClientIpAddress());
                builder.append(getEntrySeparator());
            } else {
                builder.append("CLIENT IP ADDRESS: ");
                builder.append(auditActionContext.getClientIpAddress());
                builder.append("\n");
            }
        }

        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.SERVER_IP)) {
            if (useSingleLine) {
                builder.append(auditActionContext.getServerIpAddress());
                builder.append(getEntrySeparator());
            } else {
                builder.append("SERVER IP ADDRESS: ");
                builder.append(auditActionContext.getServerIpAddress());
                builder.append("\n");
            }
        }

        if (!useSingleLine) {
            builder.append("=============================================================");
            builder.append("\n\n");
        }

        return builder.toString();
    }

    protected Map getMappedAuditActionContext(final AuditActionContext auditActionContext) {
        var map = new LinkedHashMap<String, Object>();
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.WHO)) {
            map.put("who", auditActionContext.getPrincipal());
        }
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.WHAT)) {
            map.put("what", auditActionContext.getResourceOperatedUpon());
        }
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.ACTION)) {
            map.put("action", auditActionContext.getActionPerformed());
        }
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.APPLICATION)) {
            map.put("application", auditActionContext.getApplicationCode());
        }
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.WHEN)) {
            map.put("when", auditActionContext.getWhenActionWasPerformed().toString());
        }
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.CLIENT_IP)) {
            map.put("clientIpAddress", auditActionContext.getClientIpAddress());
        }
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.SERVER_IP)) {
            map.put("serverIpAddress", auditActionContext.getServerIpAddress());
        }
        if (auditableFields.isEmpty() || auditableFields.contains(AuditableFields.USER_AGENT)) {
            map.put("userAgent", auditActionContext.getUserAgent());
        }
        return map;
    }
}
