/**
 * Licensed to Apereo under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Apereo licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License.  You may obtain a
 * copy of the License at the following location:
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apereo.inspektr.common.web;

import jakarta.servlet.http.HttpServletRequest;

import java.io.Serial;
import java.io.Serializable;
import java.net.Inet4Address;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * Captures information from the HttpServletRequest to log later.
 *
 * @author Scott Battaglia
 * @since 1.0
 */
public class ClientInfo implements Serializable {
    @Serial
    private static final long serialVersionUID = 7492721606084356617L;

    /**
     * IP Address of the server (local).
     */
    private final String serverIpAddress;

    /**
     * IP Address of the client (Remote)
     */
    private final String clientIpAddress;

    private final String geoLocation;

    private final String userAgent;

    private final Map<String, String> headers = new HashMap<>();

    private final Locale locale;

    private ClientInfo() {
        this(null);
    }

    public ClientInfo(final HttpServletRequest request) {
        this(request, null, null, false);
    }

    public ClientInfo(final HttpServletRequest request,
                      final String alternateServerAddrHeaderName,
                      final String alternateLocalAddrHeaderName,
                      final boolean useServerHostAddress) {
        try {
            this.locale = request != null ? request.getLocale() : Locale.getDefault();
            if (request != null) {
                var headerNames = request.getHeaderNames();
                while (headerNames.hasMoreElements()) {
                    var headerName = headerNames.nextElement();
                    this.headers.put(headerName, request.getHeader(headerName));
                }
            }

            var serverIpAddress = request != null ? request.getLocalAddr() : null;
            var clientIpAddress = request != null ? request.getRemoteAddr() : null;

            if (request == null) {
                this.geoLocation = "unknown";
                this.userAgent = "unknown";
            } else {
                if (useServerHostAddress) {
                    serverIpAddress = Inet4Address.getLocalHost().getHostAddress();
                } else if (alternateServerAddrHeaderName != null && !alternateServerAddrHeaderName.isEmpty()) {
                    serverIpAddress = request.getHeader(alternateServerAddrHeaderName) != null
                        ? request.getHeader(alternateServerAddrHeaderName) : request.getLocalAddr();
                }

                if (alternateLocalAddrHeaderName != null && !alternateLocalAddrHeaderName.isEmpty()) {
                    clientIpAddress = request.getHeader(alternateLocalAddrHeaderName) != null ? request.getHeader
                        (alternateLocalAddrHeaderName) : request.getRemoteAddr();
                }
                var header = request.getHeader("user-agent");
                this.userAgent = header == null ? "unknown" : header;
                var geo = request.getParameter("geolocation");
                if (geo == null) {
                    geo = request.getHeader("geolocation");
                }
                this.geoLocation = geo == null ? "unknown" : geo;
            }

            this.serverIpAddress = serverIpAddress == null ? "unknown" : serverIpAddress;
            this.clientIpAddress = clientIpAddress == null ? "unknown" : clientIpAddress;

        } catch (final Exception e) {
            throw new RuntimeException(e);
        }
    }

    public String getServerIpAddress() {
        return this.serverIpAddress;
    }

    public String getClientIpAddress() {
        return this.clientIpAddress;
    }

    public String getGeoLocation() {
        return geoLocation;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public Map<String, String> getHeaders() {
        return Collections.unmodifiableMap(this.headers);
    }

    public Locale getLocale() {
        return locale;
    }

    public static ClientInfo empty() {
        return new ClientInfo();
    }
}
