/*
 * Copyright 2012 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.artificer.integration.artifactbuilder;

import org.artificer.common.ArtifactContent;
import org.oasis_open.docs.s_ramp.ns.s_ramp_v1.BaseArtifactType;

import java.io.IOException;
import java.util.Collection;

/**
 * This contract allows both built-in and custom integrations to:
 * 
 * 1.) Define properties and relationships on the primary artifact
 * 2.) Generate derived artifacts from the primary artifact
 * 3.) Define relationships between the derived artifacts and between the derived and primary artifacts.
 * 
 * The contract is used in two separate steps.  The first, buildArtifacts, is responsible for setting properties and
 * generating all derived artifacts.  After this step, all artifacts are persisted in the repository, but *without* any
 * relationships.  The second step creates all relationships, after which all artifacts are updated in the repository
 * and the relationships persisted.  This separation of duties is mainly necessary for batch uploads, where all
 * artifacts must first be generated before relationships can be defined between them.
 * 
 * One or more AritfactBuilders are associated with one {@link ArtifactBuilderProvider}.  ArtifactBuilder logic
 * tends to be much cleaner when the builder is able to be *stateful*.  Therefore, the provider creates builders on
 * demand, rather than singleton builder use.
 * 
 * @author Brett Meyer
 */
public interface ArtifactBuilder {

    /**
     * Build-up the primary artifact properties and generate all derived artifacts.
     * 
     * @param primaryArtifact
     * @param artifactContent
     * @return ArtifactBuilder
     * @throws IOException
     */
    public ArtifactBuilder buildArtifacts(BaseArtifactType primaryArtifact, ArtifactContent artifactContent)
            throws Exception;

    /**
     * Build-up the relationships within the primary and derived artifacts.  The {@link RelationshipContext} provides
     * access to the current repository, allowing artifact lookup.
     * 
     * @param context
     * @return ArtifactBuilder
     * @throws IOException
     */
    public ArtifactBuilder buildRelationships(RelationshipContext context) throws Exception;
    
    /**
     * Provides the collection of derived artifacts generated by this builder.
     * 
     * @return Collection<BaseArtifactType>
     */
    public Collection<BaseArtifactType> getDerivedArtifacts();
}
