/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.primitives;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Longs;
import com.google.common.primitives.ParseRequest;
import java.math.BigInteger;
import java.util.Comparator;
import javax.annotation.CheckReturnValue;

@Beta
@GwtCompatible
public final class UnsignedLongs {
    public static final long MAX_VALUE = -1L;
    private static final long[] maxValueDivs = new long[37];
    private static final int[] maxValueMods = new int[37];
    private static final int[] maxSafeDigits = new int[37];

    private UnsignedLongs() {
    }

    private static long flip(long a) {
        return a ^ Long.MIN_VALUE;
    }

    @CheckReturnValue
    public static int compare(long a, long b2) {
        return Longs.compare(UnsignedLongs.flip(a), UnsignedLongs.flip(b2));
    }

    @CheckReturnValue
    public static long min(long ... array) {
        Preconditions.checkArgument(array.length > 0);
        long min2 = UnsignedLongs.flip(array[0]);
        for (int i2 = 1; i2 < array.length; ++i2) {
            long next2 = UnsignedLongs.flip(array[i2]);
            if (next2 >= min2) continue;
            min2 = next2;
        }
        return UnsignedLongs.flip(min2);
    }

    @CheckReturnValue
    public static long max(long ... array) {
        Preconditions.checkArgument(array.length > 0);
        long max2 = UnsignedLongs.flip(array[0]);
        for (int i2 = 1; i2 < array.length; ++i2) {
            long next2 = UnsignedLongs.flip(array[i2]);
            if (next2 <= max2) continue;
            max2 = next2;
        }
        return UnsignedLongs.flip(max2);
    }

    @CheckReturnValue
    public static String join(String separator, long ... array) {
        Preconditions.checkNotNull(separator);
        if (array.length == 0) {
            return "";
        }
        StringBuilder builder = new StringBuilder(array.length * 5);
        builder.append(UnsignedLongs.toString(array[0]));
        for (int i2 = 1; i2 < array.length; ++i2) {
            builder.append(separator).append(UnsignedLongs.toString(array[i2]));
        }
        return builder.toString();
    }

    @CheckReturnValue
    public static Comparator<long[]> lexicographicalComparator() {
        return LexicographicalComparator.INSTANCE;
    }

    @CheckReturnValue
    public static long divide(long dividend, long divisor) {
        long quotient;
        if (divisor < 0L) {
            if (UnsignedLongs.compare(dividend, divisor) < 0) {
                return 0L;
            }
            return 1L;
        }
        if (dividend >= 0L) {
            return dividend / divisor;
        }
        long rem = dividend - (quotient = (dividend >>> 1) / divisor << 1) * divisor;
        return quotient + (long)(UnsignedLongs.compare(rem, divisor) >= 0 ? 1 : 0);
    }

    @CheckReturnValue
    public static long remainder(long dividend, long divisor) {
        long rem;
        if (divisor < 0L) {
            if (UnsignedLongs.compare(dividend, divisor) < 0) {
                return dividend;
            }
            return dividend - divisor;
        }
        if (dividend >= 0L) {
            return dividend % divisor;
        }
        long quotient = (dividend >>> 1) / divisor << 1;
        return rem - (UnsignedLongs.compare(rem = dividend - quotient * divisor, divisor) >= 0 ? divisor : 0L);
    }

    public static long parseUnsignedLong(String s2) {
        return UnsignedLongs.parseUnsignedLong(s2, 10);
    }

    public static long decode(String stringValue) {
        ParseRequest request = ParseRequest.fromString(stringValue);
        try {
            return UnsignedLongs.parseUnsignedLong(request.rawValue, request.radix);
        }
        catch (NumberFormatException e) {
            NumberFormatException decodeException = new NumberFormatException("Error parsing value: " + stringValue);
            decodeException.initCause(e);
            throw decodeException;
        }
    }

    public static long parseUnsignedLong(String s2, int radix) {
        Preconditions.checkNotNull(s2);
        if (s2.length() == 0) {
            throw new NumberFormatException("empty string");
        }
        if (radix < 2 || radix > 36) {
            throw new NumberFormatException("illegal radix: " + radix);
        }
        int max_safe_pos = maxSafeDigits[radix] - 1;
        long value2 = 0L;
        for (int pos2 = 0; pos2 < s2.length(); ++pos2) {
            int digit = Character.digit(s2.charAt(pos2), radix);
            if (digit == -1) {
                throw new NumberFormatException(s2);
            }
            if (pos2 > max_safe_pos && UnsignedLongs.overflowInParse(value2, digit, radix)) {
                throw new NumberFormatException("Too large for unsigned long: " + s2);
            }
            value2 = value2 * (long)radix + (long)digit;
        }
        return value2;
    }

    private static boolean overflowInParse(long current2, int digit, int radix) {
        if (current2 >= 0L) {
            if (current2 < maxValueDivs[radix]) {
                return false;
            }
            if (current2 > maxValueDivs[radix]) {
                return true;
            }
            return digit > maxValueMods[radix];
        }
        return true;
    }

    @CheckReturnValue
    public static String toString(long x) {
        return UnsignedLongs.toString(x, 10);
    }

    @CheckReturnValue
    public static String toString(long x, int radix) {
        Preconditions.checkArgument(radix >= 2 && radix <= 36, "radix (%s) must be between Character.MIN_RADIX and Character.MAX_RADIX", radix);
        if (x == 0L) {
            return "0";
        }
        char[] buf = new char[64];
        int i2 = buf.length;
        if (x < 0L) {
            long quotient = UnsignedLongs.divide(x, radix);
            long rem = x - quotient * (long)radix;
            buf[--i2] = Character.forDigit((int)rem, radix);
            x = quotient;
        }
        while (x > 0L) {
            buf[--i2] = Character.forDigit((int)(x % (long)radix), radix);
            x /= (long)radix;
        }
        return new String(buf, i2, buf.length - i2);
    }

    static {
        BigInteger overflow = new BigInteger("10000000000000000", 16);
        for (int i2 = 2; i2 <= 36; ++i2) {
            UnsignedLongs.maxValueDivs[i2] = UnsignedLongs.divide(-1L, i2);
            UnsignedLongs.maxValueMods[i2] = (int)UnsignedLongs.remainder(-1L, i2);
            UnsignedLongs.maxSafeDigits[i2] = overflow.toString(i2).length() - 1;
        }
    }

    static enum LexicographicalComparator implements Comparator<long[]>
    {
        INSTANCE;


        @Override
        public int compare(long[] left2, long[] right) {
            int minLength = Math.min(left2.length, right.length);
            for (int i2 = 0; i2 < minLength; ++i2) {
                if (left2[i2] == right[i2]) continue;
                return UnsignedLongs.compare(left2[i2], right[i2]);
            }
            return left2.length - right.length;
        }
    }
}

