/*
 * Copyright (c) 2010-2012 Sonatype, Inc. All rights reserved.
 *
 * This program is licensed to you under the Apache License Version 2.0,
 * and you may not use this file except in compliance with the Apache License Version 2.0.
 * You may obtain a copy of the Apache License Version 2.0 at http://www.apache.org/licenses/LICENSE-2.0.
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the Apache License Version 2.0 is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the Apache License Version 2.0 for the specific language governing permissions and limitations there under.
 */
package org.asynchttpclient.request.body.generator;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.asynchttpclient.request.body.Body;

/**
 * A {@link BodyGenerator} backed by a byte array.
 */
public final class ByteArrayBodyGenerator implements BodyGenerator {

    private final byte[] bytes;

    public ByteArrayBodyGenerator(byte[] bytes) {
        this.bytes = bytes;
    }

    protected final class ByteBody implements Body {
        private boolean eof = false;
        private int lastPosition = 0;

        public long getContentLength() {
            return bytes.length;
        }

        public State read(ByteBuffer byteBuffer) throws IOException {

            if (eof) {
                return State.Stop;
            }

            final int remaining = bytes.length - lastPosition;
            if (remaining <= byteBuffer.capacity()) {
                byteBuffer.put(bytes, lastPosition, remaining);
                eof = true;
            } else {
                byteBuffer.put(bytes, lastPosition, byteBuffer.capacity());
                lastPosition = lastPosition + byteBuffer.capacity();
            }
            return State.Continue;
        }

        public void close() throws IOException {
            lastPosition = 0;
            eof = false;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Body createBody() {
        return new ByteBody();
    }
}
