/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.atteo.evo.config;

import javax.validation.constraints.Pattern;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlID;

import org.atteo.evo.classindex.IndexSubclasses;
import org.atteo.evo.xmlmerge.CombineChildren;
import org.atteo.evo.xmlmerge.CombineSelf;

/**
 * Any configurable element from the configuration file.
 *
 * <p>
 * Configurable contains an 'id' attribute which can be referenced
 * from other places using standard ID/IDREF functionality of the XML schema.
 * </p>
 */
@IndexSubclasses
@XmlAccessorType(XmlAccessType.NONE)
public abstract class Configurable {
	/**
	 * Id by which this element can be referenced.
	 */
	@XmlID
	@XmlAttribute
	@Pattern(regexp = "\\w[\\w\\d]*")
	private String id;

	/**
	 * Describes the behavior of node combining when merging XML trees.
	 */
	@XmlAttribute(name = CombineSelf.ATTRIBUTE_NAME)
	private CombineSelf combineSelf = CombineSelf.MERGE;

	/**
	 * Describes the behavior of children combining when merging XML trees.
	 */
	@XmlAttribute(name = CombineChildren.ATTRIBUTE_NAME)
	private CombineChildren combineChildren = CombineChildren.MERGE;

	/**
	 * Returns an ID for this element associated in the XML file.
	 * @return an ID or null, if ID not assigned
	 */
	public String getId() {
		return id;
	}

	/**
	 * Returns the behavior of node combining when merging XML trees.
	 * @return node behavior
	 */
	public CombineSelf getCombineSelf() {
		return combineSelf;
	}

	/**
	 * Returns the behavior of children combining when merging XML trees.
	 * @return node behavior
	 */
	public CombineChildren getCombineChildren() {
		return combineChildren;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append(getClass().getSimpleName());
		if (id != null) {
			builder.append("[id=\"");
			builder.append(id);
			builder.append("\"]");
		}
		return builder.toString();
	}
}
