/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.liveness.database.probes;

import java.sql.Connection;
import java.sql.SQLException;

import jakarta.annotation.PostConstruct;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;

import org.springframework.boot.actuate.autoconfigure.health.ConditionalOnEnabledHealthIndicator;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.annotation.Bean;
import org.springframework.context.event.EventListener;
import org.springframework.jdbc.core.ConnectionCallback;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import org.autumnframework.service.liveness.probes.LivenessProbe;

/**
 * 
 */
@Component
@ConditionalOnEnabledHealthIndicator("db")
@ConditionalOnProperty(name = "autumn.liveness.db", havingValue = "true", matchIfMissing = true)
public class DataSourceLivenessEventListener extends LivenessProbe {

    private final JdbcTemplate jdbcTemplate;
    
    /**
     * @param appEventPublisher
     * @param jdbcTemplate
     */
    public DataSourceLivenessEventListener(ApplicationEventPublisher appEventPublisher, JdbcTemplate jdbcTemplate) {
        super(appEventPublisher);
        this.jdbcTemplate = jdbcTemplate;
    }
    
    /**
     * 
     */
    @PostConstruct 
    public void init() {
        this.setLiveness(this.isConnectionValid(), null);
    }
    
    @Bean
    ConnectionEventListener datasourceConnectionEventListener() {
        return new ConnectionEventListener() {
            @Override
            @EventListener
            public void connectionClosed(ConnectionEvent event) {
                setLiveness(true, event);
            }

            @Override
            @EventListener
            public void connectionErrorOccurred(ConnectionEvent event) {
                setLiveness(isConnectionValid(), event);
            }
        };
    }
    
    @Override
    public String getProbeTypeName() {
        return "Database connection";
    }
    
    private Boolean isConnectionValid() {
        return this.jdbcTemplate.execute((ConnectionCallback<Boolean>) this::isConnectionValid);
    }
    
    private Boolean isConnectionValid(Connection connection) throws SQLException {
        return connection.isValid(0);
    }
}
