/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.pubsub.server.beans;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.rpc.AlreadyExistsException;
import com.google.api.gax.rpc.TransportChannelProvider;
import com.google.cloud.pubsub.v1.Publisher;
import com.google.cloud.pubsub.v1.TopicAdminClient;
import com.google.cloud.pubsub.v1.TopicAdminSettings;
import com.google.pubsub.v1.TopicName;

import jakarta.annotation.PreDestroy;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.autumnframework.service.pubsub.api.properties.PubSubApiProperties;

/**
 * 
 */
@Slf4j
@Service
public class PubSubPublisherStore {
    
    private final CredentialsProvider credentialsProvider;
    private final TransportChannelProvider transportChannelProvider;
    private final PubSubApiProperties properties; 
    
    /**
     * @param credentialsProvider
     * @param transportChannelProvider
     * @param properties
     */
    public PubSubPublisherStore(CredentialsProvider credentialsProvider,
            @Qualifier("subscriberTransportChannelProvider") TransportChannelProvider transportChannelProvider,
            PubSubApiProperties properties) {
        this.credentialsProvider = credentialsProvider;
        this.transportChannelProvider = transportChannelProvider;
        this.properties = properties;
    }
    
    @Getter
    private static Map<String, Publisher> publishers = new HashMap<>();
    
    /**
     * @param topicName
     * @throws IOException
     */
    public void createTopic(String topicName) throws IOException {
        try (TopicAdminClient topicAdminClient = TopicAdminClient
                .create(getTopicAdminSettings(credentialsProvider, transportChannelProvider))) {
            try {
                log.debug("Creating topic '{}' in project '{}'", topicName, properties.getProjectId());
                topicAdminClient.createTopic(TopicName.of(properties.getProjectId(), topicName));
            } catch (AlreadyExistsException ignore) {
                // topic already exists
            }
        }
    }
    
    /**
     * @param topicName
     * @throws IOException
     */
    public void createPublisher(String topicName) throws IOException {
        // Create a publisher instance with default settings bound to the topic
        log.trace("Creating publisher for topicName: {}", topicName);
        PubSubPublisherStore.publishers.put(topicName, Publisher.newBuilder(TopicName.formatProjectTopicName(properties.getProjectId(),  topicName)).setCredentialsProvider(credentialsProvider)
                .setChannelProvider(transportChannelProvider).setEnableMessageOrdering(true).build());
    }
    
    /**
     * @throws InterruptedException
     */
    @PreDestroy
    public void stopPublishing() throws InterruptedException {
        // get all publishers
        List<Publisher> publisherValues = new ArrayList<>(PubSubPublisherStore.publishers.values());
        // clear map to prevent more publishing
        publishers.clear();
        for (Publisher publisher : publisherValues) {
            if (publisher != null) {
                publisher.shutdown();
                publisher.awaitTermination(1, TimeUnit.MINUTES);
            }
        }
    }
    
    /**
     * @param topic
     * @return
     */
    public static Optional<Publisher> getPublisher(String topic) {
        return Optional.ofNullable(publishers.get(topic));
    }
    
    private TopicAdminSettings getTopicAdminSettings(CredentialsProvider credentialsProvider,
            TransportChannelProvider transportChannelProvider) throws IOException {
        return TopicAdminSettings.newBuilder().setCredentialsProvider(credentialsProvider)
                .setTransportChannelProvider(transportChannelProvider).build();
    }
}
