/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.pubsub.server.beans;

import static com.google.monitoring.v3.ListTimeSeriesRequest.newBuilder;
import static com.google.monitoring.v3.ProjectName.of;
import static com.google.protobuf.util.Timestamps.fromMillis;
import static java.lang.System.currentTimeMillis;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.rpc.TransportChannelProvider;
import com.google.cloud.monitoring.v3.MetricServiceClient;
import com.google.cloud.monitoring.v3.MetricServiceSettings;
import com.google.cloud.monitoring.v3.MetricServiceClient.ListTimeSeriesPagedResponse;
import com.google.monitoring.v3.ListTimeSeriesRequest;
import com.google.monitoring.v3.TimeInterval;

import lombok.extern.slf4j.Slf4j;
import org.autumnframework.service.pubsub.api.properties.PubSubApiProperties;
import org.autumnframework.service.queue.api.server.services.QueueStatsProcessor;

/**
 * 
 */
@Slf4j
@Component
public class PubSubStatsProcessor implements QueueStatsProcessor {

    private final CredentialsProvider credentialsProvider;
    
    private final TransportChannelProvider transportChannelProvider;

    private final String projectId;
    
    private List<String> subscriptions;
    
    private Integer subscriptionMaxMessages;
    
    private boolean isLocal = false;
    
    /**
     * @param credentialsProvider
     * @param transportChannelProvider
     * @param environment
     * @param properties
     */
    public PubSubStatsProcessor(CredentialsProvider credentialsProvider,
            @Qualifier("subscriberTransportChannelProvider") TransportChannelProvider transportChannelProvider, 
            Environment environment, PubSubApiProperties properties) {
        this.credentialsProvider = credentialsProvider;
        this.transportChannelProvider = transportChannelProvider;
        projectId = properties.getProjectId();
        this.subscriptions = properties.getMonitoring().getSubscriptions();
        if (this.subscriptions == null) {
            this.subscriptions = new ArrayList<>();
            log.warn("\n\n\t #### No subscriptions to monitor, you may want to add a property 'autumn.monitoring.subscriptions' to your application.yml ####\n");
        }
        this.subscriptionMaxMessages = properties.getMonitoring().getSubscriptionMaxMessages();
        if (this.subscriptionMaxMessages == null) {
            this.subscriptionMaxMessages = Integer.valueOf(5000);
            log.warn("\n\n\t #### No maximum message count defined, you may want to add a property 'autumn.monitoring.subscriptionMaxMessages' to your application.yml. Using default of 5000. ####\n");
        }
        if (ArrayUtils.contains(environment.getActiveProfiles(), "local")) {
            this.isLocal = true;
            log.warn("\n\n\t#### Cannot check for Metrics locally, calls to 'PubSubStatsProcessor.checkSubscriptionsAboveMaximumOrPubsubError()' will always return false even if the queues are too full ####\n");
        }
    }
    
    @Override
    public boolean checkQueuesAboveMaximumOrError() {

        if (this.isLocal) {
            return false; // no cloud monitoring locally, so unable to determine metrics for subscriptions
        }
        
        boolean noErrorsFound = true;
        MetricServiceClient service = null;
        
        try {
    
            TimeInterval interval = TimeInterval.newBuilder()
                                       .setStartTime(fromMillis(currentTimeMillis() - (120 * 1000)))
                                       .setEndTime(fromMillis(currentTimeMillis()))
                                       .build();
    
            ListTimeSeriesRequest request = newBuilder().setName(of(projectId).toString())
                   .setFilter("metric.type=\"pubsub.googleapis.com/subscription/num_undelivered_messages\"")
                   .setInterval(interval)
                   .setView(ListTimeSeriesRequest.TimeSeriesView.FULL)
                   .build();
    
            MetricServiceSettings settings = MetricServiceSettings.newBuilder().setCredentialsProvider(credentialsProvider).setTransportChannelProvider(transportChannelProvider).build();
            
            service = MetricServiceClient.create(settings);
            
            ListTimeSeriesPagedResponse response = service.listTimeSeries(request);
            
            for (var subscriptionData : response.iterateAll()) {
                
                log.trace("Subscription data labels: {}", subscriptionData.getResource().getLabelsMap());
                log.trace("Subscription points list: {}", subscriptionData.getPointsList());
                
                var subscription = subscriptionData.getResource().getLabelsMap().get("subscription_id");
                
                var numberOfMessages = subscriptionData.getPointsList().get(0).getValue().getInt64Value();
                
                // TODO: We could extend this check to determine whether a subscription to check is a
                // fully qualified subscription or just a service name that we can check against our
                // own service-out-topic name and the subscription instead of just checking all the
                // subscriptions ending in the service name for their message queue size.
                // But we should also check the in-topic for all the servers in the subscriptions in
                // that case, because we don't know whether we have senders using a service-name 
                // instead of the default out topic defined.
                if (subscriptions.contains(getSubscriptionName(subscription))) {
                    log.trace("Subscription {}, size: {}, result: ", subscription, numberOfMessages, numberOfMessages > subscriptionMaxMessages ? "OVER LIMIT" : "OK");
                    noErrorsFound = noErrorsFound && numberOfMessages > subscriptionMaxMessages;
                } else {
                    log.trace("Subscription {} not in check list, size: {}", subscription, numberOfMessages);
                }
            }
            
            return !noErrorsFound;
            
        } catch (Exception e) {
            log.error("Error fetching queue sizes from google: {}", e.getMessage(), e);
            return true;
        } finally {
            if (service != null && !service.isShutdown()) {
                service.shutdown();
            }
        }
    }

    private String getSubscriptionName(String subscription) {
        return subscription == null || subscription.indexOf('/') < 0 ? subscription : subscription.split("/")[subscription.split("/").length - 1];
    }
}
