/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.pubsub.server.config;

import java.io.IOException;

import org.autumnframework.service.pubsub.api.properties.PubSubApiProperties;
import org.autumnframework.service.pubsub.server.beans.PubSubPublisherStore;
import org.springframework.stereotype.Component;

import com.google.api.gax.rpc.AlreadyExistsException;
import com.google.api.gax.rpc.PermissionDeniedException;

import jakarta.annotation.PostConstruct;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;

/**
 * 
 */
@Log4j2
@Component
@RequiredArgsConstructor
public class PubSubServerChannelConfig {
    
    private final PubSubApiProperties properties; 
    private final PubSubPublisherStore store;

    /**
     * Setup the out topic
     * 
     * @param properties
     * @param credentialsProvider
     * @param transportChannelProvider
     * @throws IOException
     */
    @PostConstruct
    private void setUpChannels() throws IOException {
    	try {
    		store.createTopic(properties.getOutTopicName());
    	} catch (AlreadyExistsException ignore) {
            // topic already exists
        } catch (IOException e) {
        	log.error("Error setting up topic '{}': {}", properties.getOutTopicName(), e.getMessage(), e);
        } catch (PermissionDeniedException e) {
        	log.error("\r\n *** Permission denied setting up topic '{}': {} ***\r\n\r\n", properties.getOutTopicName(), e.getMessage(), e);
        }
    	try {
            store.createPublisher(properties.getOutTopicName());
    	} catch (IOException e) {
        	log.error("Error setting up publisher for topic '{}': {}", properties.getOutTopicName(), e.getMessage(), e);
        }
    }
    
}
