/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.pubsub.server.sender;

import static org.autumnframework.service.pubsub.api.properties.PubSubApiProperties.getCreateRoutingKey;
import static org.autumnframework.service.pubsub.api.properties.PubSubApiProperties.getDeleteRoutingKey;
import static org.autumnframework.service.pubsub.api.properties.PubSubApiProperties.getUpdateRoutingKey;

import java.io.IOException;
import java.io.Serializable;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.UUID;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.api.gax.rpc.AlreadyExistsException;
import com.google.api.gax.rpc.PermissionDeniedException;
import com.google.protobuf.ByteString;
import com.google.pubsub.v1.PubsubMessage;

import io.micrometer.core.annotation.Timed;
import lombok.extern.slf4j.Slf4j;
import org.autumnframework.service.api.ServiceProperties;
import org.autumnframework.service.identifiable.GenericIdentifiable;
import org.autumnframework.service.pubsub.api.properties.PubSubApiProperties;
import org.autumnframework.service.pubsub.server.beans.PubSubPublisherStore;
import org.autumnframework.service.queue.api.messages.GenericIdentifiableMessage;
import org.autumnframework.service.queue.api.server.services.GenericSenderService;
import org.autumnframework.service.validation.services.DefaultValidationService;

/**
 * @param <I>
 * @param <M>
 * @param <ID>
 */
@Slf4j
public abstract class GenericPubSubSender<I extends GenericIdentifiable<ID>, M extends GenericIdentifiableMessage<I, ID>, ID extends Serializable>
        implements GenericSenderService<I, M, ID> {

    private static final String NO_PUBLISHER_FOUND_FOR_TOPIC_S_SERVER_MAY_BE_SHUTTING_DOWN = "No publisher found for topic: %s, server may be shutting down";

    private final ObjectMapper objectMapper;
    private final DefaultValidationService validationService;
    private final String service;
    private final String topic;

    /**
     * @param objectMapper
     * @param properties
     * @param serviceProperties
     * @param validationService
     */
    public GenericPubSubSender(ObjectMapper objectMapper, PubSubApiProperties properties, ServiceProperties serviceProperties, DefaultValidationService validationService) {
        this.objectMapper = objectMapper;
        this.validationService = validationService;
        this.service = serviceProperties.getName();
        this.topic = properties.getOutTopicName();
    }

    /**
     * @param objectMapper
     * @param serviceName
     * @param validationService
     * @param store
     */
    public GenericPubSubSender(ObjectMapper objectMapper, String serviceName, DefaultValidationService validationService, PubSubPublisherStore store) {
        this.objectMapper = objectMapper;
        this.validationService = validationService;
        this.service = serviceName;
        this.topic = PubSubApiProperties.getInTopicName(serviceName);
        try {
            store.createTopic(topic);
        } catch (AlreadyExistsException ignore) {
            // topic already exists
        } catch (IOException e) {
        	log.error("Error setting up topic '{}': {}", this.topic, e.getMessage(), e);
        } catch (PermissionDeniedException e) {
        	log.error("\r\n *** Permission denied setting up topic '{}': {} ***\r\n\r\n", this.topic, e.getMessage(), e);
        }
        try {
            store.createPublisher(topic);
        } catch (IOException e) {
        	log.error("Error setting up publisher for topic '{}': {}", this.topic, e.getMessage(), e);
        }
    }
    
    private PubsubMessage createMessage(M message, String operation) {
        try {
            PubsubMessage msg =  PubsubMessage.newBuilder()
                    .putAttributes(PubSubApiProperties.DEFAULT_CLASSID_ATTRIBUTE, message.getClass().getName())
                    .putAttributes(PubSubApiProperties.DEFAULT_OPERATION_ATTRIBUTE, operation)
                    .setMessageId(message.getMessageChainId() == null ? UUID.randomUUID().toString() : message.getMessageChainId().toString())
                    .setOrderingKey(getOrderingKey())
                    .setData(toPayload(message))
                    .build();
            // log.trace("PubSubMessage created: {}", msg);
            return msg;
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private ByteString toPayload(M message) throws JsonProcessingException {
        return ByteString.copyFromUtf8(objectMapper.writeValueAsString(message));
    }

    protected String getOrderingKey() {
        return String.format("%d%09d", LocalDateTime.now().toInstant( ZoneOffset.UTC).toEpochMilli(), LocalDateTime.now().toInstant( ZoneOffset.UTC).getNano());
    }

    @Override
    @Timed(value = "autumn.messaging.pubsub.sent.create", description = "Number of create messages sent", extraTags = {
            "routing-key", "create" })
    public void sendCreate(M message) {
        log.trace("send create to topic:{} with routing key {} and message {}", topic,
                getCreateRoutingKey(service), message);
        validationService.onCreateValidate(message.getPayload());
        PubSubPublisherStore.getPublisher(topic).orElseThrow(() -> new IllegalStateException(String.format(NO_PUBLISHER_FOUND_FOR_TOPIC_S_SERVER_MAY_BE_SHUTTING_DOWN, topic))).publish(createMessage(message, getCreateRoutingKey(service)));
    }

    @Override
    @Timed(value = "autumn.messaging.pubsub.sent.update", description = "Number of update messages sent", extraTags = {
            "routing-key", "update" })
    public void sendUpdate(M message) {
        log.trace("send update to topic:{} with routing key {} and message {}", topic,
                getUpdateRoutingKey(service), message);
        validationService.onUpdateValidate(message.getPayload());
        PubSubPublisherStore.getPublisher(topic).orElseThrow(() -> new IllegalStateException(String.format(NO_PUBLISHER_FOUND_FOR_TOPIC_S_SERVER_MAY_BE_SHUTTING_DOWN, topic))).publish(createMessage(message, getUpdateRoutingKey(service)));
    }

    @Override
    @Timed(value = "autumn.messaging.pubsub.sent.delete", description = "Number of delete messages sent", extraTags = {
            "routing-key", "delete" })
    public void sendDelete(M message) {
        log.trace("send update to topic:{} with routing key {} and message {}", topic,
                getDeleteRoutingKey(service), message);
        validationService.onDeleteValidate(message.getPayload());
        PubSubPublisherStore.getPublisher(topic).orElseThrow(() -> new IllegalStateException(String.format(NO_PUBLISHER_FOUND_FOR_TOPIC_S_SERVER_MAY_BE_SHUTTING_DOWN, topic))).publish(createMessage(message, getDeleteRoutingKey(service)));
    }
    
}
