/*
 * Copyright (c) 2010-2022. Axon Framework
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.axonframework.extensions.multitenancy.autoconfig;

import io.axoniq.axonserver.connector.control.ControlChannel;
import org.axonframework.axonserver.connector.AxonServerConfiguration;
import org.axonframework.axonserver.connector.AxonServerConnectionManager;
import org.axonframework.axonserver.connector.processor.EventProcessorControlService;
import org.axonframework.common.Registration;
import org.axonframework.config.EventProcessingConfiguration;
import org.axonframework.extensions.multitenancy.components.MultiTenantAwareComponent;
import org.axonframework.extensions.multitenancy.components.TenantDescriptor;
import org.axonframework.extensions.multitenancy.components.eventhandeling.MultiTenantEventProcessor;
import org.axonframework.lifecycle.Phase;
import org.axonframework.lifecycle.StartHandler;

/**
 * Multi-tenant implementation of {@link EventProcessorControlService}. Enables event processor control for multi-tenant
 * environment in Axon Server dashboard.
 *
 * @author Stefan Dragisic
 * @since 4.6.0
 */
public class MultiTenantEventProcessorControlService extends EventProcessorControlService
        implements MultiTenantAwareComponent {

    public MultiTenantEventProcessorControlService(
            AxonServerConnectionManager axonServerConnectionManager,
            EventProcessingConfiguration eventProcessingConfiguration,
            AxonServerConfiguration axonServerConfiguration) {
        super(axonServerConnectionManager, eventProcessingConfiguration, "");
    }

    public MultiTenantEventProcessorControlService(
            AxonServerConnectionManager axonServerConnectionManager,
            EventProcessingConfiguration eventProcessingConfiguration, String context) {
        super(axonServerConnectionManager, eventProcessingConfiguration, "");
    }

    @StartHandler(phase = Phase.INSTRUCTION_COMPONENTS)
    @Override
    public void start() {
        if (axonServerConnectionManager != null && eventProcessingConfiguration != null) {
            eventProcessingConfiguration.eventProcessors()
                                        .forEach(
                                                (name, processor) -> {
                                                    if (processor instanceof MultiTenantEventProcessor) {
                                                        return;
                                                    }
                                                    String ctx = name.substring(name.indexOf("@") + 1);
                                                    ControlChannel ch = axonServerConnectionManager.getConnection(ctx)
                                                                                                   .controlChannel();
                                                    ch.registerEventProcessor(name,
                                                                              infoSupplier(processor),
                                                                              new AxonProcessorInstructionHandler(
                                                                                      processor,
                                                                                      name));
                                                });
        }
    }

    @Override
    public Registration registerTenant(TenantDescriptor tenantDescriptor) {
        //Already registered
        return () -> true;
    }

    @Override
    public Registration registerAndStartTenant(TenantDescriptor tenantDescriptor) {
        if (axonServerConnectionManager != null && eventProcessingConfiguration != null) {
            eventProcessingConfiguration.eventProcessors()
                                        .forEach(
                                                (name, processor) -> {
                                                    if (processor instanceof MultiTenantEventProcessor
                                                            || !name.contains(tenantDescriptor.tenantId())) {
                                                        return;
                                                    }
                                                    ControlChannel ch = axonServerConnectionManager.getConnection(
                                                                                                           tenantDescriptor.tenantId())
                                                                                                   .controlChannel();
                                                    ch.registerEventProcessor(
                                                            name,
                                                            infoSupplier(processor),
                                                            new AxonProcessorInstructionHandler(
                                                                    processor,
                                                                    name));
                                                });
        }

        return () -> true;
    }
}
