/**
 * Copyright (C) 2013 BonitaSoft S.A.
 * BonitaSoft, 32 rue Gustave Eiffel - 38000 Grenoble
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 ** 
 * @since 6.0
 */
package org.bonitasoft.engine.api;

import java.util.List;
import java.util.Map;

import org.bonitasoft.engine.exception.AlreadyExistsException;
import org.bonitasoft.engine.exception.CreationException;
import org.bonitasoft.engine.exception.DeletionException;
import org.bonitasoft.engine.exception.RetrieveException;
import org.bonitasoft.engine.exception.SearchException;
import org.bonitasoft.engine.identity.UserNotFoundException;
import org.bonitasoft.engine.profile.Profile;
import org.bonitasoft.engine.profile.ProfileEntry;
import org.bonitasoft.engine.profile.ProfileEntryNotFoundException;
import org.bonitasoft.engine.profile.ProfileMember;
import org.bonitasoft.engine.profile.ProfileMemberCreator;
import org.bonitasoft.engine.profile.ProfileNotFoundException;
import org.bonitasoft.engine.search.SearchOptions;
import org.bonitasoft.engine.search.SearchResult;
import org.bonitasoft.engine.session.InvalidSessionException;

/**
 * @author Celine Souchet
 * @author Matthieu Chaffotte
 */
public interface ProfileAPI {

    /**
     * Retrieves the profile.
     * 
     * @param id
     *            the identifier of the profile
     * @return the searched profile
     * @throws ProfileNotFoundException
     *             If the identifier does not refer to an existing profile
     * @throws RetrieveException
     *             If an exception occurs during the profile retrieving
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    Profile getProfile(long id) throws ProfileNotFoundException;

    /**
     * Retrieves the profiles of the user.
     * 
     * @param userId
     *            the identifier of the user
     * @return the profiles of the user
     * @throws UserNotFoundException
     *             If the user identifier does not refer to an existing user
     * @throws RetrieveException
     *             If an exception occurs during the profile retrieving
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    List<Profile> getProfilesForUser(long userId) throws UserNotFoundException;

    /**
     * Searches profiles according to the criteria containing in the options.
     * 
     * @param options
     *            the search criteria
     * @return the search result
     * @throws SearchException
     *             If an exception occurs during the profile searching
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    SearchResult<Profile> searchProfiles(SearchOptions options) throws SearchException;

    /**
     * Retrieves the number of profile members for the profiles. The map contains the couples profileId/numberOfProfileMembers.
     * <b>
     * If a profile does not exists, no exception is thrown and no value is added in the map.
     * 
     * @param profileIds
     *            the identifiers of the profiles
     * @return the number of profile members for the profiles
     * @throws RetrieveException
     *             If an exception occurs during the profile retrieving
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    Map<Long, Long> getNumberOfProfileMembers(List<Long> profileIds);

    /**
     * Searches profile members according to the criteria containing in the options.
     * 
     * @param memberType
     *            the member type, it can be: user, role, group, roleAndGroup.
     * @param options
     *            the search criteria
     * @return the search result
     * @throws SearchException
     *             If an exception occurs during the profile searching
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    SearchResult<ProfileMember> searchProfileMembers(String memberType, SearchOptions options) throws SearchException;

    /**
     * Searches profile entries according to the criteria containing in the options.
     * 
     * @param options
     *            the search criteria
     * @return the search result
     * @throws SearchException
     *             If an exception occurs during the profile searching
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    SearchResult<ProfileEntry> searchProfileEntries(SearchOptions options) throws SearchException;

    /**
     * Returns a profile entry according to its identifier.
     * 
     * @param id
     *            the profile entry identifier
     * @return the searched profile entry
     * @throws ProfileEntryNotFoundException
     *             occurs when the identifier does not refer to an existing profile entry
     * @since 6.0
     */

    /**
     * Retrieves the profile entry.
     * 
     * @param id
     *            the identifier of the profile entry
     * @return the searched profile entry
     * @throws ProfileEntryNotFoundException
     *             If the profile entry identifier does not refer to an existing user
     * @throws RetrieveException
     *             If an exception occurs during the user retrieving
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    ProfileEntry getProfileEntry(long id) throws ProfileEntryNotFoundException;

    /**
     * Creates a profile member.
     * 
     * @param profileId
     *            the identifier of the profile
     * @param userId
     *            the identifier of the user
     * @param groupId
     *            the identifier of the group
     * @param roleId
     *            the identifier of the role
     * @return the created profile member
     * @throws AlreadyExistsException
     *             If the tuple profileId/userId/roleId/groupId is already taken by an existing profile member
     * @throws CreationException
     *             If an exception occurs during the profile member creation
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    ProfileMember createProfileMember(Long profileId, Long userId, Long groupId, Long roleId) throws CreationException, AlreadyExistsException;

    /**
     * Creates a profile member.
     * <b>
     * It takes the values of the creator in order to create the profile member.
     * 
     * @param creator
     *            the profile member to create
     * @return the created profile member
     * @throws AlreadyExistsException
     *             If the tuple profileId/userId/roleId/groupId is already taken by an existing profile member
     * @throws CreationException
     *             If an exception occurs during the profile member creation
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    ProfileMember createProfileMember(ProfileMemberCreator creator) throws CreationException, AlreadyExistsException;

    /**
     * Deletes the profile member.
     * 
     * @param id
     *            the identifier of the profile member
     * @throws DeletionException
     *             If an exception occurs during the profile member deletion
     * @throws InvalidSessionException
     *             If the session is invalid (expired, unknown, ...)
     * @since 6.0
     */
    void deleteProfileMember(Long id) throws DeletionException;

}
