/*
 * Copyright (c) 2011-2019, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package boofcv.alg.feature.disparity.block.score;

import boofcv.alg.feature.disparity.block.DisparitySparseScoreSadRect;
import boofcv.struct.border.ImageBorder_S32;
import boofcv.struct.image.GrayU8;

import java.util.Arrays;

/**
 * <p>
 * Implementation of {@link DisparitySparseScoreSadRect} that processes images of type {@link GrayU8}.
 * </p>
 *
 * <p>
 * DO NOT MODIFY. Generated by {@link GenerateDisparitySparseScoreSadRect}.
 * </p>
 *
 * @author Peter Abeles
 */
public class DisparitySparseScoreBM_SAD_U8 extends DisparitySparseScoreSadRect<int[],GrayU8> {

	// scores up to the maximum baseline
	int[] scores;

	public DisparitySparseScoreBM_SAD_U8( int radiusX, int radiusY) {
		super(radiusX, radiusY);
	}

	@Override
	public void configure(int disparityMin, int disparityRange) {
		super.configure(disparityMin, disparityRange);
		scores = new int[ disparityRange ];
	}

	@Override
	public boolean process( int x , int y ) {
		// can't estimate disparity if there are no pixels it can estimate disparity from
		if( x < disparityMin)
			return false;

		// adjust disparity for image border
		localMaxRange = Math.min(x, disparityMax)- disparityMin +1;

		Arrays.fill(scores,0);
		if( x < localMaxRange+radiusX+ disparityMin || x >= left.width-radiusX || y < radiusY || y >= left.height-radiusY )
			scoreBorder(x,y);
		else
			scoreInner(x, y);

		return true;
	}

	private void scoreBorder(int cx, int cy) {
		ImageBorder_S32 bleft = (ImageBorder_S32)this.bleft;
		ImageBorder_S32 bright = (ImageBorder_S32)this.bright;

		// sum up horizontal errors in the region
		for (int y = -radiusY; y <= radiusY; y++) {
			for (int d = 0; d < localMaxRange; d++) {
				int score = 0;
				for (int x = -radiusX; x <= radiusX; x++) {
					int diff = bleft.get(cx+x,cy+y)-bright.get(cx+x-d- disparityMin,+cy+y);
					score += Math.abs(diff);
				}
				scores[d] += score;
			}
		}
	}

	private void scoreInner(int x, int y) {
		// sum up horizontal errors in the region
		for( int row = 0; row < regionHeight; row++ ) {
			// pixel indexes
			int startLeft = left.startIndex + left.stride*(y-radiusY+row) + x-radiusX;
			int startRight = right.startIndex + right.stride*(y-radiusY+row) + x-radiusX- disparityMin;

			for(int i = 0; i < localMaxRange; i++ ) {
				int indexLeft = startLeft;
				int indexRight = startRight-i;

				int score = 0;
				for( int j = 0; j < regionWidth; j++ ) {
					int diff = (left.data[ indexLeft++ ]& 0xFF) - (right.data[ indexRight++ ]& 0xFF);

					score += Math.abs(diff);
				}
				scores[i] += score;
			}
		}
	}

	@Override
	public int[] getScore() {
		return scores;
	}

	@Override
	public Class<GrayU8> getImageType() {
		return GrayU8.class;
	}

}
