/*
 * Copyright (c) 2011-2018, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package boofcv.alg.geo.bundle;

import boofcv.abst.geo.bundle.BundleAdjustmentSchur_DSCC;
import boofcv.abst.geo.bundle.SceneStructureMetric;
import georegression.geometry.ConvertRotation3D_F64;
import georegression.struct.so.Rodrigues_F64;

/**
 * Encodes and decodes the values in a {@link SceneStructureMetric} using the following
 * parameterization:<br>
 * <pre>
 * [ (X Y Z)*M ][ (rodX rodY rodZ Tx Ty Tz)*N ][ intrinsic*O ]
 * [ features  ][           views             ][ camera      ]
 * </pre>
 * @author Peter Abeles
 */
public class CodecSceneStructureMetric implements BundleAdjustmentSchur_DSCC.Codec<SceneStructureMetric>
{
	// local variable which stores the predicted location of the feature in the camera frame
	private Rodrigues_F64 rodrigues = new Rodrigues_F64();

	@Override
	public void decode(double[] input , SceneStructureMetric structure ) {
		int index = 0;

		for (int i = 0; i < structure.points.length; i++) {
			SceneStructureMetric.Point p = structure.points[i];
			p.coordinate[0] = input[index++];
			p.coordinate[1] = input[index++];
			p.coordinate[2] = input[index++];
			if( structure.isHomogenous() )
				p.coordinate[3] = input[index++];
		}

		for( int viewIndex = 0; viewIndex < structure.views.length; viewIndex++ ) {
			SceneStructureMetric.View view = structure.views[viewIndex];
			// Decode the rigid body transform from world to view
			if( !view.known ) {
				double rodX = input[index++];
				double rodY = input[index++];
				double rodZ = input[index++];

				view.worldToView.T.x = input[index++];
				view.worldToView.T.y = input[index++];
				view.worldToView.T.z = input[index++];

				rodrigues.setParamVector(rodX,rodY,rodZ);

				ConvertRotation3D_F64.rodriguesToMatrix(rodrigues,view.worldToView.R);
			}
		}

		for (int i = 0; i < structure.cameras.length; i++) {
			SceneStructureMetric.Camera camera = structure.cameras[i];
			if( !camera.known ) {
				camera.model.setIntrinsic(input,index);
				index += camera.model.getIntrinsicCount();
			}
		}
	}

	@Override
	public void encode(SceneStructureMetric structure , double[] output ) {
		int index = 0;

		for (int i = 0; i < structure.points.length; i++) {
			SceneStructureMetric.Point p = structure.points[i];
			output[index++] = p.coordinate[0];
			output[index++] = p.coordinate[1];
			output[index++] = p.coordinate[2];
			if( structure.isHomogenous() )
				output[index++] = p.coordinate[3];
		}

		for( int viewIndex = 0; viewIndex < structure.views.length; viewIndex++ ) {
			SceneStructureMetric.View view = structure.views[viewIndex];
			// Decode the rigid body transform from world to view
			if( !view.known ) {
				ConvertRotation3D_F64.matrixToRodrigues(view.worldToView.R,rodrigues);
				rodrigues.unitAxisRotation.scale(rodrigues.theta);
				output[index++] = rodrigues.unitAxisRotation.x;
				output[index++] = rodrigues.unitAxisRotation.y;
				output[index++] = rodrigues.unitAxisRotation.z;

				output[index++] = view.worldToView.T.x;
				output[index++] = view.worldToView.T.y;
				output[index++] = view.worldToView.T.z;
			}
		}

		for (int i = 0; i < structure.cameras.length; i++) {
			SceneStructureMetric.Camera camera = structure.cameras[i];
			if( !camera.known ) {
				camera.model.getIntrinsic(output,index);
				index += camera.model.getIntrinsicCount();
			}
		}
	}
}
