/*
 * Copyright (c) 2011-2019, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package boofcv.alg.misc;

import boofcv.struct.image.*;
import boofcv.alg.misc.impl.ImplImageBandMath;
import boofcv.alg.misc.impl.ImplImageBandMath_MT;
import boofcv.concurrency.BoofConcurrency;

import javax.annotation.Generated;
import javax.annotation.Nullable;

/**
 *
 * <p>DO NOT MODIFY: Generated by boofcv.alg.misc.GenerateImageBandMath.</p>
 *
 * Collection of functions that project Bands of Planar images onto
 * a single image. Can be used to perform projections such as
 * minimum, maximum, average, median, standard Deviation.
 * @author Nico Stuurman
 * @author Peter Abeles
 */
@Generated("boofcv.alg.misc.GenerateImageBandMath")
@SuppressWarnings("Duplicates")
public class ImageBandMath {

	public static <T extends ImageGray<T>> void checkInput(Planar<T> input, int startBand, int lastBand) {
		if (startBand < 0 || lastBand < 0) {
			throw new IllegalArgumentException("startBand or lastBand is less than zero");
		}
		if (startBand > lastBand) {
			throw new IllegalArgumentException("startBand should <= lastBand");
		}
		if (lastBand >= input.getNumBands()) {
			throw new IllegalArgumentException("lastBand should be less than number of Bands in input");
		}
	}


	/**
	 * Computes the minimum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 */
	public static void minimum(Planar<GrayU8> input, GrayU8 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.minimum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the minimum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void minimum(Planar<GrayU8> input, GrayU8 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.minimum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the maximum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 */
	public static void maximum(Planar<GrayU8> input, GrayU8 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.maximum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the maximum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void maximum(Planar<GrayU8> input, GrayU8 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.maximum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 */
	public static void average(Planar<GrayU8> input, GrayU8 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.average(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the average for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void average(Planar<GrayU8> input, GrayU8 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.average(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the median for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 */
	public static void median(Planar<GrayU8> input, GrayU8 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.median(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the median for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void median(Planar<GrayU8> input, GrayU8 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.median(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	*/
	public static void stdDev(Planar<GrayU8> input, GrayU8 output, @Nullable GrayU8 avg) {
		stdDev(input,output,avg,0,input.getNumBands() - 1);
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	*/
	public static void stdDev(Planar<GrayU8> input, GrayU8 output, @Nullable GrayU8 avg, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if( avg == null ) {
			avg = new GrayU8(input.width,input.height);
			average(input,avg,startBand,lastBand);
		}
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.stdDev(input,output,avg,startBand,lastBand);
		} else {
			ImplImageBandMath.stdDev(input,output,avg,startBand,lastBand);
		}
	}

	/**
	 * Computes the minimum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 */
	public static void minimum(Planar<GrayS16> input, GrayS16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.minimum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the minimum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void minimum(Planar<GrayS16> input, GrayS16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.minimum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the maximum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 */
	public static void maximum(Planar<GrayS16> input, GrayS16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.maximum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the maximum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void maximum(Planar<GrayS16> input, GrayS16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.maximum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 */
	public static void average(Planar<GrayS16> input, GrayS16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.average(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the average for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void average(Planar<GrayS16> input, GrayS16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.average(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the median for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 */
	public static void median(Planar<GrayS16> input, GrayS16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.median(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the median for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void median(Planar<GrayS16> input, GrayS16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.median(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	*/
	public static void stdDev(Planar<GrayS16> input, GrayS16 output, @Nullable GrayS16 avg) {
		stdDev(input,output,avg,0,input.getNumBands() - 1);
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	*/
	public static void stdDev(Planar<GrayS16> input, GrayS16 output, @Nullable GrayS16 avg, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if( avg == null ) {
			avg = new GrayS16(input.width,input.height);
			average(input,avg,startBand,lastBand);
		}
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.stdDev(input,output,avg,startBand,lastBand);
		} else {
			ImplImageBandMath.stdDev(input,output,avg,startBand,lastBand);
		}
	}

	/**
	 * Computes the minimum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 */
	public static void minimum(Planar<GrayU16> input, GrayU16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.minimum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the minimum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void minimum(Planar<GrayU16> input, GrayU16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.minimum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the maximum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 */
	public static void maximum(Planar<GrayU16> input, GrayU16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.maximum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the maximum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void maximum(Planar<GrayU16> input, GrayU16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.maximum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 */
	public static void average(Planar<GrayU16> input, GrayU16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.average(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the average for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void average(Planar<GrayU16> input, GrayU16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.average(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the median for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 */
	public static void median(Planar<GrayU16> input, GrayU16 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.median(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the median for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void median(Planar<GrayU16> input, GrayU16 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.median(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	*/
	public static void stdDev(Planar<GrayU16> input, GrayU16 output, @Nullable GrayU16 avg) {
		stdDev(input,output,avg,0,input.getNumBands() - 1);
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	*/
	public static void stdDev(Planar<GrayU16> input, GrayU16 output, @Nullable GrayU16 avg, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if( avg == null ) {
			avg = new GrayU16(input.width,input.height);
			average(input,avg,startBand,lastBand);
		}
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.stdDev(input,output,avg,startBand,lastBand);
		} else {
			ImplImageBandMath.stdDev(input,output,avg,startBand,lastBand);
		}
	}

	/**
	 * Computes the minimum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 */
	public static void minimum(Planar<GrayS32> input, GrayS32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.minimum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the minimum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void minimum(Planar<GrayS32> input, GrayS32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.minimum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the maximum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 */
	public static void maximum(Planar<GrayS32> input, GrayS32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.maximum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the maximum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void maximum(Planar<GrayS32> input, GrayS32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.maximum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 */
	public static void average(Planar<GrayS32> input, GrayS32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.average(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the average for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void average(Planar<GrayS32> input, GrayS32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.average(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the median for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 */
	public static void median(Planar<GrayS32> input, GrayS32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.median(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the median for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void median(Planar<GrayS32> input, GrayS32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.median(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	*/
	public static void stdDev(Planar<GrayS32> input, GrayS32 output, @Nullable GrayS32 avg) {
		stdDev(input,output,avg,0,input.getNumBands() - 1);
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	*/
	public static void stdDev(Planar<GrayS32> input, GrayS32 output, @Nullable GrayS32 avg, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if( avg == null ) {
			avg = new GrayS32(input.width,input.height);
			average(input,avg,startBand,lastBand);
		}
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.stdDev(input,output,avg,startBand,lastBand);
		} else {
			ImplImageBandMath.stdDev(input,output,avg,startBand,lastBand);
		}
	}

	/**
	 * Computes the minimum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 */
	public static void minimum(Planar<GrayS64> input, GrayS64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.minimum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the minimum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void minimum(Planar<GrayS64> input, GrayS64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.minimum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the maximum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 */
	public static void maximum(Planar<GrayS64> input, GrayS64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.maximum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the maximum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void maximum(Planar<GrayS64> input, GrayS64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.maximum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 */
	public static void average(Planar<GrayS64> input, GrayS64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.average(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the average for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void average(Planar<GrayS64> input, GrayS64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.average(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the median for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 */
	public static void median(Planar<GrayS64> input, GrayS64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.median(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the median for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void median(Planar<GrayS64> input, GrayS64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.median(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	*/
	public static void stdDev(Planar<GrayS64> input, GrayS64 output, @Nullable GrayS64 avg) {
		stdDev(input,output,avg,0,input.getNumBands() - 1);
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	*/
	public static void stdDev(Planar<GrayS64> input, GrayS64 output, @Nullable GrayS64 avg, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if( avg == null ) {
			avg = new GrayS64(input.width,input.height);
			average(input,avg,startBand,lastBand);
		}
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.stdDev(input,output,avg,startBand,lastBand);
		} else {
			ImplImageBandMath.stdDev(input,output,avg,startBand,lastBand);
		}
	}

	/**
	 * Computes the minimum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 */
	public static void minimum(Planar<GrayF32> input, GrayF32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.minimum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the minimum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void minimum(Planar<GrayF32> input, GrayF32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.minimum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the maximum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 */
	public static void maximum(Planar<GrayF32> input, GrayF32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.maximum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the maximum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void maximum(Planar<GrayF32> input, GrayF32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.maximum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 */
	public static void average(Planar<GrayF32> input, GrayF32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.average(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the average for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void average(Planar<GrayF32> input, GrayF32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.average(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the median for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 */
	public static void median(Planar<GrayF32> input, GrayF32 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.median(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the median for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void median(Planar<GrayF32> input, GrayF32 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.median(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	*/
	public static void stdDev(Planar<GrayF32> input, GrayF32 output, @Nullable GrayF32 avg) {
		stdDev(input,output,avg,0,input.getNumBands() - 1);
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	*/
	public static void stdDev(Planar<GrayF32> input, GrayF32 output, @Nullable GrayF32 avg, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if( avg == null ) {
			avg = new GrayF32(input.width,input.height);
			average(input,avg,startBand,lastBand);
		}
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.stdDev(input,output,avg,startBand,lastBand);
		} else {
			ImplImageBandMath.stdDev(input,output,avg,startBand,lastBand);
		}
	}

	/**
	 * Computes the minimum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 */
	public static void minimum(Planar<GrayF64> input, GrayF64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.minimum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the minimum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing minimum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void minimum(Planar<GrayF64> input, GrayF64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.minimum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.minimum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the maximum for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 */
	public static void maximum(Planar<GrayF64> input, GrayF64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.maximum(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the maximum for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing maximum pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void maximum(Planar<GrayF64> input, GrayF64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.maximum(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.maximum(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the average for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 */
	public static void average(Planar<GrayF64> input, GrayF64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.average(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the average for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing average pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void average(Planar<GrayF64> input, GrayF64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.average(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.average(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the median for each pixel across all bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 */
	public static void median(Planar<GrayF64> input, GrayF64 output) {
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, 0, input.getNumBands() - 1);
		} else {
			ImplImageBandMath.median(input, output, 0, input.getNumBands() - 1);
		}
	}

	/**
	 * Computes the median for each pixel across selected bands in the {@link Planar} image.
	 *
	 * @param input     Planar image
	 * @param output    Gray scale image containing median pixel values
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	 */
	public static void median(Planar<GrayF64> input, GrayF64 output, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.median(input, output, startBand, lastBand);
		} else {
			ImplImageBandMath.median(input, output, startBand, lastBand);
		}
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	*/
	public static void stdDev(Planar<GrayF64> input, GrayF64 output, @Nullable GrayF64 avg) {
		stdDev(input,output,avg,0,input.getNumBands() - 1);
	}

	/**
	 * Computes the standard deviation for each pixel across all bands in the {@link Planar}
	 * image.
	 * @param input     Planar image - not modified
	 * @param output    Gray scale image containing average pixel values - modified
	 * @param avg       Input Gray scale image containing average image.  Can be null
	 * @param startBand First band to be included in the projection
	 * @param lastBand  Last band to be included in the projection
	*/
	public static void stdDev(Planar<GrayF64> input, GrayF64 output, @Nullable GrayF64 avg, int startBand, int lastBand) {
		checkInput(input, startBand, lastBand);
		output.reshape(input.width,input.height);
		if( avg == null ) {
			avg = new GrayF64(input.width,input.height);
			average(input,avg,startBand,lastBand);
		}
		if (BoofConcurrency.USE_CONCURRENT) {
			ImplImageBandMath_MT.stdDev(input,output,avg,startBand,lastBand);
		} else {
			ImplImageBandMath.stdDev(input,output,avg,startBand,lastBand);
		}
	}

}
