/*
 * Copyright (c) 2011-2016, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package boofcv.alg.feature.disparity.impl;

import boofcv.alg.feature.disparity.DisparityScoreSadRect;
import boofcv.alg.feature.disparity.DisparitySelect;
import boofcv.struct.image.GrayU8;
import boofcv.struct.image.ImageGray;

/**
 * <p>
 * Implementation of {@link boofcv.alg.feature.disparity.DisparityScoreSadRect} for processing
 * input images of type {@link GrayU8}.
 * </p>
 * <p>
 * DO NOT MODIFY. Generated by {@link GenerateDisparityScoreSadRect}.
 * </p>
 * 
 * @author Peter Abeles
 */
public class ImplDisparityScoreSadRect_U8<Disparity extends ImageGray>
	extends DisparityScoreSadRect<GrayU8,Disparity>
{

	// Computes disparity from scores
	DisparitySelect<int[],Disparity> computeDisparity;

	// stores the local scores for the width of the region
	int elementScore[];
	// scores along horizontal axis for current block
	// To allow right to left validation all disparity scores are stored for the entire row
	// size = num columns * maxDisparity
	// disparity for column i is stored in elements i*maxDisparity to (i+1)*maxDisparity
	int horizontalScore[][];
	// summed scores along vertical axis
	// This is simply the sum of like elements in horizontal score
	int verticalScore[];

	public ImplDisparityScoreSadRect_U8( int minDisparity , int maxDisparity,
										int regionRadiusX, int regionRadiusY,
										DisparitySelect<int[],Disparity> computeDisparity) {
		super(minDisparity,maxDisparity,regionRadiusX,regionRadiusY);

		this.computeDisparity = computeDisparity;
	}

	@Override
	public void _process(GrayU8 left , GrayU8 right , Disparity disparity ) {
		if( horizontalScore == null || verticalScore.length < lengthHorizontal ) {
			horizontalScore = new int[regionHeight][lengthHorizontal];
			verticalScore = new int[lengthHorizontal];
			elementScore = new int[ left.width ];
		}

		computeDisparity.configure(disparity,minDisparity,maxDisparity,radiusX);

		// initialize computation
		computeFirstRow(left, right);
		// efficiently compute rest of the rows using previous results to avoid repeat computations
		computeRemainingRows(left, right);
	}

	/**
	 * Initializes disparity calculation by finding the scores for the initial block of horizontal
	 * rows.
	 */
	private void computeFirstRow(GrayU8 left, GrayU8 right ) {
		// compute horizontal scores for first row block
		for( int row = 0; row < regionHeight; row++ ) {

			int scores[] = horizontalScore[row];

			UtilDisparityScore.computeScoreRow(left, right, row, scores,
					minDisparity,maxDisparity,regionWidth,elementScore);
		}

		// compute score for the top possible row
		for( int i = 0; i < lengthHorizontal; i++ ) {
			int sum = 0;
			for( int row = 0; row < regionHeight; row++ ) {
				sum += horizontalScore[row][i];
			}
			verticalScore[i] = sum;
		}

		// compute disparity
		computeDisparity.process(radiusY, verticalScore);
	}

	/**
	 * Using previously computed results it efficiently finds the disparity in the remaining rows.
	 * When a new block is processes the last row/column is subtracted and the new row/column is
	 * added.
	 */
	private void computeRemainingRows(GrayU8 left, GrayU8 right )
	{
		for( int row = regionHeight; row < left.height; row++ ) {
			int oldRow = row%regionHeight;

			// subtract first row from vertical score
			int scores[] = horizontalScore[oldRow];
			for( int i = 0; i < lengthHorizontal; i++ ) {
				verticalScore[i] -= scores[i];
			}

			UtilDisparityScore.computeScoreRow(left, right, row, scores,
					minDisparity,maxDisparity,regionWidth,elementScore);

			// add the new score
			for( int i = 0; i < lengthHorizontal; i++ ) {
				verticalScore[i] += scores[i];
			}

			// compute disparity
			computeDisparity.process(row - regionHeight + 1 + radiusY, verticalScore);
		}
	}

	@Override
	public Class<GrayU8> getInputType() {
		return GrayU8.class;
	}

	@Override
	public Class<Disparity> getDisparityType() {
		return computeDisparity.getDisparityType();
	}

}
