/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.bwa;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.broadinstitute.hellbender.utils.bwa.BwaMemPairEndStats;
import org.broadinstitute.hellbender.utils.bwa.CouldNotCreateIndexException;
import org.broadinstitute.hellbender.utils.bwa.CouldNotCreateIndexImageException;
import org.broadinstitute.hellbender.utils.bwa.CouldNotReadImageException;
import org.broadinstitute.hellbender.utils.bwa.CouldNotReadIndexException;
import org.broadinstitute.hellbender.utils.bwa.CouldNotReadReferenceException;
import org.broadinstitute.hellbender.utils.bwa.InvalidFileFormatException;

public final class BwaMemIndex
implements AutoCloseable {
    private static final int MAXIMUM_NUMBER_OF_CHARACTER_BEFORE_FIRST_HEADER_IN_FASTA_FILES = 4092;
    private static final char FASTA_HEADER_PREFIX_CHAR = '>';
    public static final List<String> INDEX_FILE_EXTENSIONS = Collections.unmodifiableList(Arrays.asList(".amb", ".ann", ".bwt", ".pac", ".sa"));
    public static final String IMAGE_FILE_EXTENSION = ".img";
    public static final List<String> FASTA_FILE_EXTENSIONS = Collections.unmodifiableList(Arrays.asList(".fasta", ".fa"));
    private final String indexImageFile;
    private volatile long indexAddress;
    private final AtomicInteger refCount;
    private final List<String> refContigNames;
    private static volatile boolean nativeLibLoaded = false;

    private static String resolveFastaFileExtension(String fasta) {
        Optional<String> extension = FASTA_FILE_EXTENSIONS.stream().filter(fasta::endsWith).findFirst();
        if (!extension.isPresent()) {
            throw new IllegalArgumentException(String.format("the fasta file provided '%s' does not have any of the standard fasta extensions: %s", fasta, FASTA_FILE_EXTENSIONS.stream().collect(Collectors.joining(", "))));
        }
        return extension.get();
    }

    public static void createIndexImageFromIndexFiles(String indexPrefix, String imageFile) {
        if (indexPrefix == null) {
            throw new IllegalArgumentException("the index prefix cannot be null");
        }
        if (imageFile == null) {
            throw new IllegalArgumentException("the image file cannot be null");
        }
        BwaMemIndex.assertLooksLikeIndexPrefix(indexPrefix);
        BwaMemIndex.loadNativeLibrary();
        BwaMemIndex.createIndexImageFile(indexPrefix, imageFile);
    }

    public static String createIndexImageFromFastaFile(String fasta) {
        String imageFile = BwaMemIndex.getDefaultIndexImageNameFromFastaFile(fasta);
        BwaMemIndex.createIndexImageFromFastaFile(fasta, imageFile);
        return imageFile;
    }

    public static String getDefaultIndexImageNameFromFastaFile(String fasta) {
        if (fasta == null) {
            throw new IllegalArgumentException("the input fasta file name cannot be null");
        }
        String extension = BwaMemIndex.resolveFastaFileExtension(fasta);
        String prefix = fasta.substring(0, fasta.length() - extension.length());
        return prefix + IMAGE_FILE_EXTENSION;
    }

    public static void createIndexImageFromFastaFile(String fasta, String imageFile) {
        BwaMemIndex.createIndexImageFromFastaFile(fasta, imageFile, Algorithm.AUTO);
    }

    public static void createIndexImageFromFastaFile(String fasta, String imageFile, Algorithm algo) {
        BwaMemIndex.assertLooksLikeFastaFile(fasta);
        BwaMemIndex.assertCanCreateOrOverwriteImageFile(imageFile);
        if (algo == null) {
            throw new IllegalArgumentException("the input algorithm must not be null");
        }
        File indexPrefix = BwaMemIndex.createTempIndexPrefix(fasta);
        BwaMemIndex.loadNativeLibrary();
        BwaMemIndex.createReferenceIndex(fasta, indexPrefix.getPath(), algo.toBwaName());
        BwaMemIndex.createIndexImageFile(indexPrefix.getPath(), imageFile);
        BwaMemIndex.deleteIndexFiles(indexPrefix);
    }

    private static void assertCanCreateOrOverwriteImageFile(String imageFile) {
        if (imageFile == null) {
            throw new IllegalArgumentException("the image file cannot be null");
        }
        File file = new File(imageFile);
        try {
            if (!file.createNewFile()) {
                if (!file.isFile() || !file.canWrite()) {
                    throw new CouldNotCreateIndexImageException(imageFile, "already exists as a non-regular or unwritable file");
                }
            } else {
                file.delete();
            }
        }
        catch (IOException ex) {
            throw new CouldNotCreateIndexImageException(imageFile, ex.getMessage(), ex);
        }
    }

    private static void assertLooksLikeIndexPrefix(String indexPrefix) {
        if (indexPrefix == null) {
            throw new IllegalArgumentException("the input index prefix cannot be null");
        }
        INDEX_FILE_EXTENSIONS.stream().map(ext -> indexPrefix + ext).forEach(file -> BwaMemIndex.assertNonEmptyReadableIndexFile(indexPrefix, file));
    }

    private static void deleteIndexFiles(File indexPrefix) {
        INDEX_FILE_EXTENSIONS.stream().map(ext -> new File(indexPrefix + ext)).forEach(File::delete);
        indexPrefix.delete();
    }

    private static File createTempIndexPrefix(String fasta) {
        File indexPrefix;
        try {
            indexPrefix = File.createTempFile("temporal-index", "");
        }
        catch (IOException ex) {
            throw new CouldNotCreateIndexException(fasta, "no-location", "failure to create a temporal file");
        }
        indexPrefix.deleteOnExit();
        INDEX_FILE_EXTENSIONS.stream().map(ext -> new File(indexPrefix + ext)).forEach(File::deleteOnExit);
        return indexPrefix;
    }

    private static void assertLooksLikeFastaFile(String fasta) {
        BwaMemIndex.resolveFastaFileExtension(fasta);
        if (!BwaMemIndex.nonEmptyReadableFile(fasta)) {
            throw new CouldNotReadReferenceException(fasta, "input file unreachable or not a file");
        }
        try (BufferedReader reader = new BufferedReader(new FileReader(fasta));){
            int c;
            int offset = 0;
            while (offset++ < 4092 && (c = reader.read()) != -1) {
                if (Character.isSpaceChar(c)) continue;
                if (c == 62) {
                    break;
                }
                throw new InvalidFileFormatException(fasta, "the file provided does not seem to be a fasta file (first non-space character in the first 4K is not '>'");
            }
        }
        catch (IOException ex) {
            throw new InvalidFileFormatException(fasta, "problems reading the content of the reference fasta file'", ex);
        }
    }

    public BwaMemIndex(String indexImageFile) {
        this.indexImageFile = indexImageFile;
        BwaMemIndex.loadNativeLibrary();
        this.assertNonEmptyReadableImageFile(indexImageFile);
        this.refCount = new AtomicInteger();
        this.indexAddress = BwaMemIndex.openIndex(indexImageFile);
        if (this.indexAddress == 0L) {
            throw new CouldNotReadImageException(indexImageFile, "unable to open bwa-mem index");
        }
        ByteBuffer refContigNamesBuf = BwaMemIndex.getRefContigNames(this.indexAddress);
        if (refContigNamesBuf == null) {
            throw new CouldNotReadImageException("unable to retrieve reference contig names from bwa-mem index");
        }
        refContigNamesBuf.order(ByteOrder.nativeOrder()).position(0).limit(refContigNamesBuf.capacity());
        int nRefContigNames = refContigNamesBuf.getInt();
        this.refContigNames = new ArrayList<String>(nRefContigNames);
        for (int idx = 0; idx < nRefContigNames; ++idx) {
            int nameLen = refContigNamesBuf.getInt();
            byte[] nameBytes = new byte[nameLen];
            refContigNamesBuf.get(nameBytes);
            this.refContigNames.add(new String(nameBytes));
        }
        BwaMemIndex.destroyByteBuffer(refContigNamesBuf);
    }

    private void assertNonEmptyReadableImageFile(String image) {
        if (!BwaMemIndex.nonEmptyReadableFile(image)) {
            throw new CouldNotReadImageException(image, "is empty or is not readable");
        }
    }

    public boolean isOpen() {
        return this.indexAddress != 0L;
    }

    public long refIndex() {
        this.refCount.incrementAndGet();
        if (this.indexAddress == 0L) {
            throw new IllegalStateException("Index image " + this.indexImageFile + " has been closed");
        }
        return this.indexAddress;
    }

    public void deRefIndex() {
        this.refCount.decrementAndGet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void close() {
        long addr = this.indexAddress;
        if (addr == 0L) return;
        Class<BwaMemIndex> clazz = BwaMemIndex.class;
        synchronized (BwaMemIndex.class) {
            addr = this.indexAddress;
            if (addr == 0L) return;
            if (this.refCount.intValue() != 0) {
                throw new IllegalStateException("Index image " + this.indexImageFile + " can't be closed:  it's in use.");
            }
            this.indexAddress = 0L;
            BwaMemIndex.destroyIndex(addr);
            // ** MonitorExit[var3_2] (shouldn't be in output)
            return;
        }
    }

    public List<String> getReferenceContigNames() {
        return this.refContigNames;
    }

    public static String getBWAVersion() {
        BwaMemIndex.loadNativeLibrary();
        return BwaMemIndex.getVersion();
    }

    ByteBuffer doAlignment(ByteBuffer seqs, ByteBuffer opts, BwaMemPairEndStats peStats) {
        ByteBuffer alignments = BwaMemIndex.createAlignments(seqs, this.indexAddress, opts, peStats);
        if (alignments == null) {
            throw new IllegalStateException("Unable to get alignments from bwa-mem index " + this.indexImageFile + ": We don't know why.");
        }
        return alignments;
    }

    private static void assertNonEmptyReadableIndexFile(String index, String fileName) {
        if (!BwaMemIndex.nonEmptyReadableFile(fileName)) {
            throw new CouldNotReadIndexException(index, "Missing bwa index file: " + fileName);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean nonEmptyReadableFile(String fileName) {
        if (fileName == null) {
            throw new IllegalArgumentException("the input file name cannot be null");
        }
        try (FileInputStream is = new FileInputStream(fileName);){
            boolean bl = is.read() != -1;
            return bl;
        }
        catch (IOException ioe) {
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static void loadNativeLibrary() {
        if (nativeLibLoaded) return;
        Class<BwaMemIndex> clazz = BwaMemIndex.class;
        synchronized (BwaMemIndex.class) {
            if (nativeLibLoaded) return;
            String libNameOverride = System.getProperty("LIBBWA_PATH");
            if (libNameOverride != null) {
                System.load(libNameOverride);
            } else {
                String libName;
                String osName = System.getProperty("os.name", "unknown").toUpperCase();
                String osArch = System.getProperty("os.arch");
                if (!"x86_64".equals(osArch) && !"amd64".equals(osArch)) {
                    throw new IllegalStateException("We have pre-built fermi-lite binaries only for x86_64 and amd64.  Your os.arch is " + osArch + ".Set property LIBBWA_PATH to point to a native library for your architecture.");
                }
                if (osName.startsWith("MAC")) {
                    libName = "/libbwa.Darwin.dylib";
                } else {
                    if (!osName.startsWith("LINUX")) throw new IllegalStateException("We have pre-built fermi-lite binaries only for Linux and Mac.  Your os.name is " + osName + ".Set property LIBBWA_PATH to point to a native library for your operating system.");
                    libName = "/libbwa.Linux.so";
                }
                try (InputStream is = BwaMemIndex.class.getResourceAsStream(libName);){
                    if (is == null) {
                        throw new IllegalStateException("Can't find resource " + libName);
                    }
                    File tmpFile = File.createTempFile("libbwa.", ".jnilib");
                    tmpFile.deleteOnExit();
                    Files.copy(is, tmpFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                    System.load(tmpFile.getPath());
                }
                catch (IOException ioe) {
                    throw new IllegalStateException("Misconfiguration: Unable to load fermi-lite native library " + libName, ioe);
                }
            }
            nativeLibLoaded = true;
            // ** MonitorExit[var0] (shouldn't be in output)
            return;
        }
    }

    private static native boolean createReferenceIndex(String var0, String var1, String var2);

    private static native boolean createIndexImageFile(String var0, String var1);

    private static native long openIndex(String var0);

    private static native int destroyIndex(long var0);

    static native ByteBuffer createDefaultOptions();

    private static native ByteBuffer getRefContigNames(long var0);

    private static native ByteBuffer createAlignments(ByteBuffer var0, long var1, ByteBuffer var3, BwaMemPairEndStats var4);

    static native void destroyByteBuffer(ByteBuffer var0);

    private static native String getVersion();

    public static enum Algorithm {
        AUTO,
        IS,
        RB2;


        public String toBwaName() {
            return this.toString().toLowerCase();
        }
    }
}

