/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.engine;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.util.Locatable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.IntervalUtils;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.clipping.ReadClipper;
import org.broadinstitute.hellbender.utils.read.GATKRead;
import org.broadinstitute.hellbender.utils.read.ReadCoordinateComparator;
import org.broadinstitute.hellbender.utils.read.ReadUtils;

public final class AssemblyRegion
implements Locatable {
    private final SAMFileHeader header;
    private final List<GATKRead> reads;
    private final SimpleInterval activeSpan;
    private final SimpleInterval paddedSpan;
    private boolean isActive;
    private boolean hasBeenFinalized;

    public AssemblyRegion(SimpleInterval activeSpan, boolean isActive, int padding, SAMFileHeader header) {
        this(activeSpan, AssemblyRegion.makePaddedSpan(activeSpan, padding, header), isActive, header);
    }

    private static SimpleInterval makePaddedSpan(SimpleInterval activeSpan, int padding, SAMFileHeader header) {
        String contig = activeSpan.getContig();
        SAMSequenceRecord sequence = header.getSequence(contig);
        if (sequence == null) {
            throw new UserException.MissingContigInSequenceDictionary(contig, header.getSequenceDictionary());
        }
        return IntervalUtils.trimIntervalToContig(contig, activeSpan.getStart() - padding, activeSpan.getEnd() + padding, sequence.getSequenceLength());
    }

    public AssemblyRegion(SimpleInterval activeSpan, SimpleInterval paddedSpan, boolean isActive, SAMFileHeader header) {
        this.header = Utils.nonNull(header);
        this.activeSpan = Utils.nonNull(activeSpan);
        this.paddedSpan = Utils.nonNull(paddedSpan);
        Utils.validateArg(activeSpan.size() > 0, () -> "Active region cannot be of zero size, but got " + activeSpan);
        Utils.validate(paddedSpan.contains(activeSpan), "Padded span must contain active span.");
        this.reads = new ArrayList<GATKRead>();
        this.isActive = isActive;
    }

    public AssemblyRegion(SimpleInterval activeSpan, int padding, SAMFileHeader header) {
        this(activeSpan, true, padding, header);
    }

    public String getContig() {
        return this.activeSpan.getContig();
    }

    public int getStart() {
        return this.activeSpan.getStart();
    }

    public int getEnd() {
        return this.activeSpan.getEnd();
    }

    public String toString() {
        return "AssemblyRegion " + this.activeSpan.toString() + " active?=" + this.isActive + " nReads=" + this.reads.size();
    }

    public boolean isActive() {
        return this.isActive;
    }

    void setIsActive(boolean value) {
        this.isActive = value;
    }

    public SimpleInterval getPaddedSpan() {
        return this.paddedSpan;
    }

    public SimpleInterval getSpan() {
        return this.activeSpan;
    }

    public List<GATKRead> getReads() {
        return Collections.unmodifiableList(new ArrayList<GATKRead>(this.reads));
    }

    public SAMFileHeader getHeader() {
        return this.header;
    }

    public AssemblyRegion trim(SimpleInterval span, int padding) {
        Utils.nonNull(span, "Active region extent cannot be null");
        Utils.validateArg(padding >= 0, "the padding size must be 0 or greater");
        SimpleInterval paddedSpan = span.expandWithinContig(padding, this.header.getSequenceDictionary());
        return this.trim(span, paddedSpan);
    }

    public AssemblyRegion trim(SimpleInterval span, SimpleInterval paddedSpan) {
        Utils.nonNull(span, "Active region extent cannot be null");
        Utils.nonNull(paddedSpan, "Active region padded span cannot be null");
        Utils.validateArg(paddedSpan.contains(span), "The requested padded span must fully contain the requested span");
        SimpleInterval newActiveSpan = this.getSpan().intersect(span);
        SimpleInterval newPaddedSpan = this.getPaddedSpan().intersect(paddedSpan);
        AssemblyRegion result = new AssemblyRegion(newActiveSpan, newPaddedSpan, this.isActive, this.header);
        List<GATKRead> trimmedReads = this.reads.stream().map(read -> {
            GATKRead clipped = ReadClipper.hardClipToRegion(read, newPaddedSpan.getStart(), newPaddedSpan.getEnd());
            return clipped;
        }).filter(read -> !read.isEmpty() && read.overlaps(result.paddedSpan)).sorted(new ReadCoordinateComparator(this.header)).collect(Collectors.toList());
        result.clearReads();
        result.addAll(trimmedReads);
        return result;
    }

    public void add(GATKRead read) {
        Utils.nonNull(read, "Read cannot be null");
        SimpleInterval readLoc = new SimpleInterval(read);
        Utils.validateArg(this.paddedSpan.overlaps(read), () -> "Read location " + readLoc + " doesn't overlap with active region padded span " + this.paddedSpan);
        if (!this.reads.isEmpty()) {
            GATKRead lastRead = this.reads.get(this.size() - 1);
            Utils.validateArg(Objects.equals(lastRead.getContig(), read.getContig()), () -> "Attempting to add a read to ActiveRegion not on the same contig as other reads: lastRead " + lastRead + " attempting to add " + read);
            Utils.validateArg(read.getStart() >= lastRead.getStart(), () -> "Attempting to add a read to ActiveRegion out of order w.r.t. other reads: lastRead " + lastRead + " at " + lastRead.getStart() + " attempting to add " + read + " at " + read.getStart());
        }
        this.reads.add(read);
    }

    public int size() {
        return this.reads.size();
    }

    public void clearReads() {
        this.reads.clear();
    }

    public void removeAll(Collection<GATKRead> readsToRemove) {
        Utils.nonNull(readsToRemove);
        this.reads.removeAll(readsToRemove);
    }

    public void addAll(Collection<GATKRead> readsToAdd) {
        Utils.nonNull(readsToAdd).forEach(r -> this.add((GATKRead)r));
    }

    private static byte[] getReference(ReferenceSequenceFile referenceReader, int padding, SimpleInterval genomeLoc) {
        Utils.nonNull(referenceReader, "referenceReader cannot be null");
        Utils.nonNull(genomeLoc, "genomeLoc cannot be null");
        Utils.validateArg(padding >= 0, () -> "padding must be a positive integer but got " + padding);
        Utils.validateArg(genomeLoc.size() > 0, () -> "GenomeLoc must have size > 0 but got " + genomeLoc);
        String contig = genomeLoc.getContig();
        SAMSequenceDictionary sequenceDictionary = referenceReader.getSequenceDictionary();
        SAMSequenceRecord sequence = sequenceDictionary.getSequence(contig);
        if (sequence == null) {
            throw new UserException.MissingContigInSequenceDictionary("Contig: " + contig + " not found in reference dictionary.\nPlease check that you are using a compatible reference for your data.\nReference Contigs: " + ReadUtils.prettyPrintSequenceRecords(sequenceDictionary));
        }
        return referenceReader.getSubsequenceAt(contig, (long)Math.max(1, genomeLoc.getStart() - padding), (long)Math.min(sequence.getSequenceLength(), genomeLoc.getEnd() + padding)).getBases();
    }

    public byte[] getAssemblyRegionReference(ReferenceSequenceFile referenceReader) {
        return this.getAssemblyRegionReference(referenceReader, 0);
    }

    public byte[] getAssemblyRegionReference(ReferenceSequenceFile referenceReader, int padding) {
        return AssemblyRegion.getReference(referenceReader, padding, this.paddedSpan);
    }

    public void setFinalized(boolean value) {
        this.hasBeenFinalized = value;
    }

    public boolean isFinalized() {
        return this.hasBeenFinalized;
    }
}

