/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.engine;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.samtools.MergingSamRecordIterator;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SamFileHeaderMerger;
import htsjdk.samtools.SamInputResource;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.util.CloseableIterator;
import htsjdk.samtools.util.IOUtil;
import java.io.IOException;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.engine.ReadsDataSource;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.IntervalUtils;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.gcs.BucketUtils;
import org.broadinstitute.hellbender.utils.iterators.SAMRecordToReadIterator;
import org.broadinstitute.hellbender.utils.iterators.SamReaderQueryingIterator;
import org.broadinstitute.hellbender.utils.read.GATKRead;
import org.broadinstitute.hellbender.utils.read.ReadConstants;

public final class ReadsPathDataSource
implements ReadsDataSource {
    private static final Logger logger = LogManager.getLogger(ReadsPathDataSource.class);
    private final Map<SamReader, CloseableIterator<SAMRecord>> readers;
    private final Map<SamReader, Path> backingPaths;
    private List<SimpleInterval> intervalsForTraversal;
    private boolean traverseUnmapped;
    private final SamFileHeaderMerger headerMerger;
    private boolean indicesAvailable;
    private boolean isClosed;

    public ReadsPathDataSource(Path samFile) {
        this(samFile != null ? Arrays.asList(samFile) : null, (SamReaderFactory)null);
    }

    public ReadsPathDataSource(List<Path> samFiles) {
        this(samFiles, (SamReaderFactory)null);
    }

    public ReadsPathDataSource(Path samPath, SamReaderFactory customSamReaderFactory) {
        this(samPath != null ? Arrays.asList(samPath) : null, customSamReaderFactory);
    }

    public ReadsPathDataSource(List<Path> samPaths, SamReaderFactory customSamReaderFactory) {
        this(samPaths, null, customSamReaderFactory, 0, 0);
    }

    public ReadsPathDataSource(List<Path> samPaths, List<Path> samIndices) {
        this(samPaths, samIndices, null, 0, 0);
    }

    public ReadsPathDataSource(List<Path> samPaths, List<Path> samIndices, SamReaderFactory customSamReaderFactory) {
        this(samPaths, samIndices, customSamReaderFactory, 0, 0);
    }

    public ReadsPathDataSource(List<Path> samPaths, List<Path> samIndices, SamReaderFactory customSamReaderFactory, int cloudPrefetchBuffer, int cloudIndexPrefetchBuffer) {
        this(samPaths, samIndices, customSamReaderFactory, BucketUtils.getPrefetchingWrapper(cloudPrefetchBuffer), BucketUtils.getPrefetchingWrapper(cloudIndexPrefetchBuffer));
    }

    public ReadsPathDataSource(List<Path> samPaths, List<Path> samIndices, SamReaderFactory customSamReaderFactory, Function<SeekableByteChannel, SeekableByteChannel> cloudWrapper, Function<SeekableByteChannel, SeekableByteChannel> cloudIndexWrapper) {
        Utils.nonNull(samPaths);
        Utils.nonEmpty(samPaths, "ReadsPathDataSource cannot be created from empty file list");
        if (samIndices != null && samPaths.size() != samIndices.size()) {
            throw new UserException(String.format("Must have the same number of BAM/CRAM/SAM paths and indices. Saw %d BAM/CRAM/SAMs but %d indices", samPaths.size(), samIndices.size()));
        }
        this.readers = new LinkedHashMap<SamReader, CloseableIterator<SAMRecord>>(samPaths.size() * 2);
        this.backingPaths = new LinkedHashMap<SamReader, Path>(samPaths.size() * 2);
        this.indicesAvailable = true;
        SamReaderFactory samReaderFactory = customSamReaderFactory == null ? SamReaderFactory.makeDefault().validationStringency(ReadConstants.DEFAULT_READ_VALIDATION_STRINGENCY) : customSamReaderFactory;
        int samCount = 0;
        for (Path samPath : samPaths) {
            SamReader reader;
            Function indexWrapper;
            try {
                IOUtil.assertFileIsReadable((Path)samPath);
            }
            catch (SAMException | IllegalArgumentException e) {
                throw new UserException.CouldNotReadInputFile(samPath.toString(), (Exception)e);
            }
            Function<SeekableByteChannel, SeekableByteChannel> wrapper = BucketUtils.isEligibleForPrefetching(samPath) ? cloudWrapper : Function.identity();
            Function<Object, Object> function = indexWrapper = samIndices != null && BucketUtils.isEligibleForPrefetching(samIndices.get(samCount)) || samIndices == null && BucketUtils.isEligibleForPrefetching(samPath) ? cloudIndexWrapper : Function.identity();
            if (samIndices == null) {
                reader = samReaderFactory.open(samPath, wrapper, indexWrapper);
            } else {
                SamInputResource samResource = SamInputResource.of((Path)samPath, wrapper);
                Path indexPath = samIndices.get(samCount);
                samResource.index(indexPath, indexWrapper);
                reader = samReaderFactory.open(samResource);
            }
            if (!reader.hasIndex()) {
                this.indicesAvailable = false;
            }
            this.readers.put(reader, null);
            this.backingPaths.put(reader, samPath);
            ++samCount;
        }
        this.headerMerger = samPaths.size() > 1 ? this.createHeaderMerger() : null;
    }

    public boolean indicesAvailable() {
        return this.indicesAvailable;
    }

    @Override
    public boolean isQueryableByInterval() {
        return this.indicesAvailable();
    }

    @Override
    public void setTraversalBounds(List<SimpleInterval> intervals, boolean traverseUnmapped) {
        this.intervalsForTraversal = intervals != null && !intervals.isEmpty() ? intervals : null;
        this.traverseUnmapped = traverseUnmapped;
        if (this.traversalIsBounded() && !this.indicesAvailable) {
            this.raiseExceptionForMissingIndex("Traversal by intervals was requested but some input files are not indexed.");
        }
    }

    @Override
    public boolean traversalIsBounded() {
        return this.intervalsForTraversal != null || this.traverseUnmapped;
    }

    private void raiseExceptionForMissingIndex(String reason) {
        String commandsToIndex = this.backingPaths.entrySet().stream().filter(f -> !((SamReader)f.getKey()).hasIndex()).map(Map.Entry::getValue).map(Path::toAbsolutePath).map(f -> "samtools index " + f).collect(Collectors.joining("\n", "\n", "\n"));
        throw new UserException(reason + "\nPlease index all input files:\n" + commandsToIndex);
    }

    @Override
    public Iterator<GATKRead> iterator() {
        logger.debug("Preparing readers for traversal");
        return this.prepareIteratorsForTraversal(this.intervalsForTraversal, this.traverseUnmapped);
    }

    @Override
    public Iterator<GATKRead> query(SimpleInterval interval) {
        if (!this.indicesAvailable) {
            this.raiseExceptionForMissingIndex("Cannot query reads data source by interval unless all files are indexed");
        }
        return this.prepareIteratorsForTraversal(Arrays.asList(interval));
    }

    @Override
    public Iterator<GATKRead> queryUnmapped() {
        if (!this.indicesAvailable) {
            this.raiseExceptionForMissingIndex("Cannot query reads data source by interval unless all files are indexed");
        }
        return this.prepareIteratorsForTraversal(null, true);
    }

    @Override
    public SAMFileHeader getHeader() {
        return this.headerMerger != null ? this.headerMerger.getMergedHeader() : this.readers.entrySet().iterator().next().getKey().getFileHeader();
    }

    private Iterator<GATKRead> prepareIteratorsForTraversal(List<SimpleInterval> queryIntervals) {
        return this.prepareIteratorsForTraversal(queryIntervals, false);
    }

    private Iterator<GATKRead> prepareIteratorsForTraversal(List<SimpleInterval> queryIntervals, boolean queryUnmapped) {
        this.closePreviousIterationsIfNecessary();
        boolean traversalIsBounded = queryIntervals != null && !queryIntervals.isEmpty() || queryUnmapped;
        for (Map.Entry<SamReader, CloseableIterator<SAMRecord>> readerEntry : this.readers.entrySet()) {
            if (traversalIsBounded) {
                readerEntry.setValue(new SamReaderQueryingIterator(readerEntry.getKey(), this.readers.size() > 1 ? this.getIntervalsOverlappingReader(readerEntry.getKey(), queryIntervals) : queryIntervals, queryUnmapped));
                continue;
            }
            readerEntry.setValue((CloseableIterator<SAMRecord>)readerEntry.getKey().iterator());
        }
        Iterator startingIterator = null;
        startingIterator = this.readers.size() == 1 ? (Iterator)this.readers.entrySet().iterator().next().getValue() : new MergingSamRecordIterator(this.headerMerger, this.readers, true);
        return new SAMRecordToReadIterator(startingIterator);
    }

    private List<SimpleInterval> getIntervalsOverlappingReader(SamReader samReader, List<SimpleInterval> queryIntervals) {
        SAMSequenceDictionary sequenceDictionary = samReader.getFileHeader().getSequenceDictionary();
        return queryIntervals.stream().filter(interval -> IntervalUtils.intervalIsOnDictionaryContig(interval, sequenceDictionary)).collect(Collectors.toList());
    }

    private SamFileHeaderMerger createHeaderMerger() {
        ArrayList<SAMFileHeader> headers = new ArrayList<SAMFileHeader>(this.readers.size());
        for (Map.Entry<SamReader, CloseableIterator<SAMRecord>> readerEntry : this.readers.entrySet()) {
            headers.add(readerEntry.getKey().getFileHeader());
        }
        SamFileHeaderMerger headerMerger = new SamFileHeaderMerger(ReadsPathDataSource.identifySortOrder(headers), headers, true);
        return headerMerger;
    }

    @VisibleForTesting
    static SAMFileHeader.SortOrder identifySortOrder(List<SAMFileHeader> headers) {
        SAMFileHeader.SortOrder order;
        Set sortOrders = headers.stream().map(SAMFileHeader::getSortOrder).collect(Collectors.toSet());
        if (sortOrders.size() == 1) {
            order = (SAMFileHeader.SortOrder)sortOrders.iterator().next();
        } else {
            order = SAMFileHeader.SortOrder.unsorted;
            logger.warn("Inputs have different sort orders. Assuming {} sorted reads for all of them.", new Object[]{order});
        }
        return order;
    }

    @Override
    public boolean supportsSerialIteration() {
        return !this.hasSAMInputs();
    }

    @Override
    public void close() {
        if (this.isClosed) {
            return;
        }
        this.isClosed = true;
        this.closePreviousIterationsIfNecessary();
        try {
            for (Map.Entry<SamReader, CloseableIterator<SAMRecord>> readerEntry : this.readers.entrySet()) {
                readerEntry.getKey().close();
            }
        }
        catch (IOException e) {
            throw new GATKException("Error closing SAMReader");
        }
    }

    boolean isClosed() {
        return this.isClosed;
    }

    private void closePreviousIterationsIfNecessary() {
        for (Map.Entry<SamReader, CloseableIterator<SAMRecord>> readerEntry : this.readers.entrySet()) {
            CloseableIterator<SAMRecord> readerIterator = readerEntry.getValue();
            if (readerIterator == null) continue;
            readerIterator.close();
            readerEntry.setValue(null);
        }
    }

    private boolean hasSAMInputs() {
        return this.readers.keySet().stream().anyMatch(r -> r.type().equals(SamReader.Type.SAM_TYPE));
    }

    @Override
    public SAMSequenceDictionary getSequenceDictionary() {
        return this.getHeader().getSequenceDictionary();
    }
}

