/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.samtools.Cigar;
import htsjdk.samtools.CigarElement;
import htsjdk.samtools.CigarOperator;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.TextCigarCodec;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Stream;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.BetaFeature;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import org.broadinstitute.hellbender.cmdline.programgroups.CoverageAnalysisProgramGroup;
import org.broadinstitute.hellbender.engine.GATKPath;
import org.broadinstitute.hellbender.engine.GATKTool;
import org.broadinstitute.hellbender.engine.ReferenceDataSource;
import org.broadinstitute.hellbender.engine.filters.ReadFilterLibrary;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.tools.spark.utils.HopscotchMap;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Tail;
import org.broadinstitute.hellbender.utils.gcs.BucketUtils;
import org.broadinstitute.hellbender.utils.read.CigarUtils;
import org.broadinstitute.hellbender.utils.read.GATKRead;
import org.broadinstitute.hellbender.utils.read.SAMFileGATKReadWriter;

@DocumentedFeature
@CommandLineProgramProperties(summary="(Experimental) Processes reads from a MITESeq or other saturation mutagenesis experiment.\nMain output is a tab-delimited text file reportNamePrefix.variantCounts.\nColumns are:\n1 - Number of times the described variant was observed\n2 - Number of molecules that covered the variant region and its flanks\n3 - Mean length of trimmed, aligned reference coverage in the observed variant molecules\n4 - The number of base calls that varied\n5 - The variant base calls as a comma-delimited string, e.g., 17:A>T says reference base A was called as T at reference position 17\n    A missing base call is represented as a hyphen.\n6 - The number of variant codons\n7 - The variant codons as a comma-delimited string, e.g., 3:AAG>CGA says the 3rd codon went from AAG to CGA\n8 - The variant amino-acids that result from the variant codons, e.g., M:K>R indicates a missense variation from Lysine to Arginine\nAll reference coordinates are 1-based.", oneLineSummary="(EXPERIMENTAL) Processes reads from a MITESeq or other saturation mutagenesis experiment.", usageExample="gatk AnalyzeSaturationMutagenesis -I input_reads.bam -R referenceGene.fasta --orf 128-1285 -O /path/to/output/and/reportNamePrefix", programGroup=CoverageAnalysisProgramGroup.class)
@BetaFeature
public final class AnalyzeSaturationMutagenesis
extends GATKTool {
    @Argument(doc="minimum quality score for analyzed portion of read", fullName="min-q")
    @VisibleForTesting
    static int minQ = 30;
    @Argument(doc="minimum size of high-quality portion of read", fullName="min-length")
    @VisibleForTesting
    static int minLength = 15;
    @Argument(doc="minimum number of wt calls flanking variant", fullName="min-flanking-length")
    @VisibleForTesting
    static int minFlankingLength = 2;
    @Argument(doc="minimum map quality for read alignment.  reads having alignments with MAPQs less than this are treated as unmapped.", fullName="min-mapq")
    private static int minMapQ = 4;
    @Argument(doc="reference interval(s) of the ORF (1-based, inclusive), for example, '134-180,214-238' (no spaces)", fullName="orf")
    private static String orfCoords;
    @Argument(doc="minimum number of observations of reported variants", fullName="min-variant-obs")
    private static long minVariantObservations;
    @Argument(doc="examine supplemental alignments to find large deletions", fullName="find-large-deletions")
    @VisibleForTesting
    static boolean findLargeDels;
    @Argument(doc="minimum length of supplemental alignment", fullName="min-alt-length")
    @VisibleForTesting
    static int minAltLength;
    @Argument(doc="codon translation (a string of 64 amino acid codes", fullName="codon-translation")
    private static String codonTranslation;
    @Argument(doc="output file prefix", fullName="output-file-prefix", shortName="O")
    private static String outputFilePrefix;
    @Argument(doc="paired mode evaluation of variants (combine mates, when possible)", fullName="paired-mode")
    private static boolean pairedMode;
    @Argument(doc="write BAM of rejected reads", fullName="write-rejected-reads")
    private static boolean writeRejectedReads;
    @VisibleForTesting
    static Reference reference;
    @VisibleForTesting
    static CodonTracker codonTracker;
    @VisibleForTesting
    static SAMFileGATKReadWriter rejectedReadsBAMWriter;
    private static final HopscotchMap<SNVCollectionCount, Long, SNVCollectionCount> variationCounts;
    private static long totalBaseCalls;
    private static final ReportTypeCounts readCounts;
    private static final ReportTypeCounts unpairedCounts;
    private static final ReportTypeCounts disjointPairCounts;
    private static final ReportTypeCounts overlappingPairCounts;
    private GATKRead read1 = null;
    private static final int UPPERCASE_MASK = 223;
    private static final byte NO_CALL = 45;
    private static final String[] LABEL_FOR_CODON_VALUE;
    private static Logger staticLogger;

    @Override
    public boolean requiresReads() {
        return true;
    }

    @Override
    public boolean requiresReference() {
        return true;
    }

    @Override
    public void onTraversalStart() {
        super.onTraversalStart();
        staticLogger = this.logger;
        SAMFileHeader header = this.getHeaderForReads();
        if (pairedMode && header != null && header.getSortOrder() == SAMFileHeader.SortOrder.coordinate) {
            throw new UserException("In paired mode the BAM cannot be coordinate sorted.  Mates must be adjacent.");
        }
        if (codonTranslation.length() != 64) {
            throw new UserException("codon-translation string must contain exactly 64 characters");
        }
        reference = new Reference(ReferenceDataSource.of(this.referenceArguments.getReferencePath()));
        codonTracker = new CodonTracker(orfCoords, reference.getRefSeq(), this.logger);
        if (writeRejectedReads) {
            rejectedReadsBAMWriter = this.createSAMWriter(new GATKPath(outputFilePrefix + ".rejected.bam"), false);
        }
    }

    @Override
    public void traverse() {
        Stream<GATKRead> reads = this.getTransformedReadStream(ReadFilterLibrary.PRIMARY_LINE);
        try {
            if (!pairedMode) {
                reads.forEach(read -> {
                    this.read1 = read;
                    AnalyzeSaturationMutagenesis.processReport(read, AnalyzeSaturationMutagenesis.getReadReport(read), unpairedCounts);
                });
            } else {
                this.read1 = null;
                reads.forEach(read -> {
                    if (!read.isPaired()) {
                        if (this.read1 != null) {
                            this.processRead1();
                        }
                        this.read1 = read;
                        AnalyzeSaturationMutagenesis.processReport(read, AnalyzeSaturationMutagenesis.getReadReport(read), unpairedCounts);
                        this.read1 = null;
                    } else if (this.read1 == null) {
                        this.read1 = read;
                    } else if (!this.read1.getName().equals(read.getName())) {
                        this.processRead1();
                        this.read1 = read;
                    } else {
                        AnalyzeSaturationMutagenesis.updateCountsForPair(this.read1, AnalyzeSaturationMutagenesis.getReadReport(this.read1), read, AnalyzeSaturationMutagenesis.getReadReport(read));
                        this.read1 = null;
                    }
                });
                if (this.read1 != null) {
                    this.processRead1();
                }
            }
        }
        catch (Exception exception) {
            throw new GATKException("Caught unexpected exception on read " + readCounts.totalCounts() + ": " + this.read1.getName(), exception);
        }
    }

    private void processRead1() {
        this.logger.warn("Read " + this.read1.getName() + " has no mate.");
        AnalyzeSaturationMutagenesis.processReport(this.read1, AnalyzeSaturationMutagenesis.getReadReport(this.read1), unpairedCounts);
    }

    @Override
    public Object onTraversalSuccess() {
        AnalyzeSaturationMutagenesis.writeVariationCounts(AnalyzeSaturationMutagenesis.getVariationEntries());
        AnalyzeSaturationMutagenesis.writeRefCoverage();
        AnalyzeSaturationMutagenesis.writeCodonCounts();
        AnalyzeSaturationMutagenesis.writeCodonFractions();
        AnalyzeSaturationMutagenesis.writeAACounts();
        AnalyzeSaturationMutagenesis.writeAAFractions();
        AnalyzeSaturationMutagenesis.writeReadCounts();
        AnalyzeSaturationMutagenesis.writeCoverageSizeHistogram();
        if (rejectedReadsBAMWriter != null) {
            rejectedReadsBAMWriter.close();
        }
        return super.onTraversalSuccess();
    }

    private static List<SNVCollectionCount> getVariationEntries() {
        long outputSize = variationCounts.stream().filter(entry -> entry.getCount() >= minVariantObservations).count();
        ArrayList<SNVCollectionCount> variationEntries = new ArrayList<SNVCollectionCount>((int)outputSize);
        for (SNVCollectionCount entry2 : variationCounts) {
            long count = entry2.getCount();
            if (count < minVariantObservations) continue;
            variationEntries.add(entry2);
        }
        variationEntries.sort(Comparator.naturalOrder());
        return variationEntries;
    }

    private static void writeVariationCounts(List<SNVCollectionCount> variationEntries) {
        int refSeqLength = reference.getRefSeqLength();
        String variantsFile = outputFilePrefix + ".variantCounts";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(variantsFile)));){
            DecimalFormat formatter = new DecimalFormat("0.0");
            for (SNVCollectionCount entry : variationEntries) {
                writer.write(Long.toString(entry.getCount()));
                writer.write(9);
                List<SNV> snvs = entry.getSNVs();
                int start = Math.max(0, snvs.get(0).getRefIndex() - minFlankingLength);
                int end = Math.min(refSeqLength, snvs.get(snvs.size() - 1).getRefIndex() + minFlankingLength);
                writer.write(Long.toString(reference.countSpanners(start, end)));
                writer.write(9);
                writer.write(formatter.format(entry.getMeanRefCoverage()));
                writer.write(9);
                writer.write(Integer.toString(snvs.size()));
                String sep = "\t";
                for (SNV snv : snvs) {
                    writer.write(sep);
                    sep = ", ";
                    writer.write(snv.toString());
                }
                AnalyzeSaturationMutagenesis.describeVariantsAsCodons(writer, snvs);
                writer.newLine();
            }
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + variantsFile, ioe);
        }
    }

    private static void describeVariantsAsCodons(BufferedWriter writer, List<SNV> snvs) throws IOException {
        int codonId;
        List<CodonVariation> codonVariations = codonTracker.encodeSNVsAsCodons(snvs);
        if (codonVariations.size() == 0) {
            writer.write("\t0");
            return;
        }
        writer.write(9);
        writer.write(Integer.toString(codonVariations.size()));
        int[] refCodonValues = codonTracker.getRefCodonValues();
        String sep = "\t";
        for (CodonVariation variation : codonVariations) {
            writer.write(sep);
            sep = ", ";
            codonId = variation.getCodonId();
            writer.write(Integer.toString(codonId + 1));
            writer.write(58);
            if (variation.isFrameshift()) {
                writer.write("FS");
                continue;
            }
            writer.write(variation.isInsertion() ? "---" : LABEL_FOR_CODON_VALUE[refCodonValues[codonId]]);
            writer.write(62);
            writer.write(variation.isDeletion() ? "---" : LABEL_FOR_CODON_VALUE[variation.getCodonValue()]);
        }
        sep = "\t";
        for (CodonVariation variation : codonVariations) {
            char toAA;
            writer.write(sep);
            sep = ", ";
            codonId = variation.getCodonId();
            if (variation.isFrameshift()) {
                writer.write("FS");
                continue;
            }
            if (variation.isInsertion()) {
                writer.write("I:->");
                writer.write(codonTranslation.charAt(variation.getCodonValue()));
                continue;
            }
            if (variation.isDeletion()) {
                writer.write("D:");
                writer.write(codonTranslation.charAt(refCodonValues[codonId]));
                writer.write(">-");
                continue;
            }
            char fromAA = codonTranslation.charAt(refCodonValues[codonId]);
            int label = fromAA == (toAA = codonTranslation.charAt(variation.getCodonValue())) ? 83 : (CodonTracker.isStop(variation.getCodonValue()) ? 78 : 77);
            writer.write(label);
            writer.write(58);
            writer.write(fromAA);
            writer.write(62);
            writer.write(toAA);
        }
        sep = "\t";
        CodonVariationGroup codonVariationGroup = null;
        for (CodonVariation variation : codonVariations) {
            if (codonVariationGroup == null) {
                if (AnalyzeSaturationMutagenesis.isSynonymous(variation, refCodonValues)) continue;
                codonVariationGroup = new CodonVariationGroup(refCodonValues, variation);
                continue;
            }
            if (codonVariationGroup.addVariation(variation)) continue;
            writer.write(sep);
            sep = ";";
            writer.write(codonVariationGroup.asHGVSString());
            codonVariationGroup = null;
            if (AnalyzeSaturationMutagenesis.isSynonymous(variation, refCodonValues)) continue;
            codonVariationGroup = new CodonVariationGroup(refCodonValues, variation);
        }
        if (codonVariationGroup != null && !codonVariationGroup.isEmpty()) {
            writer.write(sep);
            writer.write(codonVariationGroup.asHGVSString());
        }
    }

    private static boolean isSynonymous(CodonVariation variation, int[] refCodonValues) {
        return variation.getVariationType() == CodonVariationType.MODIFICATION && codonTranslation.charAt(variation.getCodonValue()) == codonTranslation.charAt(refCodonValues[variation.getCodonId()]);
    }

    private static void writeRefCoverage() {
        String refCoverageFile = outputFilePrefix + ".refCoverage";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(refCoverageFile)));){
            writer.write("RefPos\tCoverage");
            writer.newLine();
            int refPos = 1;
            for (long coverage : reference.getCoverage()) {
                writer.write(Integer.toString(refPos++));
                writer.write(9);
                writer.write(Long.toString(coverage));
                writer.newLine();
            }
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + refCoverageFile, ioe);
        }
    }

    private static void writeCodonCounts() {
        String codonCountsFile = outputFilePrefix + ".codonCounts";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(codonCountsFile)));){
            long[][] codonCounts = codonTracker.getCodonCounts();
            for (String codonLabel : LABEL_FOR_CODON_VALUE) {
                writer.write(codonLabel);
                writer.write(9);
            }
            writer.write("NFS\tFS\tTotal");
            writer.newLine();
            int nCodons = codonCounts.length;
            for (int codonId = 0; codonId != nCodons; ++codonId) {
                long[] rowCounts = codonCounts[codonId];
                long total = 0L;
                for (long count : rowCounts) {
                    writer.write(Long.toString(count));
                    writer.write(9);
                    total += count;
                }
                writer.write(Long.toString(total));
                writer.newLine();
            }
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + codonCountsFile, ioe);
        }
    }

    private static void writeCodonFractions() {
        String codonFractFile = outputFilePrefix + ".codonFractions";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(codonFractFile)));){
            long[][] codonCounts = codonTracker.getCodonCounts();
            writer.write("Codon");
            for (String codonLabel : LABEL_FOR_CODON_VALUE) {
                writer.write("   ");
                writer.write(codonLabel);
            }
            writer.write("   NFS    FS    Total");
            writer.newLine();
            int nCodons = codonCounts.length;
            for (int codonId = 0; codonId != nCodons; ++codonId) {
                writer.write(String.format("%5d", codonId + 1));
                long[] rowCounts = codonCounts[codonId];
                long total = Arrays.stream(rowCounts).sum();
                for (long count : rowCounts) {
                    writer.write(String.format("%6.2f", 100.0 * (double)count / (double)total));
                }
                writer.write(String.format("%9d", total));
                writer.newLine();
            }
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + codonFractFile, ioe);
        }
    }

    private static void writeAACounts() {
        String aaCountsFile = outputFilePrefix + ".aaCounts";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(aaCountsFile)));){
            long[][] codonCounts = codonTracker.getCodonCounts();
            int nCodons = codonCounts.length;
            for (int codonId = 0; codonId != nCodons; ++codonId) {
                Iterator<Object> iterator;
                long[] rowCounts = codonCounts[codonId];
                TreeMap<Character, Long> aaCounts = new TreeMap<Character, Long>();
                for (int codonValue = 0; codonValue != 64; ++codonValue) {
                    aaCounts.merge(Character.valueOf(codonTranslation.charAt(codonValue)), rowCounts[codonValue], Long::sum);
                }
                if (codonId == 0) {
                    String prefix = "";
                    iterator = aaCounts.keySet().iterator();
                    while (iterator.hasNext()) {
                        char chr = ((Character)iterator.next()).charValue();
                        writer.write(prefix);
                        prefix = "\t";
                        writer.write(chr);
                    }
                    writer.newLine();
                }
                String prefix = "";
                iterator = aaCounts.values().iterator();
                while (iterator.hasNext()) {
                    long count = (Long)iterator.next();
                    writer.write(prefix);
                    prefix = "\t";
                    writer.write(Long.toString(count));
                }
                writer.newLine();
            }
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + aaCountsFile, ioe);
        }
    }

    private static void writeAAFractions() {
        String aaFractFile = outputFilePrefix + ".aaFractions";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(aaFractFile)));){
            long[][] codonCounts = codonTracker.getCodonCounts();
            int nCodons = codonCounts.length;
            for (int codonId = 0; codonId != nCodons; ++codonId) {
                long[] rowCounts = codonCounts[codonId];
                TreeMap<Character, Long> aaCounts = new TreeMap<Character, Long>();
                for (int codonValue = 0; codonValue != 64; ++codonValue) {
                    aaCounts.merge(Character.valueOf(codonTranslation.charAt(codonValue)), rowCounts[codonValue], Long::sum);
                }
                if (codonId == 0) {
                    writer.write("Codon");
                    Iterator codonValue = aaCounts.keySet().iterator();
                    while (codonValue.hasNext()) {
                        char chr = ((Character)codonValue.next()).charValue();
                        writer.write("     ");
                        writer.write(chr);
                    }
                    writer.write("    Total");
                    writer.newLine();
                }
                writer.write(String.format("%5d", codonId + 1));
                long total = Arrays.stream(rowCounts).sum();
                Iterator iterator = aaCounts.values().iterator();
                while (iterator.hasNext()) {
                    long count = (Long)iterator.next();
                    writer.write(String.format("%6.2f", 100.0 * (double)count / (double)total));
                }
                writer.write(String.format("%9d", total));
                writer.newLine();
            }
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + aaFractFile, ioe);
        }
    }

    private static void writeReadCounts() {
        String readCountsFile = outputFilePrefix + ".readCounts";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(readCountsFile)));){
            DecimalFormat df = new DecimalFormat("0.000");
            long totalReads = readCounts.totalCounts();
            writer.write("Total Reads:\t" + totalReads + "\t100.000%");
            writer.newLine();
            long nUnmappedReads = readCounts.getCount(ReportType.UNMAPPED);
            writer.write(">Unmapped Reads:\t" + nUnmappedReads + "\t" + df.format(100.0 * (double)nUnmappedReads / (double)totalReads) + "%");
            writer.newLine();
            long nLowQualityReads = readCounts.getCount(ReportType.LOW_QUALITY);
            writer.write(">LowQ Reads:\t" + nLowQualityReads + "\t" + df.format(100.0 * (double)nLowQualityReads / (double)totalReads) + "%");
            writer.newLine();
            long nEvaluableReads = readCounts.getCount(ReportType.EVALUABLE);
            writer.write(">Evaluable Reads:\t" + nEvaluableReads + "\t" + df.format(100.0 * (double)nEvaluableReads / (double)totalReads) + "%");
            writer.newLine();
            long nUnpairedReads = unpairedCounts.totalCounts();
            if (nUnpairedReads > 0L) {
                writer.write(">>Unpaired reads:\t" + nUnpairedReads + "\t" + df.format(100.0 * (double)nUnpairedReads / (double)nEvaluableReads) + "%");
                writer.newLine();
                AnalyzeSaturationMutagenesis.writeReportTypeCounts(unpairedCounts, df, writer);
            }
            long nDisjointReads = disjointPairCounts.totalCounts();
            writer.write(">>Reads in disjoint pairs evaluated separately:\t" + nDisjointReads + "\t" + df.format(100.0 * (double)nDisjointReads / (double)nEvaluableReads) + "%");
            writer.newLine();
            AnalyzeSaturationMutagenesis.writeReportTypeCounts(disjointPairCounts, df, writer);
            long nOverlappingReads = 2L * overlappingPairCounts.totalCounts();
            writer.write(">>Reads in overlapping pairs evaluated together:\t" + nOverlappingReads + "\t" + df.format(100.0 * (double)nOverlappingReads / (double)nEvaluableReads) + "%");
            writer.newLine();
            AnalyzeSaturationMutagenesis.writeReportTypeCounts(overlappingPairCounts, df, writer);
            long totalBases = totalBaseCalls;
            writer.write("Total base calls:\t" + totalBases + "\t100.000%");
            writer.newLine();
            long totalCoverage = reference.getTotalCoverage();
            writer.write(">Base calls evaluated for variants:\t" + totalCoverage + "\t" + df.format(100.0 * (double)totalCoverage / (double)totalBases) + "%");
            writer.newLine();
            long totalUnevaluatedBases = totalBases - totalCoverage;
            writer.write(">Base calls unevaluated:\t" + totalUnevaluatedBases + "\t" + df.format(100.0 * (double)totalUnevaluatedBases / (double)totalBases) + "%");
            writer.newLine();
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + readCountsFile, ioe);
        }
    }

    private static void writeReportTypeCounts(ReportTypeCounts counts, DecimalFormat df, BufferedWriter writer) throws IOException {
        long totalCounts = counts.totalCounts();
        for (ReportType reportType : ReportType.values()) {
            long count = counts.getCount(reportType);
            if (count == 0L) continue;
            writer.write(">>>" + reportType.label + ":\t" + count + "\t" + df.format(100.0 * (double)count / (double)totalCounts) + "%");
            writer.newLine();
        }
    }

    private static void writeCoverageSizeHistogram() {
        long[] refCoverageSizeHistogram = reference.getCoverageSizeHistogram();
        String trimCountsFile = outputFilePrefix + ".coverageLengthCounts";
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(BucketUtils.createFile(trimCountsFile)));){
            int len;
            for (len = refCoverageSizeHistogram.length; len > 101 && refCoverageSizeHistogram[len - 1] <= 0L; --len) {
            }
            for (int idx = 1; idx != len; ++idx) {
                writer.write(Integer.toString(idx));
                writer.write(9);
                writer.write(Long.toString(refCoverageSizeHistogram[idx]));
                writer.newLine();
            }
        }
        catch (IOException ioe) {
            throw new UserException("Can't write " + trimCountsFile, ioe);
        }
    }

    @VisibleForTesting
    static ReadReport getReadReport(GATKRead read) {
        totalBaseCalls += (long)read.getLength();
        if (read.isUnmapped() || read.isDuplicate() || read.failsVendorQualityCheck() || read.getMappingQuality() < minMapQ) {
            return AnalyzeSaturationMutagenesis.rejectRead(read, ReportType.UNMAPPED);
        }
        Interval trim = AnalyzeSaturationMutagenesis.calculateTrim(read);
        if (trim.size() < minLength) {
            return AnalyzeSaturationMutagenesis.rejectRead(read, ReportType.LOW_QUALITY);
        }
        ReadReport readReport = new ReadReport(read, trim, reference.getRefSeq());
        if (readReport.getRefCoverage().isEmpty()) {
            return AnalyzeSaturationMutagenesis.rejectRead(read, ReportType.LOW_QUALITY);
        }
        readCounts.bumpCount(ReportType.EVALUABLE);
        return readReport;
    }

    private static ReadReport rejectRead(GATKRead read, ReportType reportType) {
        readCounts.bumpCount(reportType);
        if (rejectedReadsBAMWriter != null) {
            read.setAttribute("XX", reportType.attributeValue);
            rejectedReadsBAMWriter.addRead(read);
        }
        return ReadReport.NULL_REPORT;
    }

    private static Interval calculateTrim(GATKRead read) {
        return AnalyzeSaturationMutagenesis.calculateShortFragmentTrim(read, AnalyzeSaturationMutagenesis.calculateQualityTrim(read.getBaseQualitiesNoCopy()));
    }

    @VisibleForTesting
    static Interval calculateQualityTrim(byte[] quals) {
        int readStart;
        int hiQCount = 0;
        for (readStart = 0; readStart < quals.length; ++readStart) {
            if (quals[readStart] < minQ) {
                hiQCount = 0;
                continue;
            }
            if (++hiQCount == minLength) break;
        }
        if (readStart == quals.length) {
            return Interval.NULL_INTERVAL;
        }
        readStart -= minLength - 1;
        hiQCount = 0;
        for (int readEnd = quals.length - 1; readEnd >= 0; --readEnd) {
            if (quals[readEnd] < minQ) {
                hiQCount = 0;
                continue;
            }
            if (++hiQCount == minLength) break;
        }
        return new Interval(readStart, readEnd += minLength);
    }

    private static Interval calculateShortFragmentTrim(GATKRead read, Interval qualityTrim) {
        if (qualityTrim.size() < minLength) {
            return qualityTrim;
        }
        if (read.isProperlyPaired()) {
            int fragmentLength = Math.abs(read.getFragmentLength());
            if (read.isReverseStrand()) {
                int minStart = read.getLength() - fragmentLength;
                if (qualityTrim.getStart() < minStart) {
                    if (qualityTrim.getEnd() - minStart < minLength) {
                        return Interval.NULL_INTERVAL;
                    }
                    return new Interval(minStart, qualityTrim.getEnd());
                }
            } else if (fragmentLength < qualityTrim.getEnd()) {
                if (fragmentLength - qualityTrim.getStart() < minLength) {
                    return Interval.NULL_INTERVAL;
                }
                return new Interval(qualityTrim.getStart(), fragmentLength);
            }
        }
        return qualityTrim;
    }

    @VisibleForTesting
    static void updateCountsForPair(GATKRead read1, ReadReport report1, GATKRead read2, ReadReport report2) {
        if (report1.getRefCoverage().isEmpty()) {
            if (!report2.getRefCoverage().isEmpty()) {
                AnalyzeSaturationMutagenesis.processReport(read2, report2, disjointPairCounts);
            }
        } else if (report2.getRefCoverage().isEmpty()) {
            AnalyzeSaturationMutagenesis.processReport(read1, report1, disjointPairCounts);
        } else {
            int overlapEnd;
            int overlapStart = Math.max(report1.getFirstRefIndex(), report2.getFirstRefIndex());
            if (overlapStart <= (overlapEnd = Math.min(report1.getLastRefIndex(), report2.getLastRefIndex()))) {
                ReadReport combinedReport = new ReadReport(report1, report2);
                ReportType reportType = combinedReport.updateCounts(codonTracker, variationCounts, reference);
                overlappingPairCounts.bumpCount(reportType);
                if (reportType.attributeValue != null && rejectedReadsBAMWriter != null) {
                    read1.setAttribute("XX", reportType.attributeValue);
                    rejectedReadsBAMWriter.addRead(read1);
                    read2.setAttribute("XX", reportType.attributeValue);
                    rejectedReadsBAMWriter.addRead(read2);
                }
            } else {
                ReportType ignoredMate = ReportType.IGNORED_MATE;
                if (read1.isFirstOfPair()) {
                    AnalyzeSaturationMutagenesis.processReport(read1, report1, disjointPairCounts);
                    if (rejectedReadsBAMWriter != null) {
                        read2.setAttribute("XX", ignoredMate.attributeValue);
                        rejectedReadsBAMWriter.addRead(read2);
                    }
                } else {
                    AnalyzeSaturationMutagenesis.processReport(read2, report2, disjointPairCounts);
                    if (rejectedReadsBAMWriter != null) {
                        read1.setAttribute("XX", ignoredMate.attributeValue);
                        rejectedReadsBAMWriter.addRead(read1);
                    }
                }
                disjointPairCounts.bumpCount(ignoredMate);
            }
        }
    }

    private static void processReport(GATKRead read, ReadReport readReport, ReportTypeCounts counts) {
        ReportType reportType = readReport.updateCounts(codonTracker, variationCounts, reference);
        counts.bumpCount(reportType);
        if (reportType.attributeValue != null && rejectedReadsBAMWriter != null) {
            read.setAttribute("XX", reportType.attributeValue);
            rejectedReadsBAMWriter.addRead(read);
        }
    }

    static {
        minVariantObservations = 3L;
        findLargeDels = false;
        minAltLength = 15;
        codonTranslation = "KNKNTTTTRSRSIIMIQHQHPPPPRRRRLLLLEDEDAAAAGGGGVVVVXYXYSSSSXCWCLFLF";
        pairedMode = true;
        writeRejectedReads = false;
        variationCounts = new HopscotchMap(10000000);
        readCounts = new ReportTypeCounts();
        unpairedCounts = new ReportTypeCounts();
        disjointPairCounts = new ReportTypeCounts();
        overlappingPairCounts = new ReportTypeCounts();
        LABEL_FOR_CODON_VALUE = new String[]{"AAA", "AAC", "AAG", "AAT", "ACA", "ACC", "ACG", "ACT", "AGA", "AGC", "AGG", "AGT", "ATA", "ATC", "ATG", "ATT", "CAA", "CAC", "CAG", "CAT", "CCA", "CCC", "CCG", "CCT", "CGA", "CGC", "CGG", "CGT", "CTA", "CTC", "CTG", "CTT", "GAA", "GAC", "GAG", "GAT", "GCA", "GCC", "GCG", "GCT", "GGA", "GGC", "GGG", "GGT", "GTA", "GTC", "GTG", "GTT", "TAA", "TAC", "TAG", "TAT", "TCA", "TCC", "TCG", "TCT", "TGA", "TGC", "TGG", "TGT", "TTA", "TTC", "TTG", "TTT"};
    }

    @VisibleForTesting
    static final class ReadReport {
        final List<Interval> refCoverage;
        final List<SNV> snvList;
        public static ReadReport NULL_REPORT = new ReadReport(new ArrayList<Interval>(), new ArrayList<SNV>());

        public ReadReport(GATKRead read, Interval trim, byte[] refSeq) {
            this.snvList = new ArrayList<SNV>();
            this.refCoverage = new ArrayList<Interval>();
            int readStart = trim.getStart();
            int readEnd = trim.getEnd();
            Cigar cigar = findLargeDels ? ReadReport.findLargeDeletions(read) : read.getCigar();
            Iterator cigarIterator = cigar.getCigarElements().iterator();
            CigarElement cigarElement = (CigarElement)cigarIterator.next();
            CigarOperator cigarOperator = cigarElement.getOperator();
            int cigarElementCount = cigarElement.getLength();
            byte[] readSeq = read.getBasesNoCopy();
            byte[] readQuals = read.getBaseQualitiesNoCopy();
            int refIndex = read.getStart() - 1;
            int readIndex = 0;
            if (cigarOperator == CigarOperator.S) {
                refIndex -= cigarElementCount;
            }
            int refCoverageBegin = -1;
            int refCoverageEnd = -1;
            while (true) {
                if (readIndex >= readStart && refIndex >= 0) {
                    byte call;
                    if (refCoverageBegin == -1) {
                        refCoverageBegin = refIndex;
                        refCoverageEnd = refIndex;
                    }
                    if (cigarOperator == CigarOperator.D) {
                        this.snvList.add(new SNV(refIndex, refSeq[refIndex], 45, readQuals[readIndex]));
                    } else if (cigarOperator == CigarOperator.I) {
                        call = (byte)(readSeq[readIndex] & 0xDF);
                        this.snvList.add(new SNV(refIndex, 45, call, readQuals[readIndex]));
                    } else if (cigarOperator == CigarOperator.M || cigarOperator == CigarOperator.S) {
                        call = (byte)(readSeq[readIndex] & 0xDF);
                        if (call != refSeq[refIndex]) {
                            this.snvList.add(new SNV(refIndex, refSeq[refIndex], call, readQuals[readIndex]));
                        }
                        if (refIndex == refCoverageEnd) {
                            ++refCoverageEnd;
                        } else {
                            this.refCoverage.add(new Interval(refCoverageBegin, refCoverageEnd));
                            refCoverageBegin = refIndex;
                            refCoverageEnd = refIndex + 1;
                        }
                    } else {
                        throw new GATKException("unanticipated cigar operator: " + cigarOperator.toString());
                    }
                }
                if (cigarOperator != CigarOperator.D && ++readIndex == readEnd || cigarOperator != CigarOperator.I && ++refIndex == refSeq.length) break;
                if (--cigarElementCount != 0) continue;
                if (!cigarIterator.hasNext()) {
                    throw new GATKException("unexpectedly exhausted cigar iterator");
                }
                cigarElement = (CigarElement)cigarIterator.next();
                cigarOperator = cigarElement.getOperator();
                cigarElementCount = cigarElement.getLength();
            }
            if (refCoverageBegin < refCoverageEnd) {
                this.refCoverage.add(new Interval(refCoverageBegin, refCoverageEnd));
            }
        }

        public ReadReport(ReadReport report1, ReadReport report2) {
            this.refCoverage = ReadReport.combineCoverage(report1, report2);
            this.snvList = ReadReport.combineVariations(report1, report2);
        }

        public List<Interval> getRefCoverage() {
            return this.refCoverage;
        }

        public int getFirstRefIndex() {
            return this.refCoverage.get(0).getStart();
        }

        public int getLastRefIndex() {
            return this.refCoverage.get(this.refCoverage.size() - 1).getEnd();
        }

        public List<SNV> getVariations() {
            return this.snvList;
        }

        public boolean hasCleanFlanks(int minFlankingLength, int refLength) {
            return this.hasCleanLeftFlank(minFlankingLength) && this.hasCleanRightFlank(minFlankingLength, refLength);
        }

        public boolean hasCleanLeftFlank(int minFlankingLength) {
            return this.snvList.isEmpty() || Math.max(0, this.snvList.get(0).getRefIndex() - minFlankingLength) >= this.getFirstRefIndex();
        }

        public boolean hasCleanRightFlank(int minFlankingLength, int refLength) {
            return this.snvList.isEmpty() || Math.min(refLength - 1, this.snvList.get(this.snvList.size() - 1).getRefIndex() + minFlankingLength) < this.getLastRefIndex();
        }

        public ReportType updateCounts(CodonTracker codonTracker, HopscotchMap<SNVCollectionCount, Long, SNVCollectionCount> variationCounts, Reference reference) {
            List<Interval> refCoverage = this.getRefCoverage();
            if (refCoverage.isEmpty()) {
                return ReportType.LOW_QUALITY;
            }
            if (this.snvList == null) {
                return ReportType.INCONSISTENT;
            }
            if (this.snvList.stream().anyMatch(snv -> snv.getQuality() < minQ || "-ACGT".indexOf(snv.getVariantCall()) == -1)) {
                return ReportType.LOW_Q_VAR;
            }
            if (!this.hasCleanFlanks(minFlankingLength, reference.getRefSeqLength())) {
                return ReportType.NO_FLANK;
            }
            int coverage = reference.updateCoverage(refCoverage);
            Interval totalCoverage = new Interval(this.getFirstRefIndex(), this.getLastRefIndex());
            reference.updateSpan(totalCoverage);
            if (this.snvList.isEmpty()) {
                codonTracker.reportWildCodonCounts(totalCoverage);
                return ReportType.WILD_TYPE;
            }
            codonTracker.reportVariantCodonCounts(totalCoverage, codonTracker.encodeSNVsAsCodons(this.snvList));
            SNVCollectionCount newVal = new SNVCollectionCount(this.snvList, coverage);
            SNVCollectionCount oldVal = (SNVCollectionCount)variationCounts.find(newVal);
            if (oldVal != null) {
                oldVal.bumpCount(coverage);
            } else {
                variationCounts.add(newVal);
            }
            return ReportType.CALLED_VARIANT;
        }

        @VisibleForTesting
        static Cigar findLargeDeletions(GATKRead read) {
            Cigar cigar = read.getCigar();
            List elements = cigar.getCigarElements();
            int nElements = elements.size();
            if (nElements < 2) {
                return cigar;
            }
            int initialClipLength = CigarUtils.countClippedBases(cigar, Tail.LEFT);
            int finalClipLength = CigarUtils.countClippedBases(cigar, Tail.RIGHT);
            if (initialClipLength >= minAltLength || finalClipLength >= minAltLength) {
                String saTag = read.getAttributeAsString("SA");
                if (saTag == null) {
                    return cigar;
                }
                int readLength = read.getLength();
                Interval primaryReadInterval = new Interval(initialClipLength, readLength - finalClipLength);
                for (String alt : saTag.split(";")) {
                    int altRefStart;
                    int finalAltClipLength;
                    int initialAltClipLength;
                    Interval altReadInterval;
                    int overlapLength;
                    Cigar altCigar;
                    String[] fields;
                    block13: {
                        fields = alt.split(",");
                        if (fields.length != 6) {
                            staticLogger.warn("Badly formed supplemental alignment: " + alt);
                            continue;
                        }
                        try {
                            if (Integer.parseInt(fields[4]) < minMapQ) {
                            }
                            break block13;
                        }
                        catch (NumberFormatException nfe) {
                            staticLogger.warn("Can't get mapQ from supplemental alignment: " + alt);
                        }
                        continue;
                    }
                    if (!(read.isReverseStrand() ? "-" : "+").equals(fields[2])) continue;
                    try {
                        altCigar = TextCigarCodec.decode((String)fields[3]);
                    }
                    catch (IllegalArgumentException iae) {
                        staticLogger.warn("Can't parse cigar in supplemental alignment: " + alt);
                        continue;
                    }
                    List altElements = altCigar.getCigarElements();
                    int nAltElements = altElements.size();
                    if (nAltElements < 2 || Math.abs(overlapLength = primaryReadInterval.overlapLength(altReadInterval = new Interval(initialAltClipLength = CigarUtils.countClippedBases(altCigar, Tail.LEFT), readLength - (finalAltClipLength = CigarUtils.countClippedBases(altCigar, Tail.RIGHT))))) > 2) continue;
                    try {
                        altRefStart = Integer.parseInt(fields[1]) - 1;
                    }
                    catch (NumberFormatException nfe) {
                        staticLogger.warn("Can't parse starting coordinate from supplement alignment: " + alt);
                        continue;
                    }
                    if (initialClipLength < initialAltClipLength) {
                        int deletionLength = altRefStart - read.getEnd() + overlapLength;
                        if (deletionLength <= 2) continue;
                        cigar = ReadReport.replaceCigar(elements, overlapLength, deletionLength, altElements);
                        read.setCigar(cigar);
                        break;
                    }
                    int deletionLength = read.getStart() - (altRefStart + altCigar.getReferenceLength() + 1) + overlapLength;
                    if (deletionLength <= 2) continue;
                    cigar = ReadReport.replaceCigar(altElements, overlapLength, deletionLength, elements);
                    read.setCigar(cigar);
                    read.setPosition(read.getContig(), altRefStart + 1);
                    break;
                }
            }
            return cigar;
        }

        private static Cigar replaceCigar(List<CigarElement> elements1, int overlapLength, int deletionLength, List<CigarElement> elements2) {
            int nElements1 = elements1.size();
            int nElements2 = elements2.size();
            ArrayList<CigarElement> cigarElements = new ArrayList<CigarElement>(nElements1 + nElements2 - 1);
            cigarElements.addAll(elements1.subList(0, nElements1 - 1));
            cigarElements.add(new CigarElement(deletionLength, CigarOperator.D));
            if (overlapLength == 0) {
                cigarElements.addAll(elements2.subList(1, nElements2));
            } else {
                CigarElement firstMatch = elements2.get(1);
                cigarElements.add(new CigarElement(firstMatch.getLength() - overlapLength, CigarOperator.M));
                cigarElements.addAll(elements2.subList(2, nElements2));
            }
            return new Cigar(cigarElements);
        }

        private static List<Interval> combineCoverage(ReadReport report1, ReadReport report2) {
            Interval curInterval;
            List<Interval> refCoverage1 = report1.getRefCoverage();
            List<Interval> refCoverage2 = report2.getRefCoverage();
            if (refCoverage1.isEmpty()) {
                return refCoverage2;
            }
            if (refCoverage2.isEmpty()) {
                return refCoverage1;
            }
            ArrayList<Interval> combinedCoverage = new ArrayList<Interval>(refCoverage1.size() + refCoverage2.size());
            Iterator<Interval> refCoverageItr1 = refCoverage1.iterator();
            Iterator<Interval> refCoverageItr2 = refCoverage2.iterator();
            Interval refInterval1 = refCoverageItr1.next();
            Interval refInterval2 = refCoverageItr2.next();
            if (refInterval1.getStart() < refInterval2.getStart()) {
                curInterval = refInterval1;
                refInterval1 = refCoverageItr1.hasNext() ? refCoverageItr1.next() : null;
            } else {
                curInterval = refInterval2;
                Interval interval = refInterval2 = refCoverageItr2.hasNext() ? refCoverageItr2.next() : null;
            }
            while (refInterval1 != null || refInterval2 != null) {
                Interval testInterval;
                if (refInterval1 == null) {
                    testInterval = refInterval2;
                    refInterval2 = refCoverageItr2.hasNext() ? refCoverageItr2.next() : null;
                } else if (refInterval2 == null) {
                    testInterval = refInterval1;
                    refInterval1 = refCoverageItr1.hasNext() ? refCoverageItr1.next() : null;
                } else if (refInterval1.getStart() < refInterval2.getStart()) {
                    testInterval = refInterval1;
                    refInterval1 = refCoverageItr1.hasNext() ? refCoverageItr1.next() : null;
                } else {
                    testInterval = refInterval2;
                    Interval interval = refInterval2 = refCoverageItr2.hasNext() ? refCoverageItr2.next() : null;
                }
                if (curInterval.getEnd() < testInterval.getStart()) {
                    combinedCoverage.add(curInterval);
                    curInterval = testInterval;
                    continue;
                }
                curInterval = new Interval(curInterval.getStart(), Math.max(curInterval.getEnd(), testInterval.getEnd()));
            }
            combinedCoverage.add(curInterval);
            return combinedCoverage;
        }

        private static List<SNV> combineVariations(ReadReport report1, ReadReport report2) {
            SNV snv2;
            if (report1.getVariations().isEmpty()) {
                return report2.getVariations();
            }
            if (report2.getVariations().isEmpty()) {
                return report1.getVariations();
            }
            int overlapStart = Math.max(report1.getFirstRefIndex(), report2.getFirstRefIndex());
            int overlapEnd = Math.min(report1.getLastRefIndex(), report2.getLastRefIndex());
            ArrayList<SNV> combinedSNVs = new ArrayList<SNV>();
            Iterator<SNV> iterator1 = report1.getVariations().iterator();
            Iterator<SNV> iterator2 = report2.getVariations().iterator();
            SNV snv1 = iterator1.hasNext() ? iterator1.next() : null;
            SNV sNV = snv2 = iterator2.hasNext() ? iterator2.next() : null;
            while (snv1 != null || snv2 != null) {
                int refIndex;
                SNV next;
                if (snv1 == null) {
                    next = snv2;
                    snv2 = iterator2.hasNext() ? iterator2.next() : null;
                    refIndex = next.getRefIndex();
                    if (refIndex >= overlapStart && refIndex < overlapEnd) {
                        return null;
                    }
                } else if (snv2 == null) {
                    next = snv1;
                    snv1 = iterator1.hasNext() ? iterator1.next() : null;
                    refIndex = next.getRefIndex();
                    if (refIndex >= overlapStart && refIndex < overlapEnd) {
                        return null;
                    }
                } else {
                    int refIndex2;
                    int refIndex1 = snv1.getRefIndex();
                    if (refIndex1 < (refIndex2 = snv2.getRefIndex())) {
                        next = snv1;
                        SNV sNV2 = snv1 = iterator1.hasNext() ? iterator1.next() : null;
                        if (refIndex1 >= overlapStart && refIndex1 < overlapEnd) {
                            return null;
                        }
                    } else if (refIndex2 < refIndex1) {
                        next = snv2;
                        SNV sNV3 = snv2 = iterator2.hasNext() ? iterator2.next() : null;
                        if (refIndex2 >= overlapStart && refIndex2 < overlapEnd) {
                            return null;
                        }
                    } else {
                        if (!snv1.equals(snv2)) {
                            return null;
                        }
                        next = snv1.getQuality() > snv2.getQuality() ? snv1 : snv2;
                        snv1 = iterator1.hasNext() ? iterator1.next() : null;
                        snv2 = iterator2.hasNext() ? iterator2.next() : null;
                    }
                }
                combinedSNVs.add(next);
            }
            return combinedSNVs;
        }

        @VisibleForTesting
        ReadReport(List<Interval> refCoverage, List<SNV> snvList) {
            this.refCoverage = refCoverage;
            this.snvList = snvList;
        }
    }

    @VisibleForTesting
    static final class CodonTracker {
        private final byte[] refSeq;
        private final List<Interval> exonList;
        private final long[][] codonCounts;
        private final int[] refCodonValues;
        @VisibleForTesting
        static int NO_FRAME_SHIFT_CODON = -1;
        @VisibleForTesting
        static final int N_REGULAR_CODONS = 64;
        @VisibleForTesting
        static final int FRAME_PRESERVING_INDEL_INDEX = 64;
        @VisibleForTesting
        static final int FRAME_SHIFTING_INDEL_INDEX = 65;
        private static final int CODON_COUNT_ROW_SIZE = 66;

        public CodonTracker(String orfCoords, byte[] refSeq, Logger logger) {
            this.refSeq = refSeq;
            this.exonList = CodonTracker.getExons(orfCoords, refSeq.length);
            this.codonCounts = new long[this.exonList.stream().mapToInt(Interval::size).sum() / 3][];
            for (int codonId = 0; codonId != this.codonCounts.length; ++codonId) {
                this.codonCounts[codonId] = new long[66];
            }
            this.refCodonValues = CodonTracker.parseReferenceIntoCodons(refSeq, this.exonList, logger);
        }

        public int[] getRefCodonValues() {
            return this.refCodonValues;
        }

        public long[][] getCodonCounts() {
            return this.codonCounts;
        }

        public List<CodonVariation> encodeSNVsAsCodons(List<SNV> snvs) {
            int refIndex;
            ArrayList<CodonVariation> codonVariations = new ArrayList<CodonVariation>();
            Iterator<SNV> snvIterator = snvs.iterator();
            SNV snv = null;
            int orfStart = this.exonList.get(0).getStart();
            while (snvIterator.hasNext()) {
                SNV testSNV = snvIterator.next();
                int refIndex2 = testSNV.getRefIndex();
                if (refIndex2 == orfStart && testSNV.getRefCall() == 45 || !this.isExonic(refIndex2)) continue;
                snv = testSNV;
                break;
            }
            int frameShiftCodonId = this.findFrameShift(snvs);
            int lastExonEnd = this.exonList.get(this.exonList.size() - 1).getEnd();
            while (snv != null && (refIndex = snv.getRefIndex()) < lastExonEnd) {
                Interval curExon = null;
                Iterator<Interval> exonIterator = this.exonList.iterator();
                while (exonIterator.hasNext()) {
                    Interval testExon = exonIterator.next();
                    if (testExon.getStart() > refIndex || testExon.getEnd() <= refIndex) continue;
                    curExon = testExon;
                    break;
                }
                if (curExon == null) {
                    throw new GATKException("can't find current exon, even though refIndex should be exonic.");
                }
                int codonId = this.exonicBaseIndex(refIndex);
                int codonPhase = codonId % 3;
                int codonValue = this.refCodonValues[codonId /= 3];
                codonValue = codonPhase == 0 ? 0 : (codonPhase == 1 ? (codonValue >>= 4) : (codonValue >>= 2));
                int leadLag = 0;
                do {
                    boolean codonValueAltered = false;
                    boolean bumpRefIndex = false;
                    if (snv == null || snv.getRefIndex() != refIndex) {
                        codonValue = codonValue << 2 | "ACGT".indexOf(this.refSeq[refIndex]);
                        codonValueAltered = true;
                        bumpRefIndex = true;
                    } else {
                        if (snv.getVariantCall() == 45) {
                            if (codonId == frameShiftCodonId) {
                                codonVariations.add(CodonVariation.createFrameshift(codonId));
                                frameShiftCodonId = NO_FRAME_SHIFT_CODON;
                            }
                            if (--leadLag == -3) {
                                codonVariations.add(CodonVariation.createDeletion(codonId));
                                if (++codonId == this.refCodonValues.length) {
                                    return codonVariations;
                                }
                                leadLag = 0;
                            }
                            bumpRefIndex = true;
                        } else if (snv.getRefCall() == 45) {
                            ++leadLag;
                            codonValue = codonValue << 2 | "ACGT".indexOf(snv.getVariantCall());
                            codonValueAltered = true;
                        } else {
                            codonValue = codonValue << 2 | "ACGT".indexOf(snv.getVariantCall());
                            codonValueAltered = true;
                            bumpRefIndex = true;
                        }
                        snv = null;
                        while (snvIterator.hasNext()) {
                            SNV testSNV = snvIterator.next();
                            if (testSNV.getRefIndex() < lastExonEnd && !this.isExonic(testSNV.getRefIndex())) continue;
                            snv = testSNV;
                            break;
                        }
                    }
                    if (bumpRefIndex) {
                        if (++refIndex == curExon.getEnd() && exonIterator.hasNext()) {
                            curExon = exonIterator.next();
                            refIndex = curExon.getStart();
                        }
                        if (refIndex == this.refSeq.length) {
                            return codonVariations;
                        }
                    }
                    if (!codonValueAltered || ++codonPhase != 3) continue;
                    if (codonId == frameShiftCodonId) {
                        codonVariations.add(CodonVariation.createFrameshift(codonId));
                        frameShiftCodonId = NO_FRAME_SHIFT_CODON;
                    }
                    if (leadLag >= 3) {
                        codonVariations.add(CodonVariation.createInsertion(codonId, codonValue));
                        leadLag -= 3;
                        --codonId;
                    } else if (codonValue != this.refCodonValues[codonId]) {
                        codonVariations.add(CodonVariation.createModification(codonId, codonValue));
                    }
                    if (CodonTracker.isStop(codonValue)) {
                        return codonVariations;
                    }
                    if (++codonId == this.refCodonValues.length) {
                        return codonVariations;
                    }
                    codonPhase = 0;
                    codonValue = 0;
                } while (leadLag != 0 || codonPhase != 0);
            }
            return codonVariations;
        }

        public void reportVariantCodonCounts(Interval refCoverage, List<CodonVariation> variantCodons) {
            int startingCodonId = (this.exonicBaseIndex(refCoverage.getStart()) + 2) / 3;
            int endingCodonId = this.exonicBaseIndex(refCoverage.getEnd()) / 3;
            Iterator<CodonVariation> variantCodonIterator = variantCodons.iterator();
            CodonVariation codonVariation = variantCodonIterator.hasNext() ? variantCodonIterator.next() : null;
            for (int codonId = startingCodonId; codonId < endingCodonId; ++codonId) {
                while (codonVariation != null && codonVariation.getCodonId() < codonId) {
                    codonVariation = variantCodonIterator.hasNext() ? variantCodonIterator.next() : null;
                }
                if (codonVariation == null || codonVariation.getCodonId() != codonId) {
                    long[] lArray = this.codonCounts[codonId];
                    int n = this.refCodonValues[codonId];
                    lArray[n] = lArray[n] + 1L;
                    continue;
                }
                boolean framePreservingIndel = false;
                do {
                    switch (codonVariation.getVariationType()) {
                        case FRAMESHIFT: {
                            long[] lArray = this.codonCounts[codonId];
                            lArray[65] = lArray[65] + 1L;
                            break;
                        }
                        case INSERTION: 
                        case DELETION: {
                            framePreservingIndel = true;
                            break;
                        }
                        case MODIFICATION: {
                            long[] lArray = this.codonCounts[codonId];
                            int n = codonVariation.getCodonValue();
                            lArray[n] = lArray[n] + 1L;
                        }
                    }
                    CodonVariation codonVariation2 = codonVariation = variantCodonIterator.hasNext() ? variantCodonIterator.next() : null;
                } while (codonVariation != null && codonVariation.getCodonId() == codonId);
                if (!framePreservingIndel) continue;
                long[] lArray = this.codonCounts[codonId];
                lArray[64] = lArray[64] + 1L;
            }
        }

        public void reportWildCodonCounts(Interval refCoverage) {
            int startingCodonId = (this.exonicBaseIndex(refCoverage.getStart()) + 2) / 3;
            int endingCodonId = this.exonicBaseIndex(refCoverage.getEnd()) / 3;
            for (int codonId = startingCodonId; codonId < endingCodonId; ++codonId) {
                long[] lArray = this.codonCounts[codonId];
                int n = this.refCodonValues[codonId];
                lArray[n] = lArray[n] + 1L;
            }
        }

        @VisibleForTesting
        static List<Interval> getExons(String orfCoords, int refLen) {
            ArrayList<Interval> exonList = new ArrayList<Interval>();
            for (String coordPair : orfCoords.split(",")) {
                String[] coords = coordPair.split("-");
                if (coords.length != 2) {
                    throw new UserException("Can't interpret ORF as list of pairs of coords: " + orfCoords);
                }
                try {
                    int start = Integer.valueOf(coords[0]);
                    if (start < 1) {
                        throw new UserException("Coordinates of ORF are 1-based.");
                    }
                    int end = Integer.valueOf(coords[1]);
                    if (end < start) {
                        throw new UserException("Found ORF end coordinate less than start: " + orfCoords);
                    }
                    if (end > refLen) {
                        throw new UserException("Found ORF end coordinate larger than reference length: " + orfCoords);
                    }
                    Interval exon = new Interval(start - 1, end);
                    exonList.add(exon);
                }
                catch (NumberFormatException nfe) {
                    throw new UserException("Can't interpret ORF coords as integers: " + orfCoords);
                }
                for (int idx = 1; idx < exonList.size(); ++idx) {
                    if (((Interval)exonList.get(idx - 1)).getEnd() < ((Interval)exonList.get(idx)).getStart()) continue;
                    throw new UserException("ORF coordinates are not sorted: " + orfCoords);
                }
            }
            int orfLen = exonList.stream().mapToInt(Interval::size).sum();
            if (orfLen % 3 != 0) {
                throw new UserException("ORF length must be divisible by 3.");
            }
            return exonList;
        }

        @VisibleForTesting
        static int[] parseReferenceIntoCodons(byte[] refSeq, List<Interval> exonList, Logger logger) {
            int lastCodon;
            int nCodons = exonList.stream().mapToInt(Interval::size).sum() / 3;
            int[] refCodonValues = new int[nCodons];
            int codonId = 0;
            int codonPhase = 0;
            int codonValue = 0;
            for (Interval exon : exonList) {
                int exonEnd = exon.getEnd();
                for (int refIndex = exon.getStart(); refIndex != exonEnd; ++refIndex) {
                    codonValue = codonValue << 2 | "ACGT".indexOf(refSeq[refIndex]);
                    if (++codonPhase != 3) continue;
                    if (CodonTracker.isStop(codonValue) && codonId != nCodons - 1) {
                        int idx = refIndex + 1;
                        throw new UserException("There is an upstream stop codon at reference index " + idx + ".");
                    }
                    refCodonValues[codonId] = codonValue;
                    codonValue = 0;
                    codonPhase = 0;
                    ++codonId;
                }
            }
            int START_CODON = 14;
            if (refCodonValues[0] != 14) {
                logger.warn("WARNING:  Your ORF does not start with the expected ATG codon.");
            }
            if (!CodonTracker.isStop(lastCodon = refCodonValues[nCodons - 1])) {
                logger.warn("WARNING:  Your ORF does not end with the expected stop codon.");
            }
            return refCodonValues;
        }

        @VisibleForTesting
        int findFrameShift(List<SNV> snvs) {
            int codonId = NO_FRAME_SHIFT_CODON;
            int leadLag = 0;
            for (SNV snv : snvs) {
                if (!this.isExonic(snv.getRefIndex())) continue;
                if (snv.getVariantCall() == 45) {
                    if (leadLag == 0) {
                        codonId = this.exonicBaseIndex(snv.getRefIndex()) / 3;
                    }
                    if (--leadLag == -3) {
                        leadLag = 0;
                    }
                } else if (snv.getRefCall() == 45) {
                    if (leadLag == 0) {
                        codonId = this.exonicBaseIndex(snv.getRefIndex()) / 3;
                    }
                    if (++leadLag == 3) {
                        leadLag = 0;
                    }
                }
                if (leadLag != 0) continue;
                codonId = NO_FRAME_SHIFT_CODON;
            }
            return codonId;
        }

        static boolean isStop(int codonValue) {
            int STOP_OCH = 48;
            int STOP_AMB = 50;
            int STOP_OPA = 56;
            return codonValue == 48 || codonValue == 50 || codonValue == 56;
        }

        @VisibleForTesting
        boolean isExonic(int refIndex) {
            for (Interval exon : this.exonList) {
                if (exon.getStart() > refIndex) {
                    return false;
                }
                if (exon.getEnd() <= refIndex) continue;
                return true;
            }
            return false;
        }

        @VisibleForTesting
        int exonicBaseIndex(int refIndex) {
            int baseCount = 0;
            for (Interval exon : this.exonList) {
                if (refIndex >= exon.getEnd()) {
                    baseCount += exon.size();
                    continue;
                }
                if (refIndex <= exon.getStart()) break;
                baseCount += refIndex - exon.getStart();
                break;
            }
            return baseCount;
        }
    }

    @VisibleForTesting
    static final class CodonVariationGroup {
        private final int[] refCodonValues;
        private final StringBuilder altCalls;
        private int startingCodon;
        private int endingCodon;
        private boolean isFrameShift;
        private int insCount;
        private int delCount;
        private int subCount;

        public CodonVariationGroup(int[] refCodonValues, CodonVariation codonVariation) {
            this.refCodonValues = refCodonValues;
            this.altCalls = new StringBuilder();
            this.subCount = 0;
            this.delCount = 0;
            this.insCount = 0;
            switch (codonVariation.getVariationType()) {
                case FRAMESHIFT: {
                    this.isFrameShift = true;
                    break;
                }
                case INSERTION: {
                    this.insCount = 1;
                    this.altCalls.append(codonTranslation.charAt(codonVariation.getCodonValue()));
                    break;
                }
                case DELETION: {
                    this.delCount = 1;
                    break;
                }
                case MODIFICATION: {
                    this.subCount = 1;
                    this.altCalls.append(codonTranslation.charAt(codonVariation.getCodonValue()));
                }
            }
            this.startingCodon = this.endingCodon = codonVariation.getCodonId();
        }

        public boolean addVariation(CodonVariation codonVariation) {
            int codonId = codonVariation.getCodonId();
            if (codonId > this.endingCodon + 1 && !this.isFrameShift) {
                return false;
            }
            switch (codonVariation.getVariationType()) {
                case FRAMESHIFT: {
                    return false;
                }
                case INSERTION: {
                    ++this.insCount;
                    this.altCalls.append(codonTranslation.charAt(codonVariation.getCodonValue()));
                    if (!this.isFrameShift || !this.isEmpty()) break;
                    this.startingCodon = codonId;
                    break;
                }
                case DELETION: {
                    ++this.delCount;
                    break;
                }
                case MODIFICATION: {
                    char aa = codonTranslation.charAt(codonVariation.getCodonValue());
                    if (aa == codonTranslation.charAt(this.refCodonValues[codonId])) {
                        if (this.isFrameShift) {
                            if (this.isEmpty()) break;
                            this.altCalls.append(aa);
                            break;
                        }
                        return false;
                    }
                    if (this.isFrameShift && this.isEmpty()) {
                        this.startingCodon = codonId;
                    }
                    ++this.subCount;
                    this.altCalls.append(aa);
                }
            }
            this.endingCodon = codonId;
            return true;
        }

        public boolean isEmpty() {
            return this.subCount + this.insCount + this.delCount == 0;
        }

        public String toString() {
            return this.asHGVSString();
        }

        public String asHGVSString() {
            String alts = this.altCalls.toString();
            StringBuilder sb = new StringBuilder();
            if (this.isFrameShift && !alts.isEmpty()) {
                sb.append(codonTranslation.charAt(this.refCodonValues[this.startingCodon])).append(this.startingCodon + 1);
                sb.append(alts.charAt(0));
                int len = this.endingCodon - this.startingCodon + 1;
                if (len > 1) {
                    sb.append("fs*");
                    if (alts.charAt(alts.length() - 1) == 'X') {
                        sb.append(len);
                    } else {
                        sb.append('?');
                    }
                }
            } else {
                if (this.insCount != 0 && this.delCount == 0 && this.subCount == 0) {
                    --this.startingCodon;
                }
                sb.append(codonTranslation.charAt(this.refCodonValues[this.startingCodon])).append(this.startingCodon + 1);
                if (this.startingCodon != this.endingCodon) {
                    sb.append('_').append(codonTranslation.charAt(this.refCodonValues[this.endingCodon])).append(this.endingCodon + 1);
                }
                if (this.subCount == 0 && this.insCount == 0) {
                    sb.append("del");
                } else if (this.subCount == 0 && this.delCount == 0) {
                    sb.append("ins");
                } else if (this.subCount + this.delCount + this.insCount > 1) {
                    sb.append("insdel");
                }
                sb.append(alts);
            }
            return sb.toString();
        }
    }

    @VisibleForTesting
    static final class CodonVariation {
        private final int codonId;
        private final int codonValue;
        private final CodonVariationType variationType;

        private CodonVariation(int codonId, int codonValue, CodonVariationType variationType) {
            this.codonId = codonId;
            this.codonValue = codonValue;
            this.variationType = variationType;
        }

        public int getCodonId() {
            return this.codonId;
        }

        public int getCodonValue() {
            return this.codonValue;
        }

        public CodonVariationType getVariationType() {
            return this.variationType;
        }

        public boolean isFrameshift() {
            return this.variationType == CodonVariationType.FRAMESHIFT;
        }

        public boolean isInsertion() {
            return this.variationType == CodonVariationType.INSERTION;
        }

        public boolean isDeletion() {
            return this.variationType == CodonVariationType.DELETION;
        }

        public boolean isModification() {
            return this.variationType == CodonVariationType.MODIFICATION;
        }

        public boolean equals(Object obj) {
            return obj instanceof CodonVariation && this.equals((CodonVariation)obj);
        }

        public boolean equals(CodonVariation that) {
            return this.codonId == that.codonId && this.codonValue == that.codonValue && this.variationType == that.variationType;
        }

        public int hashCode() {
            return 47 * (47 * (47 * this.codonId + this.codonValue) + this.variationType.ordinal());
        }

        public static CodonVariation createFrameshift(int codonId) {
            return new CodonVariation(codonId, -1, CodonVariationType.FRAMESHIFT);
        }

        public static CodonVariation createInsertion(int codonId, int codonValue) {
            return new CodonVariation(codonId, codonValue, CodonVariationType.INSERTION);
        }

        public static CodonVariation createDeletion(int codonId) {
            return new CodonVariation(codonId, -1, CodonVariationType.DELETION);
        }

        public static CodonVariation createModification(int codonId, int codonValue) {
            return new CodonVariation(codonId, codonValue, CodonVariationType.MODIFICATION);
        }
    }

    @VisibleForTesting
    static enum CodonVariationType {
        FRAMESHIFT,
        INSERTION,
        DELETION,
        MODIFICATION;

    }

    @VisibleForTesting
    static final class SNVCollectionCount
    implements Map.Entry<SNVCollectionCount, Long>,
    Comparable<SNVCollectionCount> {
        private static final SNV[] emptyArray = new SNV[0];
        private final SNV[] snvs;
        private long count;
        private int totalRefCoverage;
        private final int hash;

        public SNVCollectionCount(List<SNV> snvs, int refCoverage) {
            this.snvs = snvs.toArray(emptyArray);
            this.count = 1L;
            this.totalRefCoverage = refCoverage;
            int hashVal = 0;
            for (SNV snv : snvs) {
                hashVal = 47 * hashVal + snv.hashCode();
            }
            this.hash = 47 * hashVal;
        }

        @Override
        public SNVCollectionCount getKey() {
            return this;
        }

        @Override
        public Long getValue() {
            return this.count;
        }

        @Override
        public Long setValue(Long value) {
            Long result = this.count;
            this.count = value;
            return result;
        }

        public List<SNV> getSNVs() {
            return Arrays.asList(this.snvs);
        }

        public long getCount() {
            return this.count;
        }

        public void bumpCount(int refCoverage) {
            ++this.count;
            this.totalRefCoverage += refCoverage;
        }

        public float getMeanRefCoverage() {
            return 1.0f * (float)this.totalRefCoverage / (float)this.count;
        }

        @Override
        public boolean equals(Object obj) {
            return obj instanceof SNVCollectionCount && this.equals((SNVCollectionCount)obj);
        }

        public boolean equals(SNVCollectionCount that) {
            return this.hash == that.hash && Arrays.equals(this.snvs, that.snvs);
        }

        @Override
        public int hashCode() {
            return this.hash;
        }

        @Override
        public int compareTo(SNVCollectionCount that) {
            int minSize = Math.min(this.snvs.length, that.snvs.length);
            for (int idx = 0; idx != minSize; ++idx) {
                int result = this.snvs[idx].compareTo(that.snvs[idx]);
                if (result == 0) continue;
                return result;
            }
            return Integer.compare(this.snvs.length, that.snvs.length);
        }
    }

    @VisibleForTesting
    static final class IntervalCounter {
        final long[][] counts;

        public IntervalCounter(int refLen) {
            this.counts = new long[refLen][];
            for (int rowIndex = 0; rowIndex != refLen; ++rowIndex) {
                this.counts[rowIndex] = new long[refLen - rowIndex + 1];
            }
        }

        public void addCount(Interval refCoverage) {
            int refStart = refCoverage.getStart();
            int refEnd = refCoverage.getEnd();
            if (refEnd > this.counts.length) {
                throw new GATKException("illegal span: [" + refStart + "," + refEnd + ")");
            }
            long[] lArray = this.counts[refStart];
            int n = refCoverage.getEnd() - refStart;
            lArray[n] = lArray[n] + 1L;
        }

        public long countSpanners(int refStart, int refEnd) {
            if (refStart < 0 || refEnd < refStart || refEnd > this.counts.length) {
                throw new GATKException("illegal span: [" + refStart + "," + refEnd + ")");
            }
            long total = 0L;
            for (int rowIndex = 0; rowIndex <= refStart; ++rowIndex) {
                long[] row = this.counts[rowIndex];
                for (int spanIndex = refEnd - rowIndex; spanIndex < row.length; ++spanIndex) {
                    total += row[spanIndex];
                }
            }
            return total;
        }
    }

    @VisibleForTesting
    static final class SNV
    implements Comparable<SNV> {
        private final int refIndex;
        private final byte refCall;
        private final byte variantCall;
        private final byte qual;

        public SNV(int refIndex, byte refCall, byte variantCall, byte qual) {
            this.refIndex = refIndex;
            this.refCall = refCall;
            this.variantCall = variantCall;
            this.qual = qual;
        }

        public int getRefIndex() {
            return this.refIndex;
        }

        public byte getRefCall() {
            return this.refCall;
        }

        public byte getVariantCall() {
            return this.variantCall;
        }

        public byte getQuality() {
            return this.qual;
        }

        public int hashCode() {
            return 47 * (47 * (47 * this.refIndex + this.refCall) + this.variantCall);
        }

        public boolean equals(Object obj) {
            return obj instanceof SNV && this.equals((SNV)obj);
        }

        public boolean equals(SNV that) {
            return this.refIndex == that.refIndex && this.refCall == that.refCall && this.variantCall == that.variantCall;
        }

        @Override
        public int compareTo(SNV that) {
            int result = Integer.compare(this.refIndex, that.refIndex);
            if (result == 0) {
                result = Byte.compare(this.refCall, that.refCall);
            }
            if (result == 0) {
                result = Byte.compare(this.variantCall, that.variantCall);
            }
            return result;
        }

        public String toString() {
            return this.refIndex + 1 + ":" + (char)this.refCall + ">" + (char)this.variantCall;
        }
    }

    @VisibleForTesting
    static final class Interval {
        private final int start;
        private final int end;
        public static Interval NULL_INTERVAL = new Interval(0, 0);

        public Interval(int start, int end) {
            if (start < 0 || end < start) {
                throw new GATKException("Illegal interval: [" + start + "," + end + ")");
            }
            this.start = start;
            this.end = end;
        }

        public int getStart() {
            return this.start;
        }

        public int getEnd() {
            return this.end;
        }

        public int size() {
            return this.end - this.start;
        }

        public int overlapLength(Interval that) {
            return Math.min(this.end, that.end) - Math.max(this.start, that.start);
        }

        public boolean equals(Object obj) {
            return obj instanceof Interval && this.equals((Interval)obj);
        }

        public boolean equals(Interval that) {
            return this.start == that.start && this.end == that.end;
        }

        public int hashCode() {
            return 47 * (47 * this.start + this.end);
        }
    }

    @VisibleForTesting
    static final class Reference {
        private final byte[] refSeq;
        private final long[] coverage;
        private final long[] coverageSizeHistogram;
        private final IntervalCounter intervalCounter;

        public Reference(ReferenceDataSource refSource) {
            SAMSequenceDictionary seqDict = refSource.getSequenceDictionary();
            if (seqDict.size() != 1) {
                throw new UserException("Expecting a reference with a single contig. The supplied reference has " + seqDict.size() + " contigs.");
            }
            SAMSequenceRecord tig0 = seqDict.getSequence(0);
            int refSeqLen = tig0.getSequenceLength();
            SimpleInterval wholeTig = new SimpleInterval(tig0.getSequenceName(), 1, refSeqLen);
            this.refSeq = Arrays.copyOf(refSource.queryAndPrefetch(wholeTig).getBases(), refSeqLen);
            int idx = 0;
            while (idx < refSeqLen) {
                int n = idx++;
                byte by = (byte)(this.refSeq[n] & 0xDF);
                this.refSeq[n] = by;
                switch (by) {
                    case 65: 
                    case 67: 
                    case 71: 
                    case 84: {
                        break;
                    }
                    default: {
                        throw new UserException("Reference sequence contains something other than A, C, G, and T.");
                    }
                }
            }
            this.coverage = new long[this.refSeq.length];
            this.coverageSizeHistogram = new long[this.refSeq.length + 1];
            this.intervalCounter = new IntervalCounter(this.refSeq.length);
        }

        @VisibleForTesting
        Reference(byte[] refSeq) {
            this.refSeq = refSeq;
            this.coverage = new long[refSeq.length];
            this.coverageSizeHistogram = new long[refSeq.length + 1];
            this.intervalCounter = new IntervalCounter(refSeq.length);
        }

        public int getRefSeqLength() {
            return this.refSeq.length;
        }

        public byte[] getRefSeq() {
            return this.refSeq;
        }

        public long getTotalCoverage() {
            return Arrays.stream(this.coverage).sum();
        }

        public long[] getCoverage() {
            return this.coverage;
        }

        public long[] getCoverageSizeHistogram() {
            return this.coverageSizeHistogram;
        }

        public long countSpanners(int refStart, int refEnd) {
            return this.intervalCounter.countSpanners(refStart, refEnd);
        }

        public void updateSpan(Interval refSpan) {
            this.intervalCounter.addCount(refSpan);
        }

        private int updateCoverage(List<Interval> refCoverageList) {
            int coverageLen = 0;
            for (Interval refInterval : refCoverageList) {
                int refIntervalStart = refInterval.getStart();
                int refIntervalEnd = refInterval.getEnd();
                coverageLen += refIntervalEnd - refIntervalStart;
                int idx = refInterval.getStart();
                while (idx != refIntervalEnd) {
                    int n = idx++;
                    this.coverage[n] = this.coverage[n] + 1L;
                }
            }
            int n = coverageLen;
            this.coverageSizeHistogram[n] = this.coverageSizeHistogram[n] + 1L;
            return coverageLen;
        }
    }

    public static final class ReportTypeCounts {
        private final long[] counts = new long[ReportType.values().length];

        public void bumpCount(ReportType reportType) {
            int n = reportType.ordinal();
            this.counts[n] = this.counts[n] + 1L;
        }

        public long getCount(ReportType reportType) {
            return this.counts[reportType.ordinal()];
        }

        public long totalCounts() {
            return Arrays.stream(this.counts).sum();
        }
    }

    static enum ReportType {
        UNMAPPED("unmapped", "Unmapped Reads"),
        LOW_QUALITY("lowQ", "LowQ Reads"),
        EVALUABLE(null, "Evaluable Reads"),
        WILD_TYPE(null, "Wild type"),
        CALLED_VARIANT(null, "Called variants"),
        INCONSISTENT("inconsistent", "Inconsistent pair"),
        IGNORED_MATE("ignoredMate", "Mate ignored"),
        LOW_Q_VAR("lowQVar", "Low quality variation"),
        NO_FLANK("noFlank", "Insufficient flank");

        public final String attributeValue;
        public final String label;
        public static final String REPORT_TYPE_ATTRIBUTE_KEY = "XX";

        private ReportType(String attributeValue, String label) {
            this.attributeValue = attributeValue;
            this.label = label;
        }
    }
}

