/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.seekablestream.SeekableStream;
import htsjdk.samtools.seekablestream.SeekableStreamFactory;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.BlockCompressedOutputStream;
import htsjdk.samtools.util.BlockCompressedStreamConstants;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.samtools.util.RuntimeIOException;
import htsjdk.tribble.AbstractFeatureReader;
import htsjdk.tribble.CloseableTribbleIterator;
import htsjdk.tribble.FeatureCodec;
import htsjdk.tribble.FeatureReader;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextComparator;
import htsjdk.variant.variantcontext.writer.Options;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterBuilder;
import htsjdk.variant.vcf.VCFCodec;
import htsjdk.variant.vcf.VCFHeader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Path;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.barclay.argparser.Advanced;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.BetaFeature;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import org.broadinstitute.hellbender.cmdline.CommandLineProgram;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.gcs.BucketUtils;
import org.broadinstitute.hellbender.utils.io.IOUtils;
import org.broadinstitute.hellbender.utils.runtime.ProgressLogger;
import picard.cmdline.programgroups.VariantManipulationProgramGroup;

@DocumentedFeature
@CommandLineProgramProperties(summary="Gathers multiple VCF files from a scatter operation into a single VCF file. Input files must be supplied in genomic order and must not have events at overlapping positions.", oneLineSummary="Gathers multiple VCF files from a scatter operation into a single VCF file", programGroup=VariantManipulationProgramGroup.class)
@BetaFeature
public final class GatherVcfsCloud
extends CommandLineProgram {
    public static final String IGNORE_SAFETY_CHECKS_LONG_NAME = "ignore-safety-checks";
    public static final String GATHER_TYPE_LONG_NAME = "gather-type";
    @Argument(fullName="input", shortName="I", doc="Input VCF file(s).")
    public List<String> inputs;
    @Argument(fullName="output", shortName="O", doc="Output VCF file.")
    public File output;
    @Argument(fullName="cloud-prefetch-buffer", shortName="CPB", doc="Size of the cloud-only prefetch buffer (in MB; 0 to disable).", optional=true)
    public int cloudPrefetchBuffer = 2;
    @Argument(fullName="create-output-variant-index", shortName="OVI", doc="If true, create a VCF index when writing a coordinate-sorted VCF file.", optional=true)
    public boolean createIndex = true;
    @Argument(fullName="gather-type", doc="Choose which method should be used to gather: BLOCK gathering is faster but onlyworks when you have both bgzipped inputs and outputs, while CONVENTIONAL gather is much slower but should work on all vcf files. AUTOMATIC chooses BLOCK if possible and CONVENTIONAL otherwise.")
    public GatherType gatherType = GatherType.AUTOMATIC;
    @Advanced
    @Argument(fullName="ignore-safety-checks", doc="Disable sanity checks to improve performance, may result in silently creating corrupted outputs data")
    public boolean ignoreSafetyChecks = false;
    @Advanced
    @Argument(fullName="disable-contig-ordering-check", doc="Don't check relative ordering of contigs when doing a conventional gather")
    public boolean disableContigOrderingCheck = false;
    private static final Logger log = LogManager.getLogger();

    @Override
    protected Object doWork() {
        log.info("Checking inputs.");
        List<Path> inputPaths = this.inputs.stream().map(IOUtils::getPath).collect(Collectors.toList());
        if (!this.ignoreSafetyChecks) {
            for (Path f : inputPaths) {
                IOUtil.assertFileIsReadable((Path)f);
            }
        }
        IOUtil.assertFileIsWritable((File)this.output);
        SAMSequenceDictionary sequenceDictionary = GatherVcfsCloud.getHeader((Path)inputPaths.get(0)).getSequenceDictionary();
        if (this.createIndex && sequenceDictionary == null) {
            throw new UserException("In order to index the resulting VCF, the input VCFs must contain ##contig lines.");
        }
        if (!this.ignoreSafetyChecks) {
            log.info("Checking file headers and first records to ensure compatibility.");
            GatherVcfsCloud.assertSameSamplesAndValidOrdering(inputPaths, this.disableContigOrderingCheck);
        }
        if (this.gatherType == GatherType.AUTOMATIC) {
            this.gatherType = GatherVcfsCloud.canBlockCopy(inputPaths, this.output) ? GatherType.BLOCK : GatherType.CONVENTIONAL;
        }
        if (this.gatherType == GatherType.BLOCK && !GatherVcfsCloud.canBlockCopy(inputPaths, this.output)) {
            throw new UserException.BadInput("Requested block copy but some files are not bgzipped, all inputs and the output must be bgzipped to block copy");
        }
        switch (this.gatherType) {
            case BLOCK: {
                log.info("Gathering by copying gzip blocks. Will not be able to validate position non-overlap of files.");
                if (this.createIndex) {
                    log.warn("Index creation not currently supported when gathering block compressed VCFs.");
                }
                GatherVcfsCloud.gatherWithBlockCopying(inputPaths, this.output, this.cloudPrefetchBuffer);
                break;
            }
            case CONVENTIONAL: {
                log.info("Gathering by conventional means.");
                GatherVcfsCloud.gatherConventionally(sequenceDictionary, this.createIndex, inputPaths, this.output, this.cloudPrefetchBuffer, this.disableContigOrderingCheck);
                break;
            }
            default: {
                throw new GATKException.ShouldNeverReachHereException("Invalid gather type: " + (Object)((Object)this.gatherType) + ".  Please report this bug to the developers.");
            }
        }
        return null;
    }

    private static boolean canBlockCopy(List<Path> inputPaths, File output) {
        return GatherVcfsCloud.areAllBlockCompressed(inputPaths) && GatherVcfsCloud.areAllBlockCompressed(CollectionUtil.makeList((Object[])new Path[]{output.toPath()}));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static VCFHeader getHeader(Path path) {
        try (FeatureReader<VariantContext> reader = GatherVcfsCloud.getReaderFromVCFUri(path, 0);){
            VCFHeader vCFHeader = (VCFHeader)reader.getHeader();
            return vCFHeader;
        }
        catch (IOException e) {
            throw new UserException.CouldNotReadInputFile(path, e.getMessage(), (Throwable)e);
        }
    }

    @VisibleForTesting
    static boolean areAllBlockCompressed(List<Path> input) {
        for (Path path : input) {
            if (path == null) {
                return false;
            }
            String pathString = path.toUri().toString();
            if (!pathString.endsWith(".bcf") && IOUtil.hasBlockCompressedExtension((String)pathString)) continue;
            return false;
        }
        return true;
    }

    private static FeatureReader<VariantContext> getReaderFromVCFUri(Path variantPath, int cloudPrefetchBuffer) {
        String variantURI = variantPath.toUri().toString();
        return AbstractFeatureReader.getFeatureReader((String)variantURI, null, (FeatureCodec)new VCFCodec(), (boolean)false, BucketUtils.getPrefetchingWrapper(cloudPrefetchBuffer), Function.identity());
    }

    private static void assertSameSamplesAndValidOrdering(List<Path> inputFiles, boolean disableContigOrderingCheck) {
        VCFHeader firstHeader = GatherVcfsCloud.getHeader(inputFiles.get(0));
        SAMSequenceDictionary dict = firstHeader.getSequenceDictionary();
        if (dict == null) {
            throw new UserException.BadInput("The first VCF specified is missing the required sequence dictionary. This is required to perform validation.  You can skip this validation using --ignore-safety-checks but ignoring safety checks can result in invalid output.");
        }
        VariantContextComparator comparator = new VariantContextComparator(dict);
        List samples = firstHeader.getGenotypeSamples();
        Path lastFile = null;
        VariantContext lastContext = null;
        for (Path f : inputFiles) {
            FeatureReader<VariantContext> in = GatherVcfsCloud.getReaderFromVCFUri(f, 0);
            VCFHeader header = (VCFHeader)in.getHeader();
            dict.assertSameDictionary(header.getSequenceDictionary());
            List theseSamples = header.getGenotypeSamples();
            if (!samples.equals(theseSamples)) {
                TreeSet s1 = new TreeSet(samples);
                TreeSet s2 = new TreeSet(theseSamples);
                s1.removeAll(theseSamples);
                s2.removeAll(samples);
                throw new IllegalArgumentException("VCFs do not have identical sample lists. Samples unique to first file: " + s1 + ". Samples unique to " + f.toUri().toString() + ": " + s2 + ".");
            }
            try (CloseableTribbleIterator variantIterator = in.iterator();){
                if (variantIterator.hasNext()) {
                    VariantContext currentContext = (VariantContext)variantIterator.next();
                    if (lastContext != null && (disableContigOrderingCheck ? lastContext.getContig().equals(currentContext.getContig()) && lastContext.getStart() >= currentContext.getStart() : comparator.compare(lastContext, currentContext) >= 0)) {
                        throw new IllegalArgumentException("First record in file " + f.toUri().toString() + " is not after first record in previous file " + lastFile.toUri().toString());
                    }
                    lastContext = currentContext;
                    lastFile = f;
                }
            }
            catch (IOException e) {
                throw new UserException.CouldNotReadInputFile(f, e.getMessage(), (Throwable)e);
            }
            CloserUtil.close(in);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static void gatherConventionally(SAMSequenceDictionary sequenceDictionary, boolean createIndex, List<Path> inputFiles, File outputFile, int cloudPrefetchBuffer, boolean disableContigOrderingCheck) {
        EnumSet<Options> options = EnumSet.copyOf(VariantContextWriterBuilder.DEFAULT_OPTIONS);
        if (createIndex) {
            options.add(Options.INDEX_ON_THE_FLY);
        } else {
            options.remove(Options.INDEX_ON_THE_FLY);
        }
        try (VariantContextWriter out = new VariantContextWriterBuilder().setOutputFile(outputFile).setReferenceDictionary(sequenceDictionary).setOptions(options).build();){
            ProgressLogger progress = new ProgressLogger(log, 10000);
            VariantContext lastContext = null;
            Path lastFile = null;
            VCFHeader firstHeader = null;
            VariantContextComparator comparator = null;
            for (Path f : inputFiles) {
                try {
                    log.debug("Gathering from file: ", new Object[]{f.toUri().toString()});
                    FeatureReader<VariantContext> variantReader = GatherVcfsCloud.getReaderFromVCFUri(f, cloudPrefetchBuffer);
                    PeekableIterator variantIterator = new PeekableIterator((Iterator)variantReader.iterator());
                    VCFHeader header = (VCFHeader)variantReader.getHeader();
                    if (firstHeader == null) {
                        firstHeader = header;
                        out.writeHeader(firstHeader);
                        comparator = new VariantContextComparator((Collection)firstHeader.getContigLines());
                    }
                    if (lastContext != null && variantIterator.hasNext()) {
                        VariantContext vc = (VariantContext)variantIterator.peek();
                        if (disableContigOrderingCheck) {
                            if (vc.getContig().equals(lastContext.getContig()) && vc.getStart() <= lastContext.getStart()) {
                                throw new IllegalStateException("First variant in file " + f.toUri().toString() + " is at start position " + vc.getStart() + " but last variant in earlier file " + lastFile.toUri().toString() + " is at start position " + lastContext.getStart());
                            }
                        } else if (comparator.compare(vc, lastContext) <= 0) {
                            throw new IllegalStateException("First variant in file " + f.toUri().toString() + " is at " + String.format("%s:%d", vc.getContig(), vc.getStart()) + " but last variant in earlier file " + lastFile.toUri().toString() + " is at " + String.format("%s:%d", lastContext.getContig(), lastContext.getStart()));
                        }
                    }
                    while (variantIterator.hasNext()) {
                        lastContext = (VariantContext)variantIterator.next();
                        out.add(lastContext);
                        progress.record(lastContext.getContig(), lastContext.getStart());
                    }
                    lastFile = f;
                    CloserUtil.close((Object)variantIterator);
                    CloserUtil.close(variantReader);
                }
                catch (IOException e) {
                    throw new UserException.CouldNotReadInputFile(f, e.getMessage(), (Throwable)e);
                    return;
                }
            }
        }
    }

    private static void gatherWithBlockCopying(List<Path> vcfs, File output, int cloudPrefetchBuffer) {
        try (FileOutputStream out = new FileOutputStream(output);){
            boolean isFirstFile = true;
            for (Path f : vcfs) {
                log.info("Gathering " + f.toUri());
                Function<SeekableByteChannel, SeekableByteChannel> prefetcher = BucketUtils.getPrefetchingWrapper(cloudPrefetchBuffer);
                SeekableStream in = SeekableStreamFactory.getInstance().getStreamFor(f.toUri().toString(), prefetcher);
                Throwable throwable = null;
                try {
                    BlockCompressedInputStream.FileTermination term = BlockCompressedInputStream.checkTermination((Path)f);
                    if (term == BlockCompressedInputStream.FileTermination.DEFECTIVE) {
                        throw new UserException.MalformedFile(f.toUri() + " does not have a valid GZIP block at the end of the file.");
                    }
                    if (!isFirstFile) {
                        BlockCompressedInputStream blockIn = new BlockCompressedInputStream((InputStream)in, false);
                        boolean lastByteNewline = true;
                        int firstNonHeaderByteIndex = -1;
                        while (blockIn.available() > 0) {
                            int blockLength = blockIn.available();
                            byte[] blockContents = new byte[blockLength];
                            int read = blockIn.read(blockContents);
                            Utils.validate(blockLength > 0 && read == blockLength, "Could not read available bytes from BlockCompressedInputStream.");
                            firstNonHeaderByteIndex = -1;
                            for (int i = 0; i < read; ++i) {
                                boolean thisByteNewline;
                                byte b = blockContents[i];
                                boolean bl = thisByteNewline = b == 10 || b == 13;
                                if (lastByteNewline && !thisByteNewline && b != 35) {
                                    firstNonHeaderByteIndex = i;
                                    break;
                                }
                                lastByteNewline = thisByteNewline;
                            }
                            if (firstNonHeaderByteIndex < 0) continue;
                            BlockCompressedOutputStream blockOut = new BlockCompressedOutputStream((OutputStream)out, (Path)null);
                            blockOut.write(blockContents, firstNonHeaderByteIndex, blockContents.length - firstNonHeaderByteIndex);
                            blockOut.flush();
                            break;
                        }
                        if (firstNonHeaderByteIndex == -1) {
                            log.warn("Scanned the entire file " + f.toUri().toString() + " and found no variants");
                        }
                    }
                    long currentPos = in.position();
                    long length = in.length();
                    long skipLast = term == BlockCompressedInputStream.FileTermination.HAS_TERMINATOR_BLOCK ? (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length : 0L;
                    long bytesToWrite = length - skipLast - currentPos;
                    IOUtil.transferByStream((InputStream)in, (OutputStream)out, (long)bytesToWrite);
                    isFirstFile = false;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (in == null) continue;
                    if (throwable != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    in.close();
                }
            }
            out.write(BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK);
        }
        catch (IOException ioe) {
            throw new RuntimeIOException((Throwable)ioe);
        }
    }

    public static enum GatherType {
        BLOCK,
        CONVENTIONAL,
        AUTOMATIC;

    }
}

