/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.copynumber.arguments;

import com.google.common.collect.Ordering;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.util.Locatable;
import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.OptionalInt;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.cmdline.argumentcollections.IntervalArgumentCollection;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.tools.copynumber.formats.collections.AbstractLocatableCollection;
import org.broadinstitute.hellbender.tools.copynumber.formats.collections.AnnotatedIntervalCollection;
import org.broadinstitute.hellbender.tools.copynumber.formats.collections.SimpleCountCollection;
import org.broadinstitute.hellbender.tools.copynumber.formats.collections.SimpleIntervalCollection;
import org.broadinstitute.hellbender.tools.copynumber.formats.metadata.LocatableMetadata;
import org.broadinstitute.hellbender.tools.copynumber.formats.metadata.SampleLocatableMetadata;
import org.broadinstitute.hellbender.tools.copynumber.formats.metadata.SimpleLocatableMetadata;
import org.broadinstitute.hellbender.tools.copynumber.formats.records.AnnotatedInterval;
import org.broadinstitute.hellbender.utils.IntervalMergingRule;
import org.broadinstitute.hellbender.utils.IntervalSetRule;
import org.broadinstitute.hellbender.utils.IntervalUtils;
import org.broadinstitute.hellbender.utils.SequenceDictionaryUtils;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.gcs.BucketUtils;
import org.broadinstitute.hellbender.utils.io.IOUtils;

public final class CopyNumberArgumentValidationUtils {
    private CopyNumberArgumentValidationUtils() {
    }

    public static void validateIntervalArgumentCollection(IntervalArgumentCollection intervalArgumentCollection) {
        Utils.validateArg(intervalArgumentCollection.getIntervalSetRule() == IntervalSetRule.UNION, "Interval set rule must be set to UNION.");
        Utils.validateArg(intervalArgumentCollection.getIntervalExclusionPadding() == 0, "Interval exclusion padding must be set to 0.");
        Utils.validateArg(intervalArgumentCollection.getIntervalPadding() == 0, "Interval padding must be set to 0.");
        Utils.validateArg(intervalArgumentCollection.getIntervalMergingRule() == IntervalMergingRule.OVERLAPPING_ONLY, "Interval merging rule must be set to OVERLAPPING_ONLY.");
    }

    public static <T extends Locatable> void validateIntervals(List<T> intervals, SAMSequenceDictionary sequenceDictionary) {
        Utils.nonNull(intervals);
        Utils.nonNull(sequenceDictionary);
        Utils.validateArg(intervals.stream().allMatch(i -> IntervalUtils.intervalIsOnDictionaryContig(new SimpleInterval((Locatable)i), sequenceDictionary)), "Records contained at least one interval that did not validate against the sequence dictionary.");
        if (!Ordering.from(IntervalUtils.getDictionaryOrderComparator(sequenceDictionary)).isStrictlyOrdered(intervals)) {
            throw new IllegalArgumentException("Records were not strictly sorted in dictionary order.");
        }
        OptionalInt failureIndex = IntStream.range(1, intervals.size()).filter(i -> IntervalUtils.overlaps((Locatable)intervals.get(i - 1), (Locatable)intervals.get(i))).findFirst();
        if (failureIndex.isPresent()) {
            int index = failureIndex.getAsInt();
            throw new IllegalArgumentException(String.format("Records contain at least two overlapping intervals: %s and %s", intervals.get(index - 1), intervals.get(index)));
        }
    }

    public static boolean isSameDictionary(SAMSequenceDictionary dictionary1, SAMSequenceDictionary dictionary2) {
        Utils.nonNull(dictionary1);
        Utils.nonNull(dictionary2);
        if (dictionary1 == dictionary2) {
            return true;
        }
        boolean checkContigOrdering = true;
        return SequenceDictionaryUtils.compareDictionaries(dictionary1, dictionary2, true) == SequenceDictionaryUtils.SequenceDictionaryCompatibility.IDENTICAL;
    }

    public static SimpleIntervalCollection resolveIntervals(String readCountPath, IntervalArgumentCollection intervalArgumentCollection, Logger logger) {
        IOUtils.assertFileIsReadable(IOUtils.getPath(readCountPath));
        Utils.nonNull(intervalArgumentCollection);
        Utils.nonNull(logger);
        if (intervalArgumentCollection.intervalsSpecified()) {
            logger.info("Intervals specified...");
            CopyNumberArgumentValidationUtils.validateIntervalArgumentCollection(intervalArgumentCollection);
        } else {
            logger.info(String.format("Retrieving intervals from read-count file (%s)...", readCountPath));
        }
        SimpleCountCollection readCounts = BucketUtils.isGcsUrl(readCountPath) ? SimpleCountCollection.readFromGCS(readCountPath) : SimpleCountCollection.read(new File(readCountPath));
        SAMSequenceDictionary sequenceDictionary = ((SampleLocatableMetadata)readCounts.getMetadata()).getSequenceDictionary();
        SimpleLocatableMetadata metadata = new SimpleLocatableMetadata(sequenceDictionary);
        List<SimpleInterval> intervals = intervalArgumentCollection.intervalsSpecified() ? intervalArgumentCollection.getIntervals(sequenceDictionary) : readCounts.getIntervals();
        return new SimpleIntervalCollection(metadata, intervals);
    }

    public static Stream<SimpleCountCollection> streamOfSubsettedAndValidatedReadCounts(List<String> inputReadCountPaths, SimpleIntervalCollection specifiedIntervals, Logger logger) {
        Utils.nonEmpty(inputReadCountPaths);
        Utils.nonNull(specifiedIntervals);
        Utils.nonNull(logger);
        int numSamples = inputReadCountPaths.size();
        HashSet intervalSubset = new HashSet(specifiedIntervals.getRecords());
        List<SimpleInterval> mergedIntervalSubset = IntervalUtils.getIntervalsWithFlanks(specifiedIntervals.getRecords(), 0, ((LocatableMetadata)specifiedIntervals.getMetadata()).getSequenceDictionary());
        return IntStream.range(0, inputReadCountPaths.size()).boxed().map(sampleIndex -> {
            SimpleCountCollection subsetReadCounts;
            String inputReadCountPath = (String)inputReadCountPaths.get((int)sampleIndex);
            logger.info(String.format("Aggregating read-count file %s (%d / %d)", inputReadCountPath, sampleIndex + 1, numSamples));
            SimpleCountCollection simpleCountCollection = subsetReadCounts = BucketUtils.isGcsUrl(inputReadCountPath) ? SimpleCountCollection.readOverlappingSubsetFromGCS(inputReadCountPath, mergedIntervalSubset) : SimpleCountCollection.readAndSubset(new File(inputReadCountPath), intervalSubset);
            if (!CopyNumberArgumentValidationUtils.isSameDictionary(((SampleLocatableMetadata)subsetReadCounts.getMetadata()).getSequenceDictionary(), ((LocatableMetadata)specifiedIntervals.getMetadata()).getSequenceDictionary())) {
                logger.warn("Sequence dictionary for read-count file {} does not match that in other read-count files.", new Object[]{inputReadCountPath});
            }
            Utils.validateArg(subsetReadCounts.size() == intervalSubset.size(), String.format("Intervals for read-count file %s do not contain all specified intervals.", inputReadCountPath));
            return subsetReadCounts;
        });
    }

    public static AnnotatedIntervalCollection validateAnnotatedIntervals(File annotatedIntervalsFile, AbstractLocatableCollection<?, ?> locatableCollection, Logger logger) {
        Utils.nonNull(locatableCollection);
        Utils.nonNull(logger);
        if (annotatedIntervalsFile == null) {
            logger.info("No annotated intervals were provided...");
            return null;
        }
        logger.info("Reading and validating annotated intervals...");
        AnnotatedIntervalCollection annotatedIntervals = new AnnotatedIntervalCollection(annotatedIntervalsFile);
        SAMSequenceDictionary sequenceDictionary = ((LocatableMetadata)locatableCollection.getMetadata()).getSequenceDictionary();
        if (!CopyNumberArgumentValidationUtils.isSameDictionary(((LocatableMetadata)annotatedIntervals.getMetadata()).getSequenceDictionary(), sequenceDictionary)) {
            logger.warn("Sequence dictionary in annotated-intervals file does not match the master sequence dictionary.");
        }
        Utils.validateArg(annotatedIntervals.getIntervals().equals(locatableCollection.getIntervals()), "Annotated intervals do not match provided intervals.");
        return annotatedIntervals;
    }

    public static AnnotatedIntervalCollection validateAnnotatedIntervalsSubset(File annotatedIntervalsFile, AbstractLocatableCollection<?, ?> locatableCollection, Logger logger) {
        Utils.nonNull(locatableCollection);
        Utils.nonNull(logger);
        if (annotatedIntervalsFile == null) {
            logger.info("No annotated intervals were provided...");
            return null;
        }
        logger.info("Reading and validating annotated intervals...");
        IOUtils.canReadFile(annotatedIntervalsFile);
        AnnotatedIntervalCollection annotatedIntervals = new AnnotatedIntervalCollection(annotatedIntervalsFile);
        SAMSequenceDictionary sequenceDictionary = ((LocatableMetadata)locatableCollection.getMetadata()).getSequenceDictionary();
        if (!CopyNumberArgumentValidationUtils.isSameDictionary(((LocatableMetadata)annotatedIntervals.getMetadata()).getSequenceDictionary(), sequenceDictionary)) {
            logger.warn("Sequence dictionary in annotated-intervals file does not match the master sequence dictionary.");
        }
        HashSet<SimpleInterval> intervalsSubset = new HashSet<SimpleInterval>(locatableCollection.getIntervals());
        List<AnnotatedInterval> subsetAnnotatedIntervals = annotatedIntervals.getRecords().stream().filter(i -> intervalsSubset.contains(i.getInterval())).collect(Collectors.toList());
        Utils.validateArg(subsetAnnotatedIntervals.size() == intervalsSubset.size(), "Annotated intervals do not contain all specified intervals.");
        return new AnnotatedIntervalCollection((LocatableMetadata)locatableCollection.getMetadata(), subsetAnnotatedIntervals);
    }

    public static void validateInputs(File ... inputs) {
        if (inputs != null) {
            for (File input : inputs) {
                if (input == null) continue;
                if (input.isFile()) {
                    IOUtils.canReadFile(input);
                    continue;
                }
                if (!input.isDirectory() || input.canRead()) continue;
                throw new UserException.CouldNotReadInputFile(input.getAbsolutePath());
            }
        }
    }

    public static void validateInputs(String ... inputs) {
        if (inputs != null) {
            for (String input : inputs) {
                if (input == null) continue;
                IOUtils.assertFileIsReadable(IOUtils.getPath(input));
            }
        }
    }

    public static void validateOutputFiles(File ... outputFiles) {
        Utils.nonNull(outputFiles);
        for (File outputFile : outputFiles) {
            Utils.nonNull(outputFile);
            if ((!outputFile.exists() || outputFile.canWrite()) && (outputFile.exists() || outputFile.getAbsoluteFile().getParentFile().canWrite())) continue;
            throw new UserException.CouldNotCreateOutputFile(outputFile.getAbsolutePath(), ": The output file is not writeable.");
        }
    }

    public static void validateAndPrepareOutputDirectories(File ... outputDirectories) {
        Utils.nonNull(outputDirectories);
        for (File outputDirectory : outputDirectories) {
            Utils.nonNull(outputDirectory);
            if (outputDirectory.exists()) {
                if (outputDirectory.canWrite()) continue;
                throw new UserException.CouldNotCreateOutputFile(outputDirectory.getAbsolutePath(), ": The output directory is not writeable.");
            }
            try {
                IOUtils.createDirectory(outputDirectory.getAbsolutePath());
            }
            catch (IOException e) {
                throw new UserException.CouldNotCreateOutputFile(outputDirectory.getAbsolutePath(), ": The output directory does not exist and could not be created.");
            }
        }
    }

    public static String getCanonicalPath(File file) {
        Utils.nonNull(file);
        try {
            return file.getCanonicalPath();
        }
        catch (IOException e) {
            throw new UserException.BadInput(String.format("Could not resolve a canonical file path: %s", file));
        }
    }

    public static String getCanonicalPath(String filename) {
        Utils.nonEmpty(filename);
        return CopyNumberArgumentValidationUtils.getCanonicalPath(new File(filename));
    }

    public static String addTrailingSlashIfNecessary(String outputDir) {
        Utils.nonEmpty(outputDir);
        return outputDir.endsWith(File.separator) ? outputDir : outputDir + File.separator;
    }
}

