/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.copynumber.plotting;

import htsjdk.samtools.SAMSequenceDictionary;
import java.io.File;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import org.broadinstitute.hellbender.cmdline.CommandLineProgram;
import org.broadinstitute.hellbender.cmdline.programgroups.CopyNumberProgramGroup;
import org.broadinstitute.hellbender.tools.copynumber.arguments.CopyNumberArgumentValidationUtils;
import org.broadinstitute.hellbender.tools.copynumber.formats.collections.AllelicCountCollection;
import org.broadinstitute.hellbender.tools.copynumber.formats.collections.CopyRatioCollection;
import org.broadinstitute.hellbender.tools.copynumber.formats.collections.ModeledSegmentCollection;
import org.broadinstitute.hellbender.tools.copynumber.formats.metadata.SampleLocatableMetadata;
import org.broadinstitute.hellbender.tools.copynumber.formats.records.AllelicCount;
import org.broadinstitute.hellbender.tools.copynumber.formats.records.CopyRatio;
import org.broadinstitute.hellbender.tools.copynumber.formats.records.ModeledSegment;
import org.broadinstitute.hellbender.tools.copynumber.plotting.PlottingUtils;
import org.broadinstitute.hellbender.utils.R.RScriptExecutor;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.io.Resource;
import org.broadinstitute.hellbender.utils.reference.ReferenceUtils;

@CommandLineProgramProperties(summary="Creates plots of denoised and segmented copy-ratio and minor-allele-fraction estimates", oneLineSummary="Creates plots of denoised and segmented copy-ratio and minor-allele-fraction estimates", programGroup=CopyNumberProgramGroup.class)
@DocumentedFeature
public final class PlotModeledSegments
extends CommandLineProgram {
    private static final String PLOT_MODELED_SEGMENTS_R_SCRIPT = "PlotModeledSegments.R";
    private static final String MODELED_SEGMENTS_PLOT_FILE_SUFFIX = ".modeled.png";
    @Argument(doc="Input file containing denoised copy ratios (output of DenoiseReadCounts).", fullName="denoised-copy-ratios", optional=true)
    private File inputDenoisedCopyRatiosFile;
    @Argument(doc="Input file containing allelic counts at heterozygous sites (.hets.tsv output of ModelSegments).", fullName="allelic-counts", optional=true)
    private File inputAllelicCountsFile;
    @Argument(doc="Input file containing modeled segments (output of ModelSegments).", fullName="segments")
    private File inputModeledSegmentsFile;
    @Argument(doc="File containing a sequence dictionary, which specifies the contigs to be plotted and their relative lengths. The sequence dictionary must be a subset of those contained in other input files. Contigs will be plotted in the order given. Contig names should not include the string \"CONTIG_DELIMITER\". The tool only considers contigs in the given dictionary for plotting, and data for contigs absent in the dictionary generate only a warning. In other words, you may modify a reference dictionary for use with this tool to include only contigs for which plotting is desired, and sort the contigs to the order in which the plots should display the contigs.", fullName="sequence-dictionary", shortName="sequence-dictionary")
    private File inputSequenceDictionaryFile;
    @Argument(doc="Threshold length (in bp) for contigs to be plotted. Contigs with lengths less than this threshold will not be plotted. This can be used to filter out mitochondrial contigs, unlocalized contigs, etc.", fullName="minimum-contig-length", minValue=0.0, optional=true)
    private int minContigLength = 1000000;
    @Argument(doc="Maximum copy ratio to be plotted. If Infinity, the maximum copy ratio will be automatically determined.", fullName="maximum-copy-ratio", minValue=0.0, optional=true)
    private double maxCopyRatio = 4.0;
    @Argument(doc="Point size to use for plotting copy-ratio points.", fullName="point-size-copy-ratio", minValue=0.0, optional=true)
    private double pointSizeCopyRatio = 0.2;
    @Argument(doc="Point size to use for plotting allele-fraction points.", fullName="point-size-allele-fraction", minValue=0.0, optional=true)
    private double pointSizeAlleleFraction = 0.4;
    @Argument(doc="Prefix for output filenames.", fullName="output-prefix")
    private String outputPrefix;
    @Argument(doc="Output directory.  This will be created if it does not exist.", fullName="output", shortName="O")
    private File outputDir;
    private CopyRatioCollection denoisedCopyRatios;
    private AllelicCountCollection allelicCounts;
    private ModeledSegmentCollection modeledSegments;

    @Override
    protected Object doWork() {
        this.validateArguments();
        this.logger.info("Reading and validating input files...");
        this.denoisedCopyRatios = this.inputDenoisedCopyRatiosFile == null ? null : new CopyRatioCollection(this.inputDenoisedCopyRatiosFile);
        this.allelicCounts = this.inputAllelicCountsFile == null ? null : new AllelicCountCollection(this.inputAllelicCountsFile);
        this.modeledSegments = new ModeledSegmentCollection(this.inputModeledSegmentsFile);
        String sampleName = this.getSampleName();
        this.validateNumPointsPerContig();
        SAMSequenceDictionary sequenceDictionary = ((SampleLocatableMetadata)this.modeledSegments.getMetadata()).getSequenceDictionary();
        SAMSequenceDictionary sequenceDictionaryToPlot = ReferenceUtils.loadFastaDictionary(this.inputSequenceDictionaryFile);
        if (this.denoisedCopyRatios != null) {
            Utils.validateArg(CopyNumberArgumentValidationUtils.isSameDictionary(((SampleLocatableMetadata)this.denoisedCopyRatios.getMetadata()).getSequenceDictionary(), sequenceDictionary), "Sequence dictionary contained in denoised-copy-ratios file does not match that contained in other input files.");
        }
        if (this.allelicCounts != null) {
            Utils.validateArg(CopyNumberArgumentValidationUtils.isSameDictionary(((SampleLocatableMetadata)this.allelicCounts.getMetadata()).getSequenceDictionary(), sequenceDictionary), "Sequence dictionary contained in allelic-counts file does not match that contained in other input files.");
        }
        PlottingUtils.validateSequenceDictionarySubset(sequenceDictionary, sequenceDictionaryToPlot);
        Map<String, Integer> contigLengthMap = PlottingUtils.getContigLengthMap(sequenceDictionaryToPlot, this.minContigLength, this.logger);
        PlottingUtils.validateContigs(contigLengthMap, this.denoisedCopyRatios, this.inputDenoisedCopyRatiosFile, this.logger);
        PlottingUtils.validateContigs(contigLengthMap, this.allelicCounts, this.inputAllelicCountsFile, this.logger);
        PlottingUtils.validateContigs(contigLengthMap, this.modeledSegments, this.inputModeledSegmentsFile, this.logger);
        ArrayList<String> contigNames = new ArrayList<String>(contigLengthMap.keySet());
        ArrayList<Integer> contigLengths = new ArrayList<Integer>(contigLengthMap.values());
        File outputFile = new File(this.outputDir, this.outputPrefix + MODELED_SEGMENTS_PLOT_FILE_SUFFIX);
        this.logger.info(String.format("Writing plot to %s...", outputFile.getAbsolutePath()));
        this.writeModeledSegmentsPlot(sampleName, contigNames, contigLengths, outputFile);
        this.logger.info(String.format("%s complete.", this.getClass().getSimpleName()));
        return null;
    }

    private void validateArguments() {
        Utils.validateArg(this.inputDenoisedCopyRatiosFile != null || this.inputAllelicCountsFile != null, "Must provide at least a denoised-copy-ratios file or an allelic-counts file.");
        CopyNumberArgumentValidationUtils.validateInputs(this.inputDenoisedCopyRatiosFile, this.inputAllelicCountsFile, this.inputModeledSegmentsFile, this.inputSequenceDictionaryFile);
        Utils.nonEmpty(this.outputPrefix);
        CopyNumberArgumentValidationUtils.validateAndPrepareOutputDirectories(this.outputDir);
    }

    private String getSampleName() {
        if (this.inputDenoisedCopyRatiosFile != null) {
            Utils.validateArg(((SampleLocatableMetadata)this.denoisedCopyRatios.getMetadata()).equals(this.modeledSegments.getMetadata()), "Metadata in input files do not all match.");
        }
        if (this.inputAllelicCountsFile != null) {
            Utils.validateArg(((SampleLocatableMetadata)this.allelicCounts.getMetadata()).equals(this.modeledSegments.getMetadata()), "Metadata in input files do not all match.");
        }
        return ((SampleLocatableMetadata)this.modeledSegments.getMetadata()).getSampleName();
    }

    private void validateNumPointsPerContig() {
        Map modeledSegmentsContigToNumPointsMap;
        if (this.inputDenoisedCopyRatiosFile != null) {
            modeledSegmentsContigToNumPointsMap = this.modeledSegments.getRecords().stream().collect(Collectors.groupingBy(ModeledSegment::getContig, LinkedHashMap::new, Collectors.summingInt(ModeledSegment::getNumPointsCopyRatio)));
            Map denoisedCopyRatiosContigToNumPointsMap = this.denoisedCopyRatios.getRecords().stream().collect(Collectors.groupingBy(CopyRatio::getContig, LinkedHashMap::new, Collectors.summingInt(x -> 1)));
            Utils.validateArg(modeledSegmentsContigToNumPointsMap.keySet().stream().allMatch(c -> !denoisedCopyRatiosContigToNumPointsMap.containsKey(c) && (Integer)modeledSegmentsContigToNumPointsMap.get(c) == 0 || denoisedCopyRatiosContigToNumPointsMap.containsKey(c) && ((Integer)modeledSegmentsContigToNumPointsMap.get(c)).equals(denoisedCopyRatiosContigToNumPointsMap.get(c))), "Number of denoised-copy-ratio points in input modeled-segments file is inconsistent with that in input denoised-copy-ratios file.");
        }
        if (this.inputAllelicCountsFile != null) {
            modeledSegmentsContigToNumPointsMap = this.modeledSegments.getRecords().stream().collect(Collectors.groupingBy(ModeledSegment::getContig, LinkedHashMap::new, Collectors.summingInt(ModeledSegment::getNumPointsAlleleFraction)));
            Map allelicCountsContigToNumPointsMap = this.allelicCounts.getRecords().stream().collect(Collectors.groupingBy(AllelicCount::getContig, LinkedHashMap::new, Collectors.summingInt(x -> 1)));
            Utils.validateArg(modeledSegmentsContigToNumPointsMap.keySet().stream().allMatch(c -> !allelicCountsContigToNumPointsMap.containsKey(c) && (Integer)modeledSegmentsContigToNumPointsMap.get(c) == 0 || allelicCountsContigToNumPointsMap.containsKey(c) && ((Integer)modeledSegmentsContigToNumPointsMap.get(c)).equals(allelicCountsContigToNumPointsMap.get(c))), "Number of allelic-count points in input modeled-segments file is inconsistent with that in input heterozygous allelic-counts file.");
        }
    }

    private void writeModeledSegmentsPlot(String sampleName, List<String> contigNames, List<Integer> contigLengths, File outputFile) {
        String contigNamesArg = String.join((CharSequence)"CONTIG_DELIMITER", contigNames);
        String contigLengthsArg = contigLengths.stream().map(Object::toString).collect(Collectors.joining("CONTIG_DELIMITER"));
        RScriptExecutor executor = new RScriptExecutor();
        executor.addScript(new Resource("CNVPlottingLibrary.R", PlotModeledSegments.class));
        executor.addScript(new Resource(PLOT_MODELED_SEGMENTS_R_SCRIPT, PlotModeledSegments.class));
        executor.addArgs("--args", "--sample_name=" + sampleName, "--denoised_copy_ratios_file=" + (this.inputDenoisedCopyRatiosFile == null ? null : CopyNumberArgumentValidationUtils.getCanonicalPath(this.inputDenoisedCopyRatiosFile)), "--allelic_counts_file=" + (this.inputAllelicCountsFile == null ? null : CopyNumberArgumentValidationUtils.getCanonicalPath(this.inputAllelicCountsFile)), "--modeled_segments_file=" + CopyNumberArgumentValidationUtils.getCanonicalPath(this.inputModeledSegmentsFile), "--contig_names=" + contigNamesArg, "--contig_lengths=" + contigLengthsArg, "--maximum_copy_ratio=" + this.maxCopyRatio, "--point_size_copy_ratio=" + this.pointSizeCopyRatio, "--point_size_allele_fraction=" + this.pointSizeAlleleFraction, "--output_file=" + CopyNumberArgumentValidationUtils.getCanonicalPath(outputFile));
        executor.exec();
    }
}

