/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.copynumber.utils;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hdf5.HDF5File;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.io.IOUtils;
import org.broadinstitute.hellbender.utils.param.ParamUtils;

public final class HDF5Utils {
    private static final Logger logger = LogManager.getLogger(HDF5Utils.class);
    public static final String INTERVAL_CONTIG_NAMES_SUB_PATH = "/indexed_contig_names";
    public static final String INTERVAL_MATRIX_SUB_PATH = "/transposed_index_start_end";
    public static final int MAX_NUMBER_OF_VALUES_PER_HDF5_MATRIX = 0xFFFFFFF;
    public static final String NUMBER_OF_ROWS_SUB_PATH = "/num_rows";
    public static final String NUMBER_OF_COLUMNS_SUB_PATH = "/num_columns";
    public static final String NUMBER_OF_CHUNKS_SUB_PATH = "/num_chunks";
    public static final String CHUNK_INDEX_PATH_SUFFIX = "/chunk_";
    private static final int NUM_INTERVAL_FIELDS = IntervalField.values().length;

    private HDF5Utils() {
    }

    public static List<SimpleInterval> readIntervals(HDF5File file, String path) {
        String[] contigNames = file.readStringArray(path + INTERVAL_CONTIG_NAMES_SUB_PATH);
        double[][] matrix = file.readDoubleMatrix(path + INTERVAL_MATRIX_SUB_PATH);
        int numIntervals = matrix[0].length;
        return IntStream.range(0, numIntervals).mapToObj(i -> new SimpleInterval(contigNames[(int)matrix[IntervalField.CONTIG_INDEX.index][i]], (int)matrix[IntervalField.START.index][i], (int)matrix[IntervalField.END.index][i])).collect(Collectors.toList());
    }

    public static <T extends SimpleInterval> void writeIntervals(HDF5File file, String path, List<T> intervals) {
        LinkedHashMap<String, Double> contigNamesToIndexMap = new LinkedHashMap<String, Double>();
        double[][] matrix = new double[NUM_INTERVAL_FIELDS][intervals.size()];
        for (int i = 0; i < intervals.size(); ++i) {
            SimpleInterval interval = (SimpleInterval)intervals.get(i);
            contigNamesToIndexMap.putIfAbsent(interval.getContig(), Double.valueOf(contigNamesToIndexMap.keySet().size()));
            matrix[((IntervalField)IntervalField.CONTIG_INDEX).index][i] = (Double)contigNamesToIndexMap.get(interval.getContig());
            matrix[((IntervalField)IntervalField.START).index][i] = interval.getStart();
            matrix[((IntervalField)IntervalField.END).index][i] = interval.getEnd();
        }
        file.makeDoubleMatrix(path + INTERVAL_MATRIX_SUB_PATH, matrix);
        file.makeStringArray(path + INTERVAL_CONTIG_NAMES_SUB_PATH, contigNamesToIndexMap.keySet().toArray(new String[contigNamesToIndexMap.keySet().size()]));
    }

    public static double[][] readChunkedDoubleMatrix(HDF5File file, String path) {
        Utils.nonNull(file);
        IOUtils.canReadFile(file.getFile());
        Utils.nonNull(path);
        String numRowsPath = path + NUMBER_OF_ROWS_SUB_PATH;
        String numColumnsPath = path + NUMBER_OF_COLUMNS_SUB_PATH;
        String numChunksPath = path + NUMBER_OF_CHUNKS_SUB_PATH;
        Utils.validateArg(file.isPresent(numRowsPath) && file.isPresent(numColumnsPath) && file.isPresent(numChunksPath), String.format("HDF5 file %s does not contain a chunked matrix in path %s.", file.getFile().getAbsolutePath(), path));
        int numRows = (int)file.readDouble(numRowsPath);
        int numColumns = (int)file.readDouble(numColumnsPath);
        int numChunks = (int)file.readDouble(numChunksPath);
        double[][] fullMatrix = new double[numRows][numColumns];
        int numRowsRead = 0;
        for (int chunkIndex = 0; chunkIndex < numChunks; ++chunkIndex) {
            double[][] matrixChunk = file.readDoubleMatrix(path + CHUNK_INDEX_PATH_SUFFIX + chunkIndex);
            if (numRowsRead + matrixChunk.length > numRows) {
                throw new UserException.BadInput("Matrix chunk contains too many rows.");
            }
            if (matrixChunk[0].length != numColumns) {
                throw new UserException.BadInput("Matrix chunk does not contain expected number of columns.");
            }
            System.arraycopy(matrixChunk, 0, fullMatrix, numRowsRead, matrixChunk.length);
            numRowsRead += matrixChunk.length;
        }
        if (numRowsRead != numRows) {
            throw new UserException.BadInput("Matrix chunks do not contain expected total number of rows.");
        }
        return fullMatrix;
    }

    public static void writeChunkedDoubleMatrix(HDF5File file, String path, double[][] matrix, int maxChunkSize) {
        double[][] matrixChunk;
        Utils.nonNull(file);
        IOUtils.canReadFile(file.getFile());
        Utils.nonNull(path);
        Utils.nonNull(matrix);
        ParamUtils.inRange(maxChunkSize, 1, 0xFFFFFFF, String.format("Maximum chunk size must be in [1, %d].", 0xFFFFFFF));
        long numRows = matrix.length;
        Utils.validateArg(numRows > 0L, "Matrix must contain at least one row.");
        long numColumns = matrix[0].length;
        Utils.validateArg(numColumns > 0L, "Matrix must contain at least one column.");
        Utils.validateArg(numColumns <= (long)maxChunkSize, String.format("Number of columns (%d) exceeds the maximum number of values allowed per chunk (%d).", numColumns, maxChunkSize));
        int numRowsPerFilledChunk = (int)((long)maxChunkSize / numColumns);
        int numFilledChunks = numRowsPerFilledChunk == 0 ? 0 : (int)numRows / numRowsPerFilledChunk;
        boolean needPartialChunk = numFilledChunks == 0 || numRows % (long)numRowsPerFilledChunk != 0L;
        logger.debug("Number of values in matrix / maximum number allowed for HDF5 matrix: " + (double)numRows * (double)numColumns / 2.68435455E8);
        logger.debug("Maximum number of values per chunk: " + maxChunkSize);
        logger.debug("Number of filled chunks: " + numFilledChunks);
        logger.debug("Number of rows per filled chunk: " + numRowsPerFilledChunk);
        logger.debug("Partial chunk needed: " + needPartialChunk);
        String numRowsPath = path + NUMBER_OF_ROWS_SUB_PATH;
        String numColumnsPath = path + NUMBER_OF_COLUMNS_SUB_PATH;
        String numChunksPath = path + NUMBER_OF_CHUNKS_SUB_PATH;
        file.makeDouble(numRowsPath, (double)numRows);
        file.makeDouble(numColumnsPath, (double)numColumns);
        file.makeDouble(numChunksPath, needPartialChunk ? (double)(numFilledChunks + 1) : (double)numFilledChunks);
        int numRowsWritten = 0;
        for (int chunkIndex = 0; chunkIndex < numFilledChunks; ++chunkIndex) {
            matrixChunk = new double[numRowsPerFilledChunk][(int)numColumns];
            System.arraycopy(matrix, numRowsWritten, matrixChunk, 0, numRowsPerFilledChunk);
            file.makeDoubleMatrix(path + CHUNK_INDEX_PATH_SUFFIX + chunkIndex, matrixChunk);
            numRowsWritten += numRowsPerFilledChunk;
        }
        if (needPartialChunk) {
            int numRowsPartialChunk = (int)numRows - numRowsWritten;
            logger.debug("Number of rows in partial chunk: " + numRowsPartialChunk);
            matrixChunk = new double[numRowsPartialChunk][(int)numColumns];
            System.arraycopy(matrix, numRowsWritten, matrixChunk, 0, numRowsPartialChunk);
            file.makeDoubleMatrix(path + CHUNK_INDEX_PATH_SUFFIX + numFilledChunks, matrixChunk);
        }
    }

    private static enum IntervalField {
        CONTIG_INDEX(0),
        START(1),
        END(2);

        private final int index;

        private IntervalField(int index) {
            this.index = index;
        }
    }
}

