/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.funcotator.dataSources;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.tribble.Feature;
import htsjdk.variant.variantcontext.VariantContext;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.StandardOpenOption;
import java.time.LocalDate;
import java.time.Month;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.engine.FeatureInput;
import org.broadinstitute.hellbender.engine.GATKTool;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.tools.funcotator.DataSourceFuncotationFactory;
import org.broadinstitute.hellbender.tools.funcotator.FlankSettings;
import org.broadinstitute.hellbender.tools.funcotator.FuncotatorArgumentDefinitions;
import org.broadinstitute.hellbender.tools.funcotator.TranscriptSelectionMode;
import org.broadinstitute.hellbender.tools.funcotator.dataSources.cosmic.CosmicFuncotationFactory;
import org.broadinstitute.hellbender.tools.funcotator.dataSources.gencode.GencodeFuncotationFactory;
import org.broadinstitute.hellbender.tools.funcotator.dataSources.vcf.VcfFuncotationFactory;
import org.broadinstitute.hellbender.tools.funcotator.dataSources.xsv.LocatableXsvFuncotationFactory;
import org.broadinstitute.hellbender.tools.funcotator.dataSources.xsv.SimpleKeyXsvFuncotationFactory;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.codecs.gtf.GencodeGtfFeature;
import org.broadinstitute.hellbender.utils.codecs.xsvLocatableTable.XsvTableFeature;
import org.broadinstitute.hellbender.utils.io.IOUtils;

public final class DataSourceUtils {
    private static final Logger logger = LogManager.getLogger(DataSourceUtils.class);
    private static final PathMatcher configFileMatcher = FileSystems.getDefault().getPathMatcher("glob:**/*.config");
    @VisibleForTesting
    static final String MANIFEST_VERSION_LINE_START = "Version:";
    private static final String MANIFEST_SOURCE_LINE_START = "Source:";
    private static final String MANIFEST_ALT_SOURCE_LINE_START = "Alternate Source:";
    @VisibleForTesting
    static final Pattern VERSION_PATTERN = Pattern.compile("Version:\\s+(\\d+)\\.(\\d+)\\.(\\d\\d\\d\\d)(\\d\\d)(\\d\\d)(.*)");
    private static final Pattern SOURCE_PATTERN = Pattern.compile("Source:\\s+(ftp.*)");
    private static final Pattern ALT_SOURCE_PATTERN = Pattern.compile("Alternate Source:\\s+(gs.*)");
    @VisibleForTesting
    static final int MIN_MAJOR_VERSION_NUMBER = 1;
    @VisibleForTesting
    static final int MIN_MINOR_VERSION_NUMBER = 6;
    @VisibleForTesting
    static final LocalDate MIN_DATE = LocalDate.of(2019, Month.JANUARY, 24);
    @VisibleForTesting
    static final int MAX_MAJOR_VERSION_NUMBER = 1;
    @VisibleForTesting
    static final int MAX_MINOR_VERSION_NUMBER = 7;
    @VisibleForTesting
    static final LocalDate MAX_DATE = LocalDate.of(2020, Month.MAY, 21);
    public static final String CURRENT_MINIMUM_DATA_SOURCE_VERSION = DataSourceUtils.getDataSourceMinVersionString();
    public static final String CURRENT_MAXIMUM_DATA_SOURCE_VERSION = DataSourceUtils.getDataSourceMaxVersionString();
    public static final String MANIFEST_FILE_NAME = "MANIFEST.txt";
    public static final String DATA_SOURCES_FTP_PATH = "ftp://gsapubftp-anonymous@ftp.broadinstitute.org/bundle/funcotator/";
    public static final String DATA_SOURCES_BUCKET_PATH = "gs://broad-public-datasets/funcotator/";
    public static final String DATA_SOURCES_NAME_PREFIX = "funcotator_dataSources";
    public static final String DS_SOMATIC_NAME_MODIFIER = "s";
    public static final String DS_GERMLINE_NAME_MODIFIER = "g";
    public static final String DS_EXTENSION = ".tar.gz";
    public static final String DS_CHECKSUM_EXTENSION = ".sha256";
    public static final String CONFIG_FILE_FIELD_NAME_NAME = "name";
    public static final String CONFIG_FILE_FIELD_NAME_VERSION = "version";
    public static final String CONFIG_FILE_FIELD_NAME_SRC_FILE = "src_file";
    public static final String CONFIG_FILE_FIELD_NAME_ORIGIN_LOCATION = "origin_location";
    public static final String CONFIG_FILE_FIELD_NAME_PREPROCESSING_SCRIPT = "preprocessing_script";
    public static final String CONFIG_FILE_FIELD_NAME_TYPE = "type";
    public static final String CONFIG_FILE_FIELD_NAME_GENCODE_FASTA_PATH = "gencode_fasta_path";
    public static final String CONFIG_FILE_FIELD_NAME_XSV_KEY = "xsv_key";
    public static final String CONFIG_FILE_FIELD_NAME_XSV_KEY_COLUMN = "xsv_key_column";
    public static final String CONFIG_FILE_FIELD_NAME_XSV_DELIMITER = "xsv_delimiter";
    public static final String CONFIG_FILE_FIELD_NAME_XSV_PERMISSIVE_COLS = "xsv_permissive_cols";
    public static final String CONFIG_FILE_FIELD_NAME_CONTIG_COLUMN = "contig_column";
    public static final String CONFIG_FILE_FIELD_NAME_START_COLUMN = "start_column";
    public static final String CONFIG_FILE_FIELD_NAME_END_COLUMN = "end_column";
    public static final String CONFIG_FILE_FIELD_NAME_NCBI_BUILD_VERSION = "ncbi_build_version";
    public static final String CONFIG_FILE_FIELD_NAME_IS_B37_DATA_SOURCE = "isB37DataSource";
    public static final String CONFIG_FILE_FIELD_NAME_LOOKAHEAD_CACHE_BP = "lookAheadCacheBp";

    private DataSourceUtils() {
    }

    public static String getDataSourceMinVersionString() {
        return DataSourceUtils.getDataSourceVersionString(1, 6, MIN_DATE);
    }

    public static String getDataSourceMaxVersionString() {
        return DataSourceUtils.getDataSourceVersionString(1, 7, MAX_DATE);
    }

    public static String getDataSourceVersionString(int major, int minor, LocalDate date) {
        return String.format("v%d.%d.%d%02d%02d", major, minor, date.getYear(), date.getMonthValue(), date.getDayOfMonth());
    }

    public static Map<Path, Properties> getAndValidateDataSourcesFromPaths(String refVersion, List<String> dataSourceDirectories) {
        Utils.nonNull(refVersion);
        Utils.nonNull(dataSourceDirectories);
        LinkedHashMap<Path, Properties> metaData = new LinkedHashMap<Path, Properties>();
        boolean hasGencodeDataSource = false;
        LinkedHashSet<String> names = new LinkedHashSet<String>();
        for (String pathString : dataSourceDirectories) {
            logger.info("Initializing data sources from directory: " + pathString);
            Path pathToDatasources = DataSourceUtils.getCloudSafePathToDirectory(pathString);
            if (!DataSourceUtils.isValidDirectory(pathToDatasources)) {
                throw new UserException("ERROR: Given data source path is not a valid directory: " + pathToDatasources.toUri());
            }
            boolean isGoodVersionOfDataSources = DataSourceUtils.logDataSourcesInfo(pathToDatasources);
            if (!isGoodVersionOfDataSources) continue;
            try {
                for (Path dataSourceTopDir : Files.list(pathToDatasources).filter(DataSourceUtils::isValidDirectory).collect(Collectors.toSet())) {
                    Path dataSourceDir = DataSourceUtils.getCloudSafePathToDirectory(dataSourceTopDir.resolve(refVersion).toUri().toString());
                    if (!DataSourceUtils.isValidDirectory(dataSourceDir)) continue;
                    Path configFilePath = DataSourceUtils.getConfigfile(dataSourceDir);
                    Properties configFileProperties = DataSourceUtils.readConfigFileProperties(configFilePath);
                    DataSourceUtils.assertConfigFilePropertiesAreValid(configFileProperties, configFilePath);
                    if (names.contains(configFileProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME))) {
                        throw new UserException.BadInput("ERROR: contains more than one dataset of name: " + configFileProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME) + " - one is: " + configFilePath.toUri().toString());
                    }
                    names.add(configFileProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME));
                    if (FuncotatorArgumentDefinitions.DataSourceType.getEnum(configFileProperties.getProperty(CONFIG_FILE_FIELD_NAME_TYPE)) == FuncotatorArgumentDefinitions.DataSourceType.GENCODE) {
                        hasGencodeDataSource = true;
                    }
                    metaData.put(configFilePath, configFileProperties);
                }
            }
            catch (IOException ex) {
                throw new GATKException("Unable to read contents of: " + pathToDatasources.toUri().toString(), ex);
            }
        }
        if (metaData.size() == 0) {
            throw new UserException("ERROR: Could not find any data sources for given reference: " + refVersion);
        }
        if (!hasGencodeDataSource) {
            throw new UserException("ERROR: a Gencode datasource is required!");
        }
        return metaData;
    }

    private static Path getConfigfile(Path directory) {
        Utils.nonNull(directory);
        try {
            List configFileSet = Files.list(directory).filter(configFileMatcher::matches).filter(x$0 -> Files.exists(x$0, new LinkOption[0])).filter(Files::isReadable).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).collect(Collectors.toList());
            if (configFileSet.size() > 1) {
                throw new UserException("ERROR: Directory contains more than one config file: " + directory.toUri().toString());
            }
            if (configFileSet.size() == 0) {
                throw new UserException("ERROR: Directory does not contain a config file: " + directory.toUri().toString());
            }
            return (Path)configFileSet.get(0);
        }
        catch (IOException ex) {
            throw new UserException("Unable to read contents of: " + directory.toUri().toString(), ex);
        }
    }

    private static boolean isValidDirectory(Path p) {
        Utils.nonNull(p);
        return Files.exists(p, new LinkOption[0]) && Files.isReadable(p) && Files.isDirectory(p, new LinkOption[0]);
    }

    public static List<DataSourceFuncotationFactory> createDataSourceFuncotationFactoriesForDataSources(Map<Path, Properties> dataSourceMetaData, LinkedHashMap<String, String> annotationOverridesMap, TranscriptSelectionMode transcriptSelectionMode, Set<String> userTranscriptIdSet, GATKTool gatkToolInstance, int lookaheadFeatureCachingInBp, FlankSettings flankSettings, boolean doAttemptSegmentFuncotationForTranscriptDatasources, int minBasesForValidSegment) {
        Utils.nonNull(dataSourceMetaData);
        Utils.nonNull(annotationOverridesMap);
        Utils.nonNull(transcriptSelectionMode);
        Utils.nonNull(userTranscriptIdSet);
        Utils.nonNull(gatkToolInstance);
        Utils.nonNull(flankSettings);
        ArrayList<DataSourceFuncotationFactory> dataSourceFactories = new ArrayList<DataSourceFuncotationFactory>(dataSourceMetaData.size());
        for (Map.Entry<Path, Properties> entry : dataSourceMetaData.entrySet()) {
            DataSourceFuncotationFactory funcotationFactory;
            String funcotationFactoryName = entry.getValue().getProperty(CONFIG_FILE_FIELD_NAME_NAME);
            logger.debug("Creating Funcotation Factory for " + funcotationFactoryName + " ...");
            Path path = entry.getKey();
            Properties properties = entry.getValue();
            String stringType = properties.getProperty(CONFIG_FILE_FIELD_NAME_TYPE);
            switch (FuncotatorArgumentDefinitions.DataSourceType.getEnum(stringType)) {
                case LOCATABLE_XSV: {
                    FeatureInput<? extends Feature> featureInput = DataSourceUtils.createAndRegisterFeatureInputs(path, properties, gatkToolInstance, lookaheadFeatureCachingInBp, XsvTableFeature.class, true);
                    funcotationFactory = DataSourceUtils.createLocatableXsvDataSource(path, properties, annotationOverridesMap, featureInput, minBasesForValidSegment);
                    break;
                }
                case SIMPLE_XSV: {
                    funcotationFactory = DataSourceUtils.createSimpleXsvDataSource(path, properties, annotationOverridesMap, minBasesForValidSegment);
                    break;
                }
                case COSMIC: {
                    funcotationFactory = DataSourceUtils.createCosmicDataSource(path, properties, annotationOverridesMap, minBasesForValidSegment);
                    break;
                }
                case GENCODE: {
                    FeatureInput<? extends Feature> featureInput = DataSourceUtils.createAndRegisterFeatureInputs(path, properties, gatkToolInstance, lookaheadFeatureCachingInBp, GencodeGtfFeature.class, false);
                    funcotationFactory = DataSourceUtils.createGencodeDataSource(path, properties, annotationOverridesMap, transcriptSelectionMode, userTranscriptIdSet, featureInput, flankSettings, doAttemptSegmentFuncotationForTranscriptDatasources, minBasesForValidSegment);
                    break;
                }
                case VCF: {
                    FeatureInput<? extends Feature> featureInput = DataSourceUtils.createAndRegisterFeatureInputs(path, properties, gatkToolInstance, lookaheadFeatureCachingInBp, VariantContext.class, false);
                    funcotationFactory = DataSourceUtils.createVcfDataSource(path, properties, annotationOverridesMap, featureInput, minBasesForValidSegment);
                    break;
                }
                default: {
                    throw new GATKException("Unknown type of DataSourceFuncotationFactory encountered: " + stringType);
                }
            }
            dataSourceFactories.add(funcotationFactory);
        }
        logger.debug("All Data Sources have been created.");
        return dataSourceFactories;
    }

    private static FeatureInput<? extends Feature> createAndRegisterFeatureInputs(Path configFilePath, Properties dataSourceProperties, GATKTool funcotatorToolInstance, int lookaheadFeatureCachingInBp, Class<? extends Feature> featureType, boolean useConfigFilePath) {
        Utils.nonNull(configFilePath);
        Utils.nonNull(dataSourceProperties);
        String name = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME);
        String sourceFile = useConfigFilePath ? configFilePath.toUri().toString() : DataSourceUtils.resolveFilePathStringFromKnownPath(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_SRC_FILE), configFilePath).toUri().toString();
        int lookaheadCacheSizePropertyValue = DataSourceUtils.getLookAheadCacheBpPropertyValue(dataSourceProperties);
        int lookaheadCacheSizeFinal = lookaheadCacheSizePropertyValue == -1 ? lookaheadFeatureCachingInBp : lookaheadCacheSizePropertyValue;
        logger.info("Setting lookahead cache for data source: " + name + " : " + lookaheadCacheSizeFinal);
        return funcotatorToolInstance.addFeatureInputsAfterInitialization(sourceFile, name, featureType, lookaheadCacheSizeFinal);
    }

    private static LocatableXsvFuncotationFactory createLocatableXsvDataSource(Path dataSourceFile, Properties dataSourceProperties, LinkedHashMap<String, String> annotationOverridesMap, FeatureInput<? extends Feature> featureInput, int minBasesForValidSegment) {
        Utils.nonNull(dataSourceFile);
        Utils.nonNull(dataSourceProperties);
        Utils.nonNull(annotationOverridesMap);
        String name = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME);
        String version = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_VERSION);
        boolean isB37 = DataSourceUtils.getIsB37PropertyValue(dataSourceProperties);
        LocatableXsvFuncotationFactory locatableXsvFuncotationFactory = new LocatableXsvFuncotationFactory(name, version, annotationOverridesMap, featureInput, isB37, minBasesForValidSegment);
        locatableXsvFuncotationFactory.setSupportedFuncotationFields(DataSourceUtils.resolveFilePathStringFromKnownPath(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_SRC_FILE), dataSourceFile));
        return locatableXsvFuncotationFactory;
    }

    private static boolean getIsB37PropertyValue(Properties dataSourceProperties) {
        if (dataSourceProperties.containsKey(CONFIG_FILE_FIELD_NAME_IS_B37_DATA_SOURCE)) {
            return Boolean.valueOf(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_IS_B37_DATA_SOURCE).replace(" ", ""));
        }
        return false;
    }

    private static int getLookAheadCacheBpPropertyValue(Properties dataSourceProperties) {
        if (dataSourceProperties.containsKey(CONFIG_FILE_FIELD_NAME_LOOKAHEAD_CACHE_BP)) {
            return Integer.valueOf(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_LOOKAHEAD_CACHE_BP).replace(" ", ""));
        }
        return -1;
    }

    private static Path getCloudSafePathToDirectory(String dirPathString) {
        if (dirPathString.startsWith("gs://") && !dirPathString.endsWith("/")) {
            return IOUtils.getPath(dirPathString + '/');
        }
        return IOUtils.getPath(dirPathString);
    }

    private static SimpleKeyXsvFuncotationFactory createSimpleXsvDataSource(Path dataSourceFile, Properties dataSourceProperties, LinkedHashMap<String, String> annotationOverridesMap, int minBasesForValidSegment) {
        Utils.nonNull(dataSourceFile);
        Utils.nonNull(dataSourceProperties);
        Utils.nonNull(annotationOverridesMap);
        boolean isB37 = DataSourceUtils.getIsB37PropertyValue(dataSourceProperties);
        return new SimpleKeyXsvFuncotationFactory(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME), DataSourceUtils.resolveFilePathStringFromKnownPath(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_SRC_FILE), dataSourceFile), dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_VERSION), dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_XSV_DELIMITER), Integer.valueOf(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_XSV_KEY_COLUMN)), SimpleKeyXsvFuncotationFactory.XsvDataKeyType.valueOf(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_XSV_KEY)), annotationOverridesMap, 0, Boolean.valueOf(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_XSV_PERMISSIVE_COLS)), isB37, minBasesForValidSegment);
    }

    private static CosmicFuncotationFactory createCosmicDataSource(Path dataSourceFile, Properties dataSourceProperties, LinkedHashMap<String, String> annotationOverridesMap, int minBasesForValidSegment) {
        Utils.nonNull(dataSourceFile);
        Utils.nonNull(dataSourceProperties);
        Utils.nonNull(annotationOverridesMap);
        String version = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_VERSION);
        boolean isB37 = DataSourceUtils.getIsB37PropertyValue(dataSourceProperties);
        return new CosmicFuncotationFactory(DataSourceUtils.resolveFilePathStringFromKnownPath(dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_SRC_FILE), dataSourceFile), annotationOverridesMap, version, isB37, minBasesForValidSegment);
    }

    private static GencodeFuncotationFactory createGencodeDataSource(Path dataSourceFile, Properties dataSourceProperties, LinkedHashMap<String, String> annotationOverridesMap, TranscriptSelectionMode transcriptSelectionMode, Set<String> userTranscriptIdSet, FeatureInput<? extends Feature> featureInput, FlankSettings flankSettings, boolean isSegmentFuncotationEnabled, int minBasesForValidSegment) {
        Utils.nonNull(dataSourceFile);
        Utils.nonNull(dataSourceProperties);
        Utils.nonNull(annotationOverridesMap);
        Utils.nonNull(transcriptSelectionMode);
        Utils.nonNull(userTranscriptIdSet);
        Utils.nonNull(flankSettings);
        String fastaPath = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_GENCODE_FASTA_PATH);
        String version = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_VERSION);
        String name = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME);
        String ncbiBuildVersion = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_NCBI_BUILD_VERSION);
        boolean isB37 = DataSourceUtils.getIsB37PropertyValue(dataSourceProperties);
        return new GencodeFuncotationFactory(DataSourceUtils.resolveFilePathStringFromKnownPath(fastaPath, dataSourceFile), version, name, transcriptSelectionMode, userTranscriptIdSet, annotationOverridesMap, featureInput, flankSettings, isB37, ncbiBuildVersion, isSegmentFuncotationEnabled, minBasesForValidSegment);
    }

    private static VcfFuncotationFactory createVcfDataSource(Path dataSourceFile, Properties dataSourceProperties, LinkedHashMap<String, String> annotationOverridesMap, FeatureInput<? extends Feature> featureInput, int minBasesForValidSegment) {
        Utils.nonNull(dataSourceFile);
        Utils.nonNull(dataSourceProperties);
        Utils.nonNull(annotationOverridesMap);
        String name = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_NAME);
        String srcFile = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_SRC_FILE);
        String version = dataSourceProperties.getProperty(CONFIG_FILE_FIELD_NAME_VERSION);
        boolean isB37 = DataSourceUtils.getIsB37PropertyValue(dataSourceProperties);
        return new VcfFuncotationFactory(name, version, DataSourceUtils.resolveFilePathStringFromKnownPath(srcFile, dataSourceFile), annotationOverridesMap, featureInput, isB37, minBasesForValidSegment);
    }

    private static Properties readConfigFileProperties(Path configFilePath) {
        Properties configProperties = new Properties();
        try (InputStream inputStream = Files.newInputStream(configFilePath, StandardOpenOption.READ);){
            configProperties.load(inputStream);
        }
        catch (Exception ex) {
            throw new UserException.BadInput("Unable to read from data source config file: " + configFilePath.toUri().toString(), ex);
        }
        return configProperties;
    }

    private static boolean logDataSourcesInfo(Path dataSourcesPath) {
        boolean dataSourcesPathIsAcceptable = true;
        Path manifestPath = dataSourcesPath.resolve(MANIFEST_FILE_NAME);
        String version = null;
        if (Files.exists(manifestPath, new LinkOption[0]) && Files.isRegularFile(manifestPath, new LinkOption[0]) && Files.isReadable(manifestPath)) {
            try (BufferedReader reader = Files.newBufferedReader(manifestPath);){
                Integer versionMajor = null;
                Integer versionMinor = null;
                Integer versionYear = null;
                Integer versionMonth = null;
                Integer versionDay = null;
                String source = null;
                String alternateSource = null;
                String line = reader.readLine();
                while (line != null && (version == null || source == null || alternateSource == null)) {
                    Matcher m;
                    if (version == null && line.startsWith(MANIFEST_VERSION_LINE_START)) {
                        Matcher matcher = VERSION_PATTERN.matcher(line);
                        if (matcher.matches()) {
                            versionMajor = Integer.valueOf(matcher.group(1));
                            versionMinor = Integer.valueOf(matcher.group(2));
                            versionYear = Integer.valueOf(matcher.group(3));
                            versionMonth = Integer.valueOf(matcher.group(4));
                            versionDay = Integer.valueOf(matcher.group(5));
                            String versionDecorator = matcher.group(6);
                            version = versionMajor + "." + versionMinor + "." + versionYear + "" + versionMonth + "" + versionDay + versionDecorator;
                        } else {
                            logger.warn("README file has improperly formatted version string: " + line);
                        }
                    }
                    if (source == null && line.startsWith(MANIFEST_SOURCE_LINE_START)) {
                        m = SOURCE_PATTERN.matcher(line);
                        if (m.matches()) {
                            source = m.group(1);
                        } else {
                            logger.warn("README file has improperly formatted source string: " + line);
                        }
                    }
                    if (alternateSource == null && line.startsWith(MANIFEST_ALT_SOURCE_LINE_START)) {
                        m = ALT_SOURCE_PATTERN.matcher(line);
                        if (m.matches()) {
                            alternateSource = m.group(1);
                        } else {
                            logger.warn("README file has improperly formatted alternate source string: " + line);
                        }
                    }
                    line = reader.readLine();
                }
                if (version == null) {
                    logger.warn("Unable to read version information from data sources info/readme file: " + manifestPath.toUri().toString());
                } else {
                    logger.info("Data sources version: " + version);
                    dataSourcesPathIsAcceptable = DataSourceUtils.validateVersionInformation(versionMajor, versionMinor, versionYear, versionMonth, versionDay);
                }
                if (source == null) {
                    logger.warn("Unable to read source information from data sources info/readme file: " + manifestPath.toUri().toString());
                } else {
                    logger.info("Data sources source: " + source);
                }
                if (alternateSource == null) {
                    logger.warn("Unable to read alternate source information from data sources info/readme file: " + manifestPath.toUri().toString());
                } else {
                    logger.info("Data sources alternate source: " + alternateSource);
                }
            }
            catch (Exception ex) {
                logger.warn("Could not read MANIFEST.txt: unable to log data sources version information.", (Throwable)ex);
            }
        } else {
            logger.warn("Could not read MANIFEST.txt: unable to log data sources version information.");
        }
        if (!dataSourcesPathIsAcceptable) {
            String message = "";
            message = message + "ERROR: Given data source path is too old or too new!  \n";
            message = message + "       Minimum required version is: " + CURRENT_MINIMUM_DATA_SOURCE_VERSION + "\n";
            message = message + "       Maximum allowed version is:  " + CURRENT_MAXIMUM_DATA_SOURCE_VERSION + "\n";
            message = message + "       Yours:                       " + version + "\n";
            message = message + "       You must download a compatible version of the data sources from the Broad Institute FTP site: " + DATA_SOURCES_FTP_PATH + "\n";
            message = message + "       or the Broad Institute Google Bucket: " + DATA_SOURCES_BUCKET_PATH + "\n";
            throw new UserException(message);
        }
        return dataSourcesPathIsAcceptable;
    }

    @VisibleForTesting
    static boolean validateVersionInformation(int major, int minor, int year, int month, int day) {
        if (major < 1 || major > 1) {
            return false;
        }
        if (major == 1 && minor < 6) {
            return false;
        }
        if (major == 1 && minor > 7) {
            return false;
        }
        LocalDate versionDate = LocalDate.of(year, month, day);
        return !versionDate.isBefore(MIN_DATE) && !versionDate.isAfter(MAX_DATE);
    }

    private static void assertConfigFilePropertiesAreValid(Properties configFileProperties, Path configFilePath) {
        FuncotatorArgumentDefinitions.DataSourceType type;
        DataSourceUtils.assertConfigPropertiesContainsKey(CONFIG_FILE_FIELD_NAME_NAME, configFileProperties, configFilePath);
        DataSourceUtils.assertConfigPropertiesContainsKey(CONFIG_FILE_FIELD_NAME_VERSION, configFileProperties, configFilePath);
        DataSourceUtils.assertConfigPropertiesContainsKey(CONFIG_FILE_FIELD_NAME_SRC_FILE, configFileProperties, configFilePath);
        DataSourceUtils.assertConfigPropertiesContainsKey(CONFIG_FILE_FIELD_NAME_ORIGIN_LOCATION, configFileProperties, configFilePath);
        DataSourceUtils.assertConfigPropertiesContainsKey(CONFIG_FILE_FIELD_NAME_PREPROCESSING_SCRIPT, configFileProperties, configFilePath);
        DataSourceUtils.assertConfigPropertiesContainsKey(CONFIG_FILE_FIELD_NAME_TYPE, configFileProperties, configFilePath);
        DataSourceUtils.assertPathFilePropertiesField(configFileProperties, CONFIG_FILE_FIELD_NAME_SRC_FILE, configFilePath);
        String stringType = configFileProperties.getProperty(CONFIG_FILE_FIELD_NAME_TYPE);
        try {
            type = FuncotatorArgumentDefinitions.DataSourceType.getEnum(stringType);
        }
        catch (IllegalArgumentException ex) {
            throw new UserException.BadInput("ERROR in config file: " + configFilePath.toUri().toString() + " - Invalid value in \"" + CONFIG_FILE_FIELD_NAME_TYPE + "\" field: " + stringType, ex);
        }
        type.assertConfigFilePropertiesAreValid(configFileProperties, configFilePath);
    }

    public static void assertConfigPropertiesContainsKey(String key, Properties configProperties, Path configFilePath) {
        if (!configProperties.stringPropertyNames().contains(key)) {
            throw new UserException.BadInput("Config file for datasource (" + configFilePath.toUri().toString() + ") does not contain required key: \"" + key + "\"");
        }
    }

    public static void assertIntegerPropertiesField(Properties props, String field, Path filePath) {
        try {
            Integer.valueOf(props.getProperty(field));
        }
        catch (NumberFormatException ex) {
            throw new UserException.BadInput("ERROR in config file: " + filePath.toUri().toString() + " - Invalid value in \"" + field + "\" field: " + props.getProperty(field));
        }
    }

    public static void assertBooleanPropertiesField(Properties props, String field, Path filePath) {
        String comparableValue = props.getProperty(field).trim().toLowerCase();
        if (!comparableValue.equals("true") && !comparableValue.equals("false")) {
            throw new UserException.BadInput("ERROR in config file: " + filePath.toUri().toString() + " - Invalid value in \"" + field + "\" field: " + props.getProperty(field));
        }
    }

    public static Path resolveFilePathStringFromKnownPath(String filePathString, Path knownPath) {
        Path absoluteFilePath;
        Path rawFilePath = IOUtils.getPath(filePathString);
        if (rawFilePath.isAbsolute() || !rawFilePath.getFileSystem().equals(FileSystems.getDefault())) {
            absoluteFilePath = rawFilePath;
        } else {
            absoluteFilePath = knownPath.resolveSibling(filePathString);
            logger.info("Resolved data source file path: " + rawFilePath.toUri().toString() + " -> " + absoluteFilePath.toUri().toString());
        }
        return absoluteFilePath;
    }

    public static void assertPathFilePropertiesField(Properties props, String field, Path configFilePath) {
        String filePathString = props.getProperty(field);
        Path absoluteFilePath = DataSourceUtils.resolveFilePathStringFromKnownPath(filePathString, configFilePath);
        if (!Files.exists(absoluteFilePath, new LinkOption[0])) {
            throw new UserException.BadInput("ERROR in config file: " + configFilePath.toUri().toString() + " - " + field + " does not exist: " + absoluteFilePath);
        }
        if (!Files.isRegularFile(absoluteFilePath, new LinkOption[0])) {
            throw new UserException.BadInput("ERROR in config file: " + configFilePath.toUri().toString() + " -  " + field + " is not a regular file: " + absoluteFilePath);
        }
        if (!Files.isReadable(absoluteFilePath)) {
            throw new UserException.BadInput("ERROR in config file: " + configFilePath.toUri().toString() + " - " + field + " is not readable: " + absoluteFilePath);
        }
    }

    public static int datasourceComparator(DataSourceFuncotationFactory f1, DataSourceFuncotationFactory f2) {
        Utils.nonNull(f1);
        Utils.nonNull(f2);
        boolean isF1Gencode = f1.getType().equals((Object)FuncotatorArgumentDefinitions.DataSourceType.GENCODE);
        boolean isF2Gencode = f2.getType().equals((Object)FuncotatorArgumentDefinitions.DataSourceType.GENCODE);
        if (isF1Gencode == isF2Gencode) {
            return f1.getInfoString().compareTo(f2.getInfoString());
        }
        if (isF1Gencode) {
            return -1;
        }
        return 1;
    }
}

