/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.spark.pipelines.metrics;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.spark.api.java.JavaRDD;
import org.apache.spark.api.java.JavaSparkContext;
import org.apache.spark.api.java.function.Function;
import org.apache.spark.api.java.function.Function2;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.BetaFeature;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import org.broadinstitute.hellbender.engine.filters.MetricsReadFilter;
import org.broadinstitute.hellbender.engine.filters.ReadFilter;
import org.broadinstitute.hellbender.engine.filters.ReadFilterLibrary;
import org.broadinstitute.hellbender.engine.spark.GATKSparkTool;
import org.broadinstitute.hellbender.metrics.MetricsUtils;
import org.broadinstitute.hellbender.utils.R.RScriptExecutor;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.io.Resource;
import org.broadinstitute.hellbender.utils.read.GATKRead;
import org.broadinstitute.hellbender.utils.read.ReadUtils;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;

@DocumentedFeature
@CommandLineProgramProperties(summary="Program to generate a data table and pdf chart of mean base quality by cycle from a SAM/BAM file.  Works best on a single lane/run of data, but can be applied to merged BAMs. Uses R to generate chart output.", oneLineSummary="MeanQualityByCycle on Spark", programGroup=DiagnosticsAndQCProgramGroup.class)
@BetaFeature
public final class MeanQualityByCycleSpark
extends GATKSparkTool {
    private static final long serialVersionUID = 1L;
    @Argument(doc="uri for the output file: a local file path", shortName="O", fullName="output", optional=true)
    public String out;
    @Argument(shortName="C", fullName="chart", doc="A file (with .pdf extension) to write the chart to.", optional=true)
    public File chartOutput;
    @Argument(shortName="A", fullName="alignedReadsOnly", doc="If set to true calculate mean quality over aligned reads only.")
    public boolean alignedReadsOnly = false;
    @Argument(shortName="F", fullName="pfReadsOnly", doc="If set to true calculate mean quality over PF reads only.")
    public boolean pfReadsOnly = false;

    @Override
    public List<ReadFilter> getDefaultReadFilters() {
        return Collections.singletonList(ReadFilterLibrary.ALLOW_ALL_READS);
    }

    @Override
    protected void runTool(JavaSparkContext ctx) {
        JavaRDD<GATKRead> reads = this.getReads();
        MetricsFile<?, Integer> metricsFile = this.calculateMeanQualityByCycle(reads);
        this.saveResults(metricsFile, this.getHeaderForReads(), this.getReadSourceName());
    }

    public MetricsFile<?, Integer> calculateMeanQualityByCycle(JavaRDD<GATKRead> reads) {
        MetricsReadFilter metricsFilter = new MetricsReadFilter(this.pfReadsOnly, this.alignedReadsOnly);
        JavaRDD filteredReads = reads.filter((Function & Serializable)read -> metricsFilter.test((GATKRead)read));
        HistogramGeneratorPair aggregate = (HistogramGeneratorPair)filteredReads.aggregate((Object)new HistogramGeneratorPair(), (Function2 & Serializable)(hgp, read) -> hgp.addRead((GATKRead)read), (Function2 & Serializable)(hgp1, hgp2) -> hgp1.merge((HistogramGeneratorPair)hgp2));
        return this.finish(aggregate.useQuals, aggregate.useOrigQuals);
    }

    private MetricsFile<?, Integer> finish(HistogramGenerator q, HistogramGenerator oq) {
        MetricsFile metrics = this.getMetricsFile();
        metrics.addHistogram(q.getMeanQualityHistogram());
        if (!oq.isEmpty()) {
            metrics.addHistogram(oq.getMeanQualityHistogram());
        }
        return metrics;
    }

    private void saveResults(MetricsFile<?, Integer> metrics, SAMFileHeader readsHeader, String inputFileName) {
        MetricsUtils.saveMetrics(metrics, this.out);
        if (metrics.getAllHistograms().isEmpty()) {
            this.logger.warn("No valid bases found in input file.");
        } else if (this.chartOutput != null) {
            List readGroups = readsHeader.getReadGroups();
            String plotSubtitle = "";
            if (readGroups.size() == 1) {
                plotSubtitle = StringUtil.asEmptyIfNull((Object)((SAMReadGroupRecord)readGroups.get(0)).getLibrary());
            }
            RScriptExecutor executor = new RScriptExecutor();
            executor.addScript(MeanQualityByCycleSpark.getMeanQualityByCycleRScriptResource());
            executor.addArgs(this.out, this.chartOutput.getAbsolutePath(), inputFileName, plotSubtitle);
            executor.exec();
        }
    }

    @VisibleForTesting
    static Resource getMeanQualityByCycleRScriptResource() {
        String R_SCRIPT = "meanQualityByCycle.R";
        return new Resource("meanQualityByCycle.R", MeanQualityByCycleSpark.class);
    }

    private static final class HistogramGeneratorPair
    implements Serializable {
        private static final long serialVersionUID = 1L;
        HistogramGenerator useQuals = new HistogramGenerator(false);
        HistogramGenerator useOrigQuals = new HistogramGenerator(true);

        private HistogramGeneratorPair() {
        }

        HistogramGeneratorPair addRead(GATKRead read) {
            this.useQuals.addRead(read);
            this.useOrigQuals.addRead(read);
            return this;
        }

        HistogramGeneratorPair merge(HistogramGeneratorPair other) {
            this.useQuals.merge(other.useQuals);
            this.useOrigQuals.merge(other.useOrigQuals);
            return this;
        }
    }

    @VisibleForTesting
    static final class HistogramGenerator
    implements Serializable {
        private static final long serialVersionUID = 1L;
        final boolean useOriginalQualities;
        int maxLengthSoFar = 0;
        double[] firstReadTotalsByCycle = new double[this.maxLengthSoFar];
        long[] firstReadCountsByCycle = new long[this.maxLengthSoFar];
        double[] secondReadTotalsByCycle = new double[this.maxLengthSoFar];
        long[] secondReadCountsByCycle = new long[this.maxLengthSoFar];

        HistogramGenerator(boolean useOriginalQualities) {
            this.useOriginalQualities = useOriginalQualities;
        }

        HistogramGenerator addRead(GATKRead read) {
            byte[] quals;
            byte[] byArray = quals = this.useOriginalQualities ? ReadUtils.getOriginalBaseQualities(read) : read.getBaseQualities();
            if (quals == null) {
                return this;
            }
            int length = quals.length;
            boolean isReverseStrand = read.isReverseStrand();
            this.ensureArraysBigEnough(length + 1);
            for (int i = 0; i < length; ++i) {
                int cycle;
                int n = cycle = isReverseStrand ? length - i : i + 1;
                if (read.isPaired() && read.isSecondOfPair()) {
                    int n2 = cycle;
                    this.secondReadTotalsByCycle[n2] = this.secondReadTotalsByCycle[n2] + (double)quals[i];
                    int n3 = cycle;
                    this.secondReadCountsByCycle[n3] = this.secondReadCountsByCycle[n3] + 1L;
                    continue;
                }
                int n4 = cycle;
                this.firstReadTotalsByCycle[n4] = this.firstReadTotalsByCycle[n4] + (double)quals[i];
                int n5 = cycle;
                this.firstReadCountsByCycle[n5] = this.firstReadCountsByCycle[n5] + 1L;
            }
            return this;
        }

        private void ensureArraysBigEnough(int length) {
            if (length > this.maxLengthSoFar) {
                this.firstReadTotalsByCycle = Arrays.copyOf(this.firstReadTotalsByCycle, length);
                this.firstReadCountsByCycle = Arrays.copyOf(this.firstReadCountsByCycle, length);
                this.secondReadTotalsByCycle = Arrays.copyOf(this.secondReadTotalsByCycle, length);
                this.secondReadCountsByCycle = Arrays.copyOf(this.secondReadCountsByCycle, length);
                this.maxLengthSoFar = length;
            }
        }

        public HistogramGenerator merge(HistogramGenerator hg2) {
            int i;
            Utils.nonNull(hg2);
            Utils.validateArg(this.useOriginalQualities == hg2.useOriginalQualities, () -> "unequal useOriginalQualities. This has " + this.useOriginalQualities);
            this.ensureArraysBigEnough(hg2.maxLengthSoFar);
            for (i = 0; i < hg2.firstReadTotalsByCycle.length; ++i) {
                int n = i;
                this.firstReadTotalsByCycle[n] = this.firstReadTotalsByCycle[n] + hg2.firstReadTotalsByCycle[i];
            }
            for (i = 0; i < hg2.secondReadTotalsByCycle.length; ++i) {
                int n = i;
                this.secondReadTotalsByCycle[n] = this.secondReadTotalsByCycle[n] + hg2.secondReadTotalsByCycle[i];
            }
            for (i = 0; i < hg2.firstReadCountsByCycle.length; ++i) {
                int n = i;
                this.firstReadCountsByCycle[n] = this.firstReadCountsByCycle[n] + hg2.firstReadCountsByCycle[i];
            }
            for (i = 0; i < hg2.secondReadCountsByCycle.length; ++i) {
                int n = i;
                this.secondReadCountsByCycle[n] = this.secondReadCountsByCycle[n] + hg2.secondReadCountsByCycle[i];
            }
            return this;
        }

        Histogram<Integer> getMeanQualityHistogram() {
            String label = this.useOriginalQualities ? "MEAN_ORIGINAL_QUALITY" : "MEAN_QUALITY";
            Histogram meanQualities = new Histogram("CYCLE", label);
            int firstReadLength = 0;
            for (int cycle = 0; cycle < this.firstReadTotalsByCycle.length; ++cycle) {
                if (!(this.firstReadTotalsByCycle[cycle] > 0.0)) continue;
                meanQualities.increment((Comparable)Integer.valueOf(cycle), this.firstReadTotalsByCycle[cycle] / (double)this.firstReadCountsByCycle[cycle]);
                firstReadLength = cycle;
            }
            for (int i = 0; i < this.secondReadTotalsByCycle.length; ++i) {
                if (this.secondReadCountsByCycle[i] <= 0L) continue;
                int cycle = firstReadLength + i;
                meanQualities.increment((Comparable)Integer.valueOf(cycle), this.secondReadTotalsByCycle[i] / (double)this.secondReadCountsByCycle[i]);
            }
            return meanQualities;
        }

        boolean isEmpty() {
            return this.maxLengthSoFar == 0;
        }
    }
}

