/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.spark.utils;

import com.esotericsoftware.kryo.DefaultSerializer;
import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import java.util.Arrays;
import org.apache.commons.math3.distribution.AbstractIntegerDistribution;
import org.apache.commons.math3.random.JDKRandomGenerator;
import org.apache.commons.math3.random.RandomGenerator;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.param.ParamUtils;

@DefaultSerializer(value=Serializer.class)
public final class IntHistogram {
    private final long[] counts;
    private long totalObservations;

    public IntHistogram(int maxTrackedValue) {
        Utils.validateArg(maxTrackedValue >= 0, "maxTrackedValue must be non-negative");
        this.counts = new long[maxTrackedValue + 2];
        this.totalObservations = 0L;
    }

    private IntHistogram(long[] counts, long totalObservations) {
        this.counts = counts;
        this.totalObservations = totalObservations;
    }

    private IntHistogram(Kryo kryo, Input input) {
        int len = input.readInt();
        this.counts = new long[len];
        long total = 0L;
        for (int idx = 0; idx != len; ++idx) {
            long val = input.readLong();
            double idxSum = val * (long)idx;
            total += val;
            this.counts[idx] = val;
        }
        this.totalObservations = total;
    }

    private void serialize(Kryo kryo, Output output) {
        output.writeInt(this.counts.length);
        for (long val : this.counts) {
            output.writeLong(val);
        }
    }

    public void addObservation(int observedValue) {
        Utils.validateArg(observedValue >= 0, "observedValue must be non-negative");
        int n = observedValue >= this.counts.length ? this.counts.length - 1 : observedValue;
        this.counts[n] = this.counts[n] + 1L;
        ++this.totalObservations;
    }

    public void addObservations(int observedValue, long nObservations) {
        Utils.validateArg(observedValue >= 0, "observedValue must be non-negative");
        Utils.validateArg(nObservations >= 0L, "nObservations must be non-negative");
        int n = observedValue >= this.counts.length ? this.counts.length - 1 : observedValue;
        this.counts[n] = this.counts[n] + nObservations;
        this.totalObservations += nObservations;
    }

    public void addObservations(IntHistogram intHistogram) {
        long[] thoseCounts = intHistogram.counts;
        Utils.validateArg(this.counts.length == thoseCounts.length, "The supplied histogram doesn't have the right shape.");
        for (int idx = 0; idx != this.counts.length; ++idx) {
            int n = idx;
            this.counts[n] = this.counts[n] + thoseCounts[idx];
        }
        this.totalObservations += intHistogram.totalObservations;
    }

    public int getMaximumTrackedValue() {
        return this.counts.length - 2;
    }

    public long getTotalObservations() {
        return this.totalObservations;
    }

    public long getNObservations(int observedValue) {
        Utils.validateArg(observedValue >= 0 && observedValue < this.counts.length, "observedValue must be non-negative, and no more than 1 greater than the maximum tracked observed value");
        return this.counts[observedValue];
    }

    public void clear() {
        Arrays.fill(this.counts, 0L);
        this.totalObservations = 0L;
    }

    public IntHistogram trim() {
        int nBins = this.counts.length;
        long nUntrackedObservations = this.counts[this.counts.length - 1];
        if ((long)nBins * nUntrackedObservations >= this.totalObservations) {
            return this;
        }
        while ((long)(--nBins) * (nUntrackedObservations += this.counts[nBins - 2]) < this.totalObservations) {
        }
        long[] newCounts = Arrays.copyOfRange(this.counts, 0, nBins);
        newCounts[nBins - 1] = nUntrackedObservations -= this.counts[++nBins - 2];
        return new IntHistogram(newCounts, this.totalObservations);
    }

    public CDF getCDF() {
        return new CDF(this);
    }

    public AbstractIntegerDistribution empiricalDistribution(final int smoothing) {
        int i;
        ParamUtils.isPositiveOrZero(smoothing, "the smoothing must be zero or positive");
        final long[] counts = Arrays.copyOfRange(this.counts, 0, this.counts.length - 1);
        final double[] cumulativeCounts = new double[counts.length];
        double sum = 0.0;
        double sqSum = 0.0;
        for (i = 0; i < counts.length; ++i) {
            int n = i;
            long l = counts[n] + (long)smoothing;
            counts[n] = l;
            long newCount = l;
            sum += (double)(newCount * (long)i);
            sqSum += (double)((long)i * newCount * (long)i);
        }
        cumulativeCounts[0] = counts[0];
        for (i = 1; i < counts.length; ++i) {
            cumulativeCounts[i] = (double)counts[i] + cumulativeCounts[i - 1];
        }
        double totalCounts = cumulativeCounts[counts.length - 1];
        final double inverseTotalCounts = 1.0 / totalCounts;
        final double mean = sum / totalCounts;
        final double variance = sqSum / totalCounts - mean * mean;
        int seed = Arrays.hashCode(counts);
        JDKRandomGenerator rdnGen = new JDKRandomGenerator();
        rdnGen.setSeed(seed);
        return new AbstractIntegerDistribution((RandomGenerator)rdnGen){
            private static final long serialVersionUID = -1L;

            public double probability(int x) {
                if (x < 0) {
                    return 0.0;
                }
                if (x >= counts.length) {
                    return (double)smoothing * inverseTotalCounts;
                }
                return (double)counts[x] * inverseTotalCounts;
            }

            public double cumulativeProbability(int x) {
                if (x < 0) {
                    return 0.0;
                }
                if (x >= counts.length) {
                    return 1.0;
                }
                return cumulativeCounts[x] * inverseTotalCounts;
            }

            public double getNumericalMean() {
                return mean;
            }

            public double getNumericalVariance() {
                return variance;
            }

            public int getSupportLowerBound() {
                return 0;
            }

            public int getSupportUpperBound() {
                return Integer.MAX_VALUE;
            }

            public boolean isSupportConnected() {
                return true;
            }
        };
    }

    @DefaultSerializer(value=Serializer.class)
    public static final class CDF {
        final float[] cdfFractions;
        final long nCounts;

        public CDF(IntHistogram intHistogram) {
            int arrayLen = intHistogram.getMaximumTrackedValue() + 2;
            this.cdfFractions = new float[arrayLen];
            this.nCounts = intHistogram.getTotalObservations();
            long sum = 0L;
            for (int observedValue = 0; observedValue != arrayLen; ++observedValue) {
                this.cdfFractions[observedValue] = (float)(sum += intHistogram.getNObservations(observedValue)) / (float)this.nCounts;
            }
        }

        public CDF(float[] cdfFractions, long nCounts) {
            this.cdfFractions = cdfFractions;
            this.nCounts = nCounts;
        }

        private CDF(Kryo kryo, Input input) {
            int len = input.readInt();
            this.cdfFractions = new float[len];
            for (int idx = 0; idx != len; ++idx) {
                this.cdfFractions[idx] = input.readFloat();
            }
            this.nCounts = input.readLong();
        }

        private void serialize(Kryo kryo, Output output) {
            output.writeInt(this.cdfFractions.length);
            for (float val : this.cdfFractions) {
                output.writeFloat(val);
            }
            output.writeLong(this.nCounts);
        }

        public int size() {
            return this.cdfFractions.length;
        }

        public float getFraction(int idx) {
            return this.cdfFractions[idx];
        }

        public long getTotalObservations() {
            return this.nCounts;
        }

        public IntHistogram createEmptyHistogram() {
            return new IntHistogram(this.cdfFractions.length - 2);
        }

        public boolean isDifferentByKSStatistic(IntHistogram sampleHistogram, float significance) {
            long[] sampleCounts = sampleHistogram.counts;
            Utils.validateArg(significance > 0.0f && significance < 0.2f, "The significance must be specifed as a probability of a chance occurrence (a number like .01f).");
            Utils.validateArg(sampleCounts.length == this.cdfFractions.length, "The supplied histogram doesn't have the right size.");
            long mCounts = sampleHistogram.getTotalObservations();
            Utils.validateArg(mCounts > 0L, "The supplied histogram is empty.");
            double cOfAlpha = Math.sqrt(-0.5 * Math.log((double)significance / 2.0));
            float significantDiff = (float)(cOfAlpha * Math.sqrt((double)(this.nCounts + mCounts) / (double)this.nCounts / (double)mCounts));
            long sum = 0L;
            int idx = 0;
            float curVal = 0.0f;
            boolean prevZero = false;
            long trackedTotal = mCounts - sampleCounts[sampleCounts.length - 1];
            while (sum < trackedTotal) {
                long val = sampleCounts[idx];
                if (val == 0L) {
                    prevZero = true;
                } else {
                    if (prevZero && Math.abs(this.cdfFractions[idx - 1] - curVal) >= significantDiff) {
                        return true;
                    }
                    prevZero = false;
                    curVal = (float)(sum += val) / (float)mCounts;
                    if (Math.abs(this.cdfFractions[idx] - curVal) >= significantDiff) {
                        return true;
                    }
                }
                ++idx;
            }
            return false;
        }

        public int median() {
            return this.internalPopStat(0.5f);
        }

        public int leftMedianDeviation(int median) {
            return median - this.internalPopStat(0.25f);
        }

        public int rightMedianDeviation(int median) {
            return this.internalPopStat(0.75f) - median;
        }

        public int popStat(float popFraction) {
            Utils.validateArg(popFraction >= 0.0f && popFraction <= 1.0f, "popFraction must be between 0 and 1");
            return this.internalPopStat(popFraction);
        }

        private int internalPopStat(float popFraction) {
            int idx = Arrays.binarySearch(this.cdfFractions, popFraction);
            return idx < 0 ? -idx - 1 : idx;
        }

        public static final class Serializer
        extends com.esotericsoftware.kryo.Serializer<CDF> {
            public void write(Kryo kryo, Output output, CDF cdf) {
                cdf.serialize(kryo, output);
            }

            public CDF read(Kryo kryo, Input input, Class<CDF> klass) {
                return new CDF(kryo, input);
            }
        }
    }

    public static final class Serializer
    extends com.esotericsoftware.kryo.Serializer<IntHistogram> {
        public void write(Kryo kryo, Output output, IntHistogram histogram) {
            histogram.serialize(kryo, output);
        }

        public IntHistogram read(Kryo kryo, Input input, Class<IntHistogram> klass) {
            return new IntHistogram(kryo, input);
        }
    }
}

