/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.spark.utils;

import com.esotericsoftware.kryo.DefaultSerializer;
import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.google.common.annotations.VisibleForTesting;
import java.util.Arrays;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.tools.spark.sv.utils.SVUtils;
import org.broadinstitute.hellbender.tools.spark.utils.SetSizeUtils;
import org.broadinstitute.hellbender.utils.Utils;

@DefaultSerializer(value=Serializer.class)
public final class LongBloomFilter {
    private final transient Logger logger = LogManager.getLogger(this.getClass());
    private final long totalBits;
    private final int numHashes;
    private final long totalBuckets;
    private final int numBucketArrays;
    private final int bucketArraySize;
    private final int finalBucketArraySize;
    private final byte[][] buckets;
    private static final long[] legalBitSizes = new long[]{10007L, 16411L, 32771L, 65537L, 131101L, 262147L, 524309L, 0x100007L, 0x200011L, 0x40000FL, 0x800009L, 16777259L, 0x2000023L, 0x400000FL, 134217757L, 0x10000003L, 0x2000000BL, 0x40000003L, 0x8000000BL, 0x10000000FL, 0x200000011L, 17179869209L, 34359738337L, 0x100000001FL, 0x2000000009L, 0x4000000007L, 0x7FFFFFFFF9L, 0x1000000000FL};
    private static final long HASH_SEED_2 = 7848620611008010406L;

    public LongBloomFilter(long numElements, double fpp) {
        Utils.validateArg(numElements > 0L, "Number of elements must be greater than 0");
        Utils.validateArg(fpp > 0.0 && fpp < 1.0, "False positive probability must be between 0 and 1");
        long optimalNumberOfBits = LongBloomFilter.getOptimalNumberOfBits(numElements, fpp);
        long smallestLegalSize = -1L;
        for (long legalSize : legalBitSizes) {
            if (legalSize <= optimalNumberOfBits) continue;
            smallestLegalSize = legalSize;
            break;
        }
        if (smallestLegalSize == -1L) {
            throw new GATKException("Could not create Bloom filter with " + optimalNumberOfBits + " bits");
        }
        this.totalBits = smallestLegalSize;
        int optimalNumberOfHashes = (int)Math.ceil(-Math.log(fpp) / Math.log(2.0));
        this.numHashes = optimalNumberOfHashes > 0 ? optimalNumberOfHashes : 1;
        this.totalBuckets = this.totalBits / 8L + (long)(this.totalBits % 8L > 0L ? 1 : 0);
        this.bucketArraySize = SetSizeUtils.legalSizes[SetSizeUtils.legalSizes.length - 1];
        this.numBucketArrays = (int)(this.totalBuckets / (long)this.bucketArraySize) + 1;
        this.finalBucketArraySize = (int)(this.totalBuckets % (long)this.bucketArraySize);
        this.buckets = new byte[this.numBucketArrays][];
        for (int i = 0; i < this.numBucketArrays - 1; ++i) {
            this.buckets[i] = new byte[this.bucketArraySize];
        }
        this.buckets[this.numBucketArrays - 1] = new byte[this.finalBucketArraySize];
    }

    protected LongBloomFilter(Kryo kryo, Input input) {
        this.totalBits = input.readLong();
        this.totalBuckets = input.readLong();
        this.numBucketArrays = input.readInt();
        this.bucketArraySize = input.readInt();
        this.finalBucketArraySize = input.readInt();
        this.numHashes = input.readInt();
        this.buckets = new byte[this.numBucketArrays][];
        for (int i = 0; i < this.numBucketArrays - 1; ++i) {
            this.buckets[i] = input.readBytes(this.bucketArraySize);
        }
        this.buckets[this.numBucketArrays - 1] = input.readBytes(this.finalBucketArraySize);
        if (this.logger.isDebugEnabled()) {
            long X = LongBloomFilter.countBits(this.buckets);
            long estimatedSize = this.estimateSize(X);
            this.logger.debug("Deserialized: totalBits : " + this.totalBits + ", totalBuckets: " + this.totalBuckets + ", numHashes: " + this.numHashes + ", bits set: " + X + ", approximate size: " + estimatedSize);
        }
    }

    protected void serialize(Kryo kryo, Output output) {
        output.writeLong(this.totalBits);
        output.writeLong(this.totalBuckets);
        output.writeInt(this.numBucketArrays);
        output.writeInt(this.bucketArraySize);
        output.writeInt(this.finalBucketArraySize);
        output.writeInt(this.numHashes);
        for (int i = 0; i < this.numBucketArrays; ++i) {
            output.writeBytes(this.buckets[i]);
        }
    }

    public static long getOptimalNumberOfBits(long numElements, double fpp) {
        return (long)Math.ceil((double)(-numElements) * Math.log(fpp) / (Math.log(2.0) * Math.log(2.0)));
    }

    public double getTheoreticalFPP(long numElements) {
        return Math.pow(1.0 - Math.pow(1.0 - 1.0 / (double)this.totalBits, (long)this.numHashes * numElements), this.numHashes);
    }

    @VisibleForTesting
    static long countBits(byte[][] arr) {
        int[] bitCountMap = new int[256];
        for (int b = 0; b < 256; ++b) {
            bitCountMap[b] = Integer.bitCount((byte)b);
        }
        long sum = 0L;
        for (int i = 0; i < arr.length; ++i) {
            for (byte b : arr[i]) {
                sum += (long)bitCountMap[0xFF & b];
            }
        }
        return sum;
    }

    private long estimateSize(long numBitsSet) {
        return (long)(-((double)this.totalBits / (double)this.numHashes) * Math.log(1.0 - (double)numBitsSet / (double)this.totalBits));
    }

    public boolean add(long entryValue) {
        long hash1 = SVUtils.fnvLong64(entryValue);
        long hash2 = SVUtils.fnvLong64(7848620611008010406L, entryValue);
        for (int i = 0; i < this.numHashes; ++i) {
            long bitIndex = this.applyHashFunction(i, hash1, hash2);
            int bucketArray = this.bitIndexToBucketArray(bitIndex);
            int bucketIndex = this.bitIndexToBucketIndex(bitIndex);
            byte[] byArray = this.buckets[bucketArray];
            int n = bucketIndex;
            byArray[n] = (byte)(byArray[n] | this.bucketMask(bitIndex));
        }
        return true;
    }

    public boolean contains(long key) {
        long hash1 = SVUtils.fnvLong64(key);
        long hash2 = SVUtils.fnvLong64(7848620611008010406L, key);
        for (int i = 0; i < this.numHashes; ++i) {
            long bitIndex = this.applyHashFunction(i, hash1, hash2);
            int bucketArray = this.bitIndexToBucketArray(bitIndex);
            int bucketIndex = this.bitIndexToBucketIndex(bitIndex);
            if ((this.bucketMask(bitIndex) & this.buckets[bucketArray][bucketIndex]) != 0) continue;
            return false;
        }
        return true;
    }

    public void addAll(long[] entryValues) {
        for (long val : entryValues) {
            this.add(val);
        }
    }

    public boolean containsAll(long[] vals) {
        for (long val : vals) {
            if (this.contains(val)) continue;
            return false;
        }
        return true;
    }

    private long applyHashFunction(int i, long fnvHash1, long fnvHash2) {
        long result = (fnvHash1 + (long)i * fnvHash2) % this.totalBits;
        return result < 0L ? result + this.totalBits : result;
    }

    private int bitIndexToBucketArray(long bitIndex) {
        return (int)((bitIndex >>> 3) / (long)this.bucketArraySize);
    }

    private int bitIndexToBucketIndex(long bitIndex) {
        return (int)((bitIndex >>> 3) % (long)this.bucketArraySize);
    }

    private byte bucketMask(long bitIndex) {
        return (byte)(1 << (int)(bitIndex & 7L));
    }

    public void clear() {
        for (int i = 0; i < this.numBucketArrays; ++i) {
            Arrays.fill(this.buckets[i], (byte)0);
        }
    }

    public boolean isEmpty() {
        byte[][] byArray = this.buckets;
        int n = byArray.length;
        for (int i = 0; i < n; ++i) {
            byte[] array;
            for (byte b : array = byArray[i]) {
                if (b == 0) continue;
                return false;
            }
        }
        return true;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof LongBloomFilter)) {
            return false;
        }
        LongBloomFilter that = (LongBloomFilter)o;
        if (this.totalBits != that.totalBits) {
            return false;
        }
        if (this.numHashes != that.numHashes) {
            return false;
        }
        if (this.numBucketArrays != that.numBucketArrays) {
            return false;
        }
        for (int i = 0; i < this.numBucketArrays; ++i) {
            if (Arrays.equals(this.buckets[i], that.buckets[i])) continue;
            return false;
        }
        return true;
    }

    public int hashCode() {
        int result = (int)(this.totalBits ^ this.totalBits >>> 32);
        result = 31 * result + this.numHashes;
        for (byte[] array : this.buckets) {
            result = 31 * result + Arrays.hashCode(array);
        }
        return result;
    }

    public static final class Serializer
    extends com.esotericsoftware.kryo.Serializer<LongBloomFilter> {
        public void write(Kryo kryo, Output output, LongBloomFilter bloomFilter) {
            bloomFilter.serialize(kryo, output);
        }

        public LongBloomFilter read(Kryo kryo, Input input, Class<LongBloomFilter> klass) {
            return new LongBloomFilter(kryo, input);
        }
    }
}

